/**
***************************************************************************
* @file dlrCommon/exception.cpp
* 
* Source file defining some exception types.
*
* Copyright (C) 2003-2005, David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 877 $
* $Date: 2007-05-04 00:12:02 -0400 (Fri, 04 May 2007) $
***************************************************************************
**/

#include <cstdarg>
#include <cstdio>
#include <cstring>
#include <stdexcept>
#include <dlrCommon/exception.h>

// Anonymous namespace for local functions.
namespace {

  // std::snprintf() doesn't seem to be supported on some
  // architectures, so we define a crippled version here which fills
  // our needs.
  int l_snprintf(char* targetPtr, size_t targetSize, 
                 const char* formatPtr, ...) {
    char workingBuffer[1024];
    va_list varArgsList;
    va_start(varArgsList, formatPtr);

    const char* workingFormatPtr = formatPtr;
    char* workingTargetPtr = targetPtr;
    size_t spaceLeft = targetSize - 1;

    // Stop if we've used up either the available space or the format string.
    while(spaceLeft && (*workingFormatPtr != '\0')) {
      int intValue;
      double doubleValue;
      char* charPtrValue;
      size_t stringSize;
      if(*workingFormatPtr == '%') {
        ++workingFormatPtr;
        switch(*workingFormatPtr) {
        case '\0':
          break;
        case 'd':
          intValue = va_arg(varArgsList, int);
          sprintf(workingBuffer, "%d", intValue);
          std::strncpy(workingTargetPtr, workingBuffer, spaceLeft);
          stringSize = strlen(workingBuffer);
          workingTargetPtr +=
            (stringSize > spaceLeft) ? spaceLeft : stringSize;
          spaceLeft -= (stringSize > spaceLeft) ? spaceLeft : stringSize;
          ++workingFormatPtr;
          break;
        case 'f':
          doubleValue = va_arg(varArgsList, double);
          sprintf(workingBuffer, "%f", doubleValue);
          std::strncpy(workingTargetPtr, workingBuffer, spaceLeft);
          stringSize = strlen(workingBuffer);
          workingTargetPtr +=
            (stringSize > spaceLeft) ? spaceLeft : stringSize;
          spaceLeft -= (stringSize > spaceLeft) ? spaceLeft : stringSize;
          ++workingFormatPtr;
          break;
        case 's':
          charPtrValue = va_arg(varArgsList, char*);
          std::strncpy(workingTargetPtr, charPtrValue, spaceLeft);
          stringSize = strlen(charPtrValue);
          workingTargetPtr +=
            (stringSize > spaceLeft) ? spaceLeft : stringSize;
          spaceLeft -= (stringSize > spaceLeft) ? spaceLeft : stringSize;
          ++workingFormatPtr;
          break;
        default:
          va_end(varArgsList);
          std::sprintf(
            workingBuffer,
            "Exception(l_snprintf()): Format string %%%c not recognized.",
            *workingFormatPtr);
          throw std::invalid_argument(workingBuffer);
        }
      } else {
        *workingTargetPtr++ = *workingFormatPtr++;
        --spaceLeft;
      }
    }
    if(targetSize > 0) {
      *workingTargetPtr = '\0';
    }
    va_end(varArgsList);
    return static_cast<int>(workingTargetPtr - targetPtr);
  }

} // Anonymous namespace.


namespace dlr {

  namespace common {

    // This constructor sets the internal "what()" message.  The
    Exception::
    Exception(const char* message)
      throw()
      : std::exception(),
        m_traceMessage(),
        m_traceMessageIndex(0)
    {
      l_snprintf(m_message, DLR_EXCEPTION_MESSAGE_LENGTH, "Exception: %s",
                 message);
      m_traceMessage[0] = '\0';
    }


    // Constructor which accepts a few more details.
    Exception::
    Exception(const char* message, const char* fileName, int lineNumber)
      throw()
      : std::exception(),
        m_traceMessage(),
        m_traceMessageIndex(0)
    {
      l_snprintf(m_message, DLR_EXCEPTION_MESSAGE_LENGTH,
                 "Exception(%s, %d): %s",
                 fileName, lineNumber, message);
      m_traceMessage[0] = '\0';
    }


    // Constructor which accepts a even more details.
    Exception::
    Exception(const char* message, const char* functionName,
              const char* fileName, int lineNumber)
      throw()
      : std::exception(),
        m_traceMessage(),
        m_traceMessageIndex(0)
    {
      l_snprintf(m_message, DLR_EXCEPTION_MESSAGE_LENGTH,
                 "Exception(%s, %s, %d): %s",
                 functionName, fileName, lineNumber, message);
      m_traceMessage[0] = '\0';
    }


    // Copy constructor.
    Exception::
    Exception(const Exception& source)
      throw()
      : std::exception(),
        m_traceMessage(),
        m_traceMessageIndex(source.m_traceMessageIndex)
    {
      strncpy(m_message, source.m_message, DLR_EXCEPTION_MESSAGE_LENGTH);
      m_message[DLR_EXCEPTION_MESSAGE_LENGTH - 1] = '\0';

      size_t traceMessageMaxLength =
        (DLR_EXCEPTION_TRACE_REQUIRED_STACK_LEVELS
         * DLR_EXCEPTION_TRACE_MESSAGE_LENGTH);
      strncpy(m_traceMessage, source.m_traceMessage, traceMessageMaxLength);
      m_traceMessage[traceMessageMaxLength - 1] = '\0';
    }


    // Assignment operator.
    Exception& Exception::
    operator=(const Exception& source)
      throw()
    {
      strncpy(m_message, source.m_message, DLR_EXCEPTION_MESSAGE_LENGTH);
      m_message[DLR_EXCEPTION_MESSAGE_LENGTH - 1] = '\0';

      size_t traceMessageMaxLength =
        (DLR_EXCEPTION_TRACE_REQUIRED_STACK_LEVELS
         * DLR_EXCEPTION_TRACE_MESSAGE_LENGTH);
      strncpy(m_traceMessage, source.m_traceMessage, traceMessageMaxLength);
      m_traceMessage[traceMessageMaxLength - 1] = '\0';

      m_traceMessageIndex = source.m_traceMessageIndex;
    
      return *this;
    }


    // Protected constructor.
    Exception::
    Exception(const char* message, const char* childClassName)
      throw()
      : std::exception(),
        m_traceMessage(),
        m_traceMessageIndex(0)
    {
      l_snprintf(m_message, DLR_EXCEPTION_MESSAGE_LENGTH, "%s: %s",
                 childClassName, message);
      m_traceMessage[0] = '\0';
    }


    // Protected constructor.
    Exception::
    Exception(const char* message, const char* childClassName,
              const char* functionName, const char* fileName,
              int lineNumber)
      throw()
      : std::exception(),
        m_traceMessage(),
        m_traceMessageIndex(0)
    {
      if(functionName == 0) {
        l_snprintf(m_message, DLR_EXCEPTION_MESSAGE_LENGTH,
                   "%s(%s, %d): %s",
                   childClassName, fileName, lineNumber, message);
      } else {
        l_snprintf(m_message, DLR_EXCEPTION_MESSAGE_LENGTH,
                   "%s(%s, %s, %d): %s",
                   childClassName, functionName, fileName, lineNumber,
                   message);
      }
      m_traceMessage[0] = '\0';
    }


    // This public method appends the provided text to the internal
    // "trace()" message.
    void
    Exception::
    addTrace(const char* message)
      throw()
    {
      const char* ellipsisString = "...";
      size_t ellipsisLength = strlen(ellipsisString);

      // Sanity check.
      if(DLR_EXCEPTION_TRACE_MESSAGE_LENGTH < (ellipsisLength + 1)) {
        return;
      }
    
      if(message != 0) {
        // This flag tells us whether we've truncated the string.
        bool useEllipsis = false;
      
        // strlen does not count the terminator.
        size_t messageLength = std::strlen(message); 
        if(messageLength >= DLR_EXCEPTION_TRACE_MESSAGE_LENGTH) {
          messageLength = DLR_EXCEPTION_TRACE_MESSAGE_LENGTH - 1;
          useEllipsis = true;
        }

        // If the trace message is filled up, we stop recording.
        size_t remainingSpace =
          (DLR_EXCEPTION_TRACE_REQUIRED_STACK_LEVELS
           * DLR_EXCEPTION_TRACE_MESSAGE_LENGTH) - m_traceMessageIndex;
        if((messageLength + 1) > remainingSpace) {
          // Forbid future trace information as well by pretending
          // there's no space left at all.
          m_traceMessageIndex = (DLR_EXCEPTION_TRACE_REQUIRED_STACK_LEVELS
                                 * DLR_EXCEPTION_TRACE_MESSAGE_LENGTH);
          return;
        }

        // If we're going to append a "...", better make sure there's
        // room for it.
        if(useEllipsis) {
          // Sanity check.  This should never be true.
          if(messageLength < ellipsisLength) {
            return;
          }
          messageLength -= ellipsisLength;
        }

        // The messageLength parameter will prevent this from copying
        // the final terminator.
        std::strncpy(
          m_traceMessage + m_traceMessageIndex, message, messageLength);
        m_traceMessageIndex += messageLength;

        // In both of the cases below, we don't advance
        // m_traceMessageIndex past the final '\0'.  This is because we
        // want to overwrite the terminator next time we get a message.
        if(useEllipsis) {
          // This copies the terminator as well.  
          std::strcpy(m_traceMessage + m_traceMessageIndex, ellipsisString);
          m_traceMessageIndex += strlen(ellipsisString);
        } else {
          m_traceMessage[m_traceMessageIndex] = '\0';
        }
      }
    }

  } // namespace common
  
} // namespace dlr
