/**CFile***********************************************************************

  FileName    [ timer.c ]

  PackageName [ timer ]

  Synopsis    [ Routines implementing basic timer functionnality ]

  Description [ This file implements the timer package actions.
  Each provide one basic timer functionnality. ]

  Author      [ David Deharbe ]

  Copyright   [ Carnegie Mellon University - 1996 ]

******************************************************************************/

#include "timerInt.h" 

#include <memuser.h> 

static char rcsid[] = "$Id: $";


/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/
typedef enum {reset, started, stopped} timerState_c;
typedef struct timerCell_t_ {
  timerState_c state;
  struct timeval elapsed_sys_time;
  struct timeval elapsed_user_time;
  struct timeval start_sys_time;
  struct timeval start_user_time;
} timerCell_t;

/*---------------------------------------------------------------------------*/
/* Structure declarations                                                    */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/


/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static void timerElapsedTime(long sys_sec, long sys_usec, long user_sec, long user_usec, long * res_sys_sec, long * res_sys_usec, long * res_user_sec, long * res_user_usec);

/**AutomaticEnd***************************************************************/


/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [ Returns a new timer ]

  SideEffects        [ Allocates memory ]

  Description        [ Creates and return a new timer in the reset state ]

  SeeAlso            [ timer_Dispose ]

******************************************************************************/

timer
timer_New
(void)
{
  timer result;
  result = (timer) mem_get_block((SIZE_T) sizeof (timerCell_t));
  timer_Reset(result);
  return result;
}

/**Function********************************************************************

  Synopsis           [ Deletes a timer ]

  SideEffects        [ Disallocates memory ]

  SeeAlso            [ timer_New ]

******************************************************************************/
void
timer_Dispose
(timer t)
{
  mem_free_block(t);
}


/**Function********************************************************************

  Synopsis           [ Sets a timer in the reset state ]

  Description        [ The given timer is reset. All its internal time 
  counters are set to zero. The timer goes to the reset state. ]

  SideEffects        [ Modifies the state of the timer ]

  SeeAlso            [ timer_Start timer_Stop ]

******************************************************************************/
void
timer_Reset
(timer t)
{
  t->state = reset;
  t->elapsed_sys_time.tv_sec=0;
  t->elapsed_sys_time.tv_usec=0;
  t->elapsed_user_time.tv_sec=0;
  t->elapsed_user_time.tv_usec=0;
  t->start_sys_time.tv_sec=0;
  t->start_sys_time.tv_usec=0;
  t->start_user_time.tv_sec=0;
  t->start_user_time.tv_usec=0;
}

/**Function********************************************************************

  Synopsis           [ Starts the timer ]

  Description        [ The timer should be in the reset or stopped states.
  The effect of this function is that the timer starts to increment its
  internal system and user time conters and goes to the started state.. ]

  SideEffects        [ Modifies the timer state. Performs a getrusage
  system call. ]

  SeeAlso            [ timer_Reset timer_Stop ]

******************************************************************************/
void
timer_Start
(timer t)
{
  static struct rusage tmp;
  switch (t->state) {
  case reset: case stopped:
    getrusage(0, & tmp);
    t->start_sys_time.tv_sec = tmp.ru_stime.tv_sec;
    t->start_sys_time.tv_usec = tmp.ru_stime.tv_usec;
    t->start_user_time.tv_sec = tmp.ru_utime.tv_sec;
    t->start_user_time.tv_usec = tmp.ru_utime.tv_usec;
    break;
  case started:
  break;
  }
  t->state = started;
}

/**Function********************************************************************

  Synopsis           [ Stops the timer ]

  Description        [ The timer should be in the started state. The effect
  of this function is that the timer stops incrementing its internal 
  time counters and goes to the stopped state. ]

  SideEffects        [ Modifies the timer state. Performs a getrusage
  system call. ]

  SeeAlso            [ timer_Reset timer_Start ]

******************************************************************************/
void
timer_Stop
(timer t)
{
  if (t->state == started) {
    timerElapsedTime(t->start_sys_time.tv_sec,
                     t->start_sys_time.tv_usec,
                     t->start_user_time.tv_sec,
                     t->start_user_time.tv_usec,
                     & t->elapsed_sys_time.tv_sec,
                     & t->elapsed_sys_time.tv_usec,
                     & t->elapsed_user_time.tv_sec,
                     & t->elapsed_user_time.tv_usec);
    t->start_sys_time.tv_sec=0;
    t->start_sys_time.tv_usec=0;
    t->start_user_time.tv_sec=0;
    t->start_user_time.tv_usec=0;
    t->state = stopped;
  }
}

/**Function********************************************************************

  Synopsis           [ Prints the timer values ]

  Description        [ Prints message, user and and system time counters 
  (the format is seconds.milliseconds) to outstream. Does not affect the
  timer state. ]

  SideEffects        [ Sends output to outstream. Performs a getrusage
  system call if the timer is in the started state. ]

  SeeAlso            [optional]

******************************************************************************/
void
timer_Display
(FILE * outstream,
 char * message,
 timer t)
{
  long sys_sec, sys_usec, user_sec, user_usec;
  float sys, user;
  sys_sec = sys_usec = user_sec = user_usec = 0;
  switch (t->state) {
    case reset:
      return;
    case started:
      timerElapsedTime(t->start_sys_time.tv_sec,
                       t->start_sys_time.tv_usec,
                       t->start_user_time.tv_sec,
                       t->start_user_time.tv_usec,
                       & sys_sec, & sys_usec,
                       & user_sec, & user_usec);
      /* executes through on purpose */
    case stopped:
      sys_sec += t->elapsed_sys_time.tv_sec;
      sys_usec += t->elapsed_sys_time.tv_usec;
      user_sec += t->elapsed_user_time.tv_sec;
      user_usec += t->elapsed_user_time.tv_usec;
    break;
  }
  sys = (float) sys_sec + ((float) sys_usec / 1000000);
  user = (float) user_sec + ((float) user_usec / 1000000);
  fprintf(outstream, "%s\nuser time: %.2fs, system time: %.2fs\n",
          message, user, sys);
}

/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [ Compute elapsed times. ]

  Description        [ Returns the different times elapsed since 
  the given times sys_sec, sys_usec, user_sec, user_usec. Counters are
  normalised such that microseconds counts between 0 and 999999. ]

  SideEffects        [ Results is passed by reference through 
  parameters res_sys_sec, res_sys_usec, res_user_sec, res_user_usec. ]

  SeeAlso            [optional]

******************************************************************************/
static void
timerElapsedTime
(long sys_sec,
 long sys_usec,
 long user_sec,
 long user_usec,
 long * res_sys_sec,
 long * res_sys_usec,
 long * res_user_sec,
 long * res_user_usec)
{
  struct rusage tmp;
  getrusage(0, & tmp);
  sys_sec = tmp.ru_stime.tv_sec - sys_sec;
  sys_usec = tmp.ru_stime.tv_usec - sys_usec;
  user_sec = tmp.ru_utime.tv_sec - user_sec;
  user_usec = tmp.ru_utime.tv_usec - user_usec;
  while (sys_usec >= 1000000) {
    sys_usec = sys_usec - 1000000;
    sys_sec = sys_sec + 1;
  }
  while (sys_usec < 0) {
    sys_usec = sys_usec + 1000000;
    sys_sec = sys_sec - 1;
  }
  while (user_usec >= 1000000) {
    user_usec = user_usec - 1000000;
    user_sec = user_sec + 1;
  }
  while (user_usec < 0) {
    user_usec = user_usec + 1000000;
    user_sec = user_sec - 1;
  }
  * res_sys_sec = sys_sec;
  * res_sys_usec = sys_usec;
  * res_user_sec = user_sec;
  * res_user_usec = user_usec;
}

