/** CHeaderFile ********************************************************

  FileName    [ spec.h ]

  PackageName [ spec ]

  Synopsis    [ Data and actions for CVC specification parser ]

  Description [ The spec language is currently quite simple.
  A specification in spec declares abbreviations and specifies
  properties for some VHDL design entity.
  An abbreviation represents a VHDL expression and can be
  refered to in properties. 
  A property is a CTL term where the atomic propositions can be
  any legal VHDL expressions.
  VHDL expressions are represented with type tTree (see cv(3)).
  CTL terms are represented with type ctl_Term_t.
  Parsing a spec expression generates two attributes: ctl_term
  and vhdl_term. For a given expression, there is always one
  such attribute equal to 0, depending on the expression being
  a VHDL expression or a CTL formula. A VHDL expression is a
  VHDL operator or the operator '=>' (implies) applied to any
  VHDL expression. A CTL formula is any logic or temporal
  operator applied to a CTL formula, or a temporal operator 
  applied to any expression. ]

  SeeAlso     [ cv(3) ctl ]

  Author      [ David Deharbe. ]

  Copyright   [ Copyright (c) 1996, Carnegie Mellon University.
                All rights reserved. ]

  Revision    [ 4.2 ]

***********************************************************************/
#ifndef _SPEC
#define _SPEC

#include <Errors.h>
#include <cv.h>
#include <ctl.h>
#include <memuser.h>

/*--------------------------------------------------------------------*/
/* Constant declarations                                              */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Type declarations                                                  */
/*--------------------------------------------------------------------*/
typedef struct spec_NodeStruct_t_ * spec_Node_t;
/* spec nodes are divided into abbreviations, properties and terms */
typedef enum {spec_Abbrev, spec_Property, spec_Term} spec_NodeType_t;
/* spec operators are ctl or vhdl operators */
typedef enum {
  spec_Not, 
  spec_Implies, spec_And, spec_Or, spec_Xor, spec_Xnor, spec_Nand, spec_Nor,
  spec_Eq, spec_Neq, spec_Le, spec_Lt, spec_Ge, spec_Gt,
  spec_EX, spec_AX, spec_EF, spec_AF, spec_EG, spec_AG,
  spec_EU, spec_AU, spec_EW, spec_AW, spec_Stable
} spec_Operator_t;

/*--------------------------------------------------------------------*/
/* Structure declarations                                             */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Variable declarations                                              */
/*--------------------------------------------------------------------*/
char * spec_name;
char * spec_design_library;
cvn spec_design;
cvn spec_fairness;
cvn spec_invariant;
spec_Node_t spec_abbreviations;
spec_Node_t spec_properties;

/*--------------------------------------------------------------------*/
/* Macro declarations                                                 */
/*--------------------------------------------------------------------*/

/**AutomaticStart******************************************************/

/*---------------------------------------------------------------------------*/
/* Function prototypes                                                       */
/*---------------------------------------------------------------------------*/

extern ctl_term spec_ActionBinaryCTLTerm(spec_Operator_t op, ctl_term ctl1, cvn vhdl1, ctl_term ctl2, cvn vhdl2, int * nberrors, tPosition pos);
extern cvn spec_ActionBinaryVHDLTerm(spec_Operator_t op, ctl_term ctl1, cvn vhdl1, ctl_term ctl2, cvn vhdl2, int * nberrors, tPosition pos);
extern ctl_term spec_ActionUnaryCTLTerm(spec_Operator_t op, ctl_term ctl, cvn vhdl, int * nberrors, tPosition pos);
extern ctl_term spec_ActionMacroCTLTerm(spec_Operator_t op, int * nberrors, tPosition pos);
extern cvn spec_ActionUnaryVHDLTerm(spec_Operator_t op, ctl_term ctl, cvn vhdl, int * nberrors, tPosition pos);
extern cvn spec_ActionIdentifierTerm(char * identifier, int * nberrors, tPosition pos);
extern cvn spec_ActionCharLiteralTerm(char literal, int * nberrors, tPosition pos);
extern void spec_ActionStartSpecification(char * name);
extern void spec_ActionEndSpecification(void);
extern cvn spec_ActionLoadDesign(char * library, char * entity, char * architecture, tPosition pos);
extern int spec_ActionCheckAbbreviation(char * id, int * nberrors, tPosition pos);
extern void spec_ActionCheckCondition(cvn expression, int * nberrors, tPosition pos);
extern void spec_ActionCheckInvariant(cvn expression, int * nberrors, tPosition pos);
extern spec_Node_t spec_BasicNew(pointer info, char * name, spec_Node_t next, pointer value);
extern void spec_BasicFree(spec_Node_t n);
extern pointer spec_BasicQInfo(spec_Node_t n);
extern char * spec_BasicQName(spec_Node_t n);
extern spec_Node_t spec_BasicQValue(spec_Node_t n);
extern spec_Node_t spec_BasicQNext(spec_Node_t n);
extern void spec_BasicSInfo(spec_Node_t n, pointer info);
extern void spec_BasicSName(spec_Node_t n, char * name);
extern void spec_BasicSNext(spec_Node_t n, spec_Node_t next);
extern void spec_BasicSValue(spec_Node_t n, pointer value);
extern spec_Node_t spec_BasicAppend(spec_Node_t l1, spec_Node_t l2);
extern void spec_BasicDisplay(FILE * outstream, spec_Node_t n);

/**AutomaticEnd********************************************************/

#endif /* _SPEC */
