/**CFile****************************************************************

  FileName    [ modelWitness.c ]
  PackageName [ model ]
  Synopsis    [ Find and display counter examples to CTL properties ]
  Description [ Implements the counterexample generation for CTL.
  Three routines model_WitnessEX, model_WitnessEU, and model_WitnessEG
  compute paths that exhibit the counterexample for the three basic CTL
  operators EX, EU and EG ]
  SeeAlso     [ modelCheck.c ]
  Author      [ David Deharbe ]
  Copyright   [ Copyright (C) 1996, Carnegie Mellon University.
                All right reserved. ]
  Revision    [ $Id: modelWitness.c,v 1.1 1996/05/20 21:07:58 david Exp david $ ]

***********************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "modelInt.h"

static char rcsid [] = "$Id: modelWitness.c,v 1.1 1996/05/20 21:07:58 david Exp david $";

/*--------------------------------------------------------------------*/
/* Structure declarations                                             */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Type declarations                                                  */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Variable declarations                                              */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Macro declarations                                                 */
/*--------------------------------------------------------------------*/

/**AutomaticStart******************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static void _ComputeTransition(Model m, bdd source, bdd target, bdd * transition, bdd * destination);
static ModelPath _WitnessEG_NoFairness(Model m, FILE * outstream, bdd s, bdd f, bdd EGf);
static ModelPath _WitnessEG_Fairness(Model m, FILE * outstream, bdd s, bdd f, bdd EGf, vbdd * approximations);

/**AutomaticEnd********************************************************/

/*--------------------------------------------------------------------*/
/* Definition of exported functions                                   */
/*--------------------------------------------------------------------*/
/** Function **
  Synopsis    [ Finds a path going to a configuration ]
  Description [ Computes and returns a finite path that leaves an
  initial state of the model and reaches the given configuration.
  If the configuration is not reachable, the returned value is 0. ]
  SideEffects [ Memory is allocated to represent the result. ]
  SeeAlso     [ ]
 */
ModelPath
model_WitnessConfig
(Model m, /* models a VHDL architecture */
 bdd config /* a configuration of the variables of the model */)
{
  bdd_manager bddm = m->bddm;
  bdd initial_state = model_RepresentationQInitialState(m);
  vbdd reachability = model_ReachabilityQFrontiers(m);
  vbdd fixpoint = vbdd_new();
  int idx = 0;
  bdd set, temp;

  while ((temp = bdd_and(bddm, config, vbdd_nth(reachability, idx)))
          == bdd_zero(bddm)) {
    ++idx;
  }
  bdd_free(bddm, temp);
  set = config;
  vbdd_add(bddm, fixpoint, set);
/* the implemenation is not the most efficient w.r.t. time, but it 
 * reuses the code of model_WitnessEU.
 * It is also possible to build the path while going backwards instead 
 * of building the vector "fixpoint" and call model_WitnessEU that
 * builds the path going forwards (but ModelPath has been programmed
 * to be constructed going forwards only).
 */ 
  while (idx != 0) {
    bdd tmp;
    --idx;
    tmp = model_ReachabilityStepBackward(m, set);
    tmp = bdd_andup2(bddm, vbdd_nth(reachability, idx), tmp);
    vbdd_add(bddm, fixpoint, tmp);
    bdd_free(bddm, set);
    set = tmp;
  }
  return model_WitnessEU(m, initial_state, bdd_one(bddm), config, fixpoint);
}

/** Function **
  Synopsis    [ Finds a trace exhibiting that EX f is valid in a state ]
  Description [ Computes and returns a finite path that exhibits that
                EX f stands in the state s of the model m.
                This path starts at s and has exactly one transition,
                to a state where f is valid. 
                It is up to the caller of the function to free the result. ]
  SideEffects [ Allocates memory to store the result.
                Build new BDDs. ]
  SeeAlso     [ ]
 */
ModelPath
model_WitnessEX
(Model m /* Kripke structure of a VHDL architecture */ , 
 bdd s   /* A state of m, where EX f is valid */, 
 bdd f   /* A condition over the state space of m */)
{

  /* declarations */
  bdd next, label;
  ModelPath result;

  /* statements */
  result = model_ExplicitNewPath();

  _ComputeTransition(m, s, f, & label, & next);

  model_ExplicitAppendTransitionToPath(result, label);
  model_ExplicitAppendTransitionToPath(result, next);

  return result;
}

/** Function **
  Synopsis    [ Finds a trace exhibiting that EG f is valid in a state ]
  Description [ Computes and returns an infinite path that exhibits that
  EG f stands in the state s of the model m. The result path starts at 
  s and is infinite. It is up to the caller of the function to free the 
  result. 
  The algorithm is the simplest of the two algorithms presented in
  CMU-CS-94-204 "Efficient Generation of Counterexamples and Witnesses 
  in Symbolic Model Checking" E. Clarke, O. Grumberg, K. McMillan, 
  X. Zhao ]
  SideEffects [ Allocates memory to store the result.
                Build new BDDs. ]
  SeeAlso     [ ]
 */
ModelPath
model_WitnessEG
(Model m       /* Kripke structure modeling a VHDL architecture body */, 
 FILE * outstream, 
 bdd s         /* A state of m where EG f is valid */,
 bdd f         /* The set of valid states of m where f is valid */,
 bdd EGf       /* The set of valid states of m where EG f is valid */,
 vbdd * approximations /* */)
{

  if (* approximations) {
    return _WitnessEG_Fairness (m, outstream, s, f, EGf, approximations);
  } else {
    return _WitnessEG_NoFairness (m, outstream, s, f, EGf);
  }
}


/** Function **
  Synopsis    [ Finds a trace exhibiting that E[fUg] is valid in a state ]
  Description [ Computes and returns a finite path that exhibits that
                E[fUg] stands in the state s of the model m.
                The result path starts at s and is finite. It is up to 
                the caller of the function to free the result. ]
  SideEffects [ Allocates memory to store the result.
                Build new BDDs. ]
 */

ModelPath
model_WitnessEU
(Model m, bdd s, bdd f, bdd g,
 vbdd frontiers /* the list of intermediate results in the fixpoint computation */)
{
  ModelPath result;
  bdd current;
  int idx;
  bdd tmp;

  /* find the frontier set where s appear */
  idx = 0;
  
  while (((tmp = bdd_and(m->bddm, s, vbdd_nth(frontiers, idx))) 
          == bdd_zero(m->bddm))
         && (idx < vbdd_length(frontiers))) {
    ++idx;
  }
  /* if not found, there is a problem... */
  if (tmp == bdd_zero(m->bddm)) {
    fprintf(stderr, "error <model/model_WitnessEU> : inconsistency\n");
    return 0;
  }

  /* build the transitions of the path now */
  current = s;
  result = model_ExplicitNewPath();
  while (idx > 0) {
    bdd next, label;
    --idx;
    _ComputeTransition(m, current, vbdd_nth(frontiers, idx), & label, & next);
    model_ExplicitAppendTransitionToPath(result, label);
    current = next;
  }
  model_ExplicitAppendTransitionToPath(result, current);

  return result;
}

/*--------------------------------------------------------------------*/
/* Definition of internal functions                                   */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Definition of static functions                                     */
/*--------------------------------------------------------------------*/

/** Function **
  Synopsis    [ Find a transition between two sets of states ]
  Description [ "source" and "target" are two subsets of the state
  space of "m". The routine stores at the address "current" a BDD that 
  represents a valuation of the state and input variables of m that 
  belongs to the set source and transition to the set target. The
  routine also stores at the address "destination" the BDD of the
  valuations of state variables that are reached from "transition". ]
  SideEffects [ Manipulates BDDs. The parameters "transition" and
  "destination" are used to store results. ]
 */

static void
_ComputeTransition
(Model m,
 bdd source,
 bdd target,
 bdd * transition,
 bdd * destination)
{
  bdd tmp;
  * destination = 
    bdd_andup2(m->bddm, target, 
               model_ReachabilityStepForward(m, source));
  tmp =
    bdd_andup2(m->bddm, source, 
               model_ReachabilityStepBackward(m, * destination));
  * transition = model_ExplicitComputeCondition(m, tmp, MODEL_ALLMASK);
  bdd_free(m->bddm, tmp);
}


/** Function **
  Synopsis    [ Find witness for EG formula without fairness constraints ]
  SideEffects [ Allocates memory, prints to outstream. ]
 */

static ModelPath
_WitnessEG_NoFairness
(Model m       /* Kripke structure modeling a VHDL architecture body */, 
 FILE * outstream, 
 bdd s         /* A state of m where EG f is valid */,
 bdd f         /* The set of valid states of m where f is valid */,
 bdd EGf  /* The set of valid states of m where EG f is valid */)
{

  /* declarations */
  ModelPath result;
  bdd next, label;
  bdd current;
  int reached_loop;
  bdd_manager bddm;

  /* statements */

  reached_loop = 0;
  result = model_ExplicitNewPath();
  current = s;
  bddm = m->bddm;
  do {
    ModelCheckComputation loop_flag, loop_flag_aux;
    bdd tmp;
    vbdd frontiers;
    
    frontiers = vbdd_new();
    loop_flag_aux = model_CheckEU(m, outstream, f, current);
    loop_flag = model_CheckEX(m, outstream, model_CheckQSolution(loop_flag_aux));
    /* the first state of the loop has been reached */
    if ((tmp = bdd_and(bddm, current, model_CheckQSolution(loop_flag)))
        != bdd_zero(bddm)) { 
      bdd_free(bddm, tmp);
      model_CheckFreeComputation(loop_flag);

      /* pick a state in the image set of current that is at the
         start of a loop along which f holds: next:bdd, next:state */      
      {
        bdd solutions, nexts, intersection;
        solutions = model_CheckQSolution(loop_flag_aux);
        nexts = model_ReachabilityStepForward(m, current);
        intersection = bdd_andup2(bddm, solutions, nexts);
        next = model_ExplicitComputeCondition(m, intersection, 
                                                   MODEL_STATEMASK);
        bdd_free(bddm, intersection);
      }
      /* compute the value of the inputs to get from current to next:
         label:bdd, label:ModelConfig */
      tmp = bdd_andup2(bddm, current,
                       model_ReachabilityStepBackward(m, next));
      label = model_ExplicitComputeCondition(m, tmp, MODEL_ALLMASK);
      bdd_free(bddm, tmp);

      /* add the transition to the trace result */
      model_ExplicitAppendTransitionToPath(result, label);

      /* indicate that we are done with looking up for the loop */
      reached_loop = 1;

      /* add the transitions along the loop to the trace */
      model_ExplicitAppendPaths(result, 
                               model_WitnessEU(m, next, f, current,
                                               (vbdd) model_CheckQInfo(loop_flag_aux)));
      vbdd_free(bddm, frontiers);
      /* mark the end of the trace with a dummy transition to indicate
         loop */
      model_ExplicitLoopPath(result, label);

    /* still in the finite prefix segment */
    } else {
      bdd next;

      bdd_free(bddm, tmp);
      model_CheckFreeComputation(loop_flag);
      model_CheckFreeComputation(loop_flag_aux);

      /* pick a next state */
      next = model_ExplicitComputeCondition(m, 
        model_ReachabilityStepForward(m, bdd_and(bddm, current, EGf)),
        MODEL_STATEMASK);
      /* compute the corresponding input */
      label = model_ExplicitComputeCondition(m, 
        bdd_and(bddm, current,
	        model_ReachabilityStepBackward(m, next)),
        MODEL_ALLMASK);
                     
      /* add a transition */
      model_ExplicitAppendTransitionToPath(result, label);
      
      /* and keep searching for a loop */
      current = next;
    }
  } while (!reached_loop);

  return result;
}

/** Function **
  Synopsis    [ Find witness for EG formula with fairness constraints ]
  SideEffects [ Allocates memory, prints to outstream. ]
 */
static ModelPath
_WitnessEG_Fairness
(Model m       /* Kripke structure modeling a VHDL architecture body */, 
 FILE * outstream, 
 bdd s         /* A state of m where EG f is valid */,
 bdd f         /* The set of valid states of m where f is valid */,
 bdd EGf       /* The set of valid states of m where EG f is valid */,
 vbdd * approximations /* */)
{

  /* hypothesis: ref_count(s) > 0
                 ref_count(f) > 0
                 ref_count(EGf) > 0 
                 forall v in approximations:
                   forall f in v:
                     ref_count(f) > 0 */

  /* declarations */
  bdd_manager bddm = m->bddm;
  ModelPath result;         /* the successive states are appended to result */
  bdd current_state;        /* the current state in the witness production */
  int witness_completed;    /* flags indicating that the witness is computed */

  result = model_ExplicitNewPath(); /* result is an empty path */
  current_state = bdd_identity(bddm, s);
  /* :- bdd_refs(current_state) > 1 */
  witness_completed = 0;

  /* loop on the strongly connected components until the witness is computed */
  do {                                              /* INVARIANT: witness_completed == 0 */
                                                    /*   * approximations != 0  */
                                                    /*   bdd_refs(current_state) > 0 */
    ModelCheckComputation loop_flag, loop_flag_aux;
    bdd current_image_set, start_cycle, label, tmp;
    vbdd * remaining_approximations; 
    remaining_approximations = approximations;
    current_image_set = start_cycle = label = tmp = 0;
    /* determine what states have a successor where there is a path leading
       to the state current_state where f holds infinitely */
    start_cycle = 0;
    loop_flag = loop_flag_aux = 0;
    current_image_set = bdd_identity(bddm, bdd_zero(bddm));
    /* :- bdd_refs(current_image_set) > 0 */
    do {
      vbdd * approximation_ptr;
      int found, iteration;
      iteration = found = 0 ;
      /* :- bdd_refs(current_image_set) >= 0 */
      current_image_set = 
        bdd_andup2(bddm, EGf,
                   model_ReachabilityStepForward(m, current_state));
      /* :- bdd_refs(current_image_set) > 0 */
      do {                                           /* INVARIANT: found == 0 */
        approximation_ptr = remaining_approximations;
        while ((found == 0) && (* approximation_ptr != 0)) {
          tmp = bdd_and(bddm, current_image_set, 
                        vbdd_nth(* approximation_ptr, iteration));
          /* bdd_refs(tmp) > 0 */
          if (tmp != bdd_zero(bddm)) {
            bdd_free(bddm, current_image_set);
            /* bdd_refs(current_image_set) >= 0 */
            found = 1;
          } else {
            bdd_free(bddm, tmp);
            /* bdd_refs(tmp) >= 0 */
            ++approximation_ptr;
          }
        }
        if (!found) {
          ++iteration;
        }
      } while (found == 0);
      /* (found == 1) && (bdd_refs(current_image_set) >= 0) && (bdd_refs(tmp) > 0) &&
         (bdd_refs(current_state) > 0) */
      /* compute which successor of the state current_state is closest to
         a fairness constraint: pick it from tmp */
      _ComputeTransition(m, current_state, tmp, & label, & current_image_set);
      /* (found == 1) && (bdd_refs(current_image_set) > 0) && (bdd_refs(tmp) > 0) &&
         (bdd_refs(current_state) > 0) && (bdd_refs(label) > 0) */
      bdd_free(bddm, current_state);
      /* (found == 1) && (bdd_refs(current_image_set) > 0) && (bdd_refs(tmp) > 0) &&
         (bdd_refs(current_state) >= 0) && (bdd_refs(label) > 0) */
      bdd_free(bddm, tmp);
      /* (found == 1) && (bdd_refs(current_image_set) > 0) && (bdd_refs(tmp) >= 0) &&
         (bdd_refs(current_state) > 0) && (bdd_refs(label) > 0) */
      model_ExplicitAppendTransitionToPath(result, label); 
      /* label |=   EX E[f U (EG f)&h] */
      if (start_cycle == 0) {
        start_cycle = label;
        loop_flag_aux = model_CheckEU(m, outstream, f, start_cycle);
        loop_flag = model_CheckEX(m, outstream, model_CheckQSolution(loop_flag_aux));
      }
      current_state = current_image_set;
      /* (found == 1) && (bdd_refs(current_image_set) > 0) && (bdd_refs(tmp) >= 0) &&
         (bdd_refs(current_state) > 0) && (bdd_refs(label) > 0) */
      {
         vbdd fairness;
         fairness = * approximation_ptr;
         * approximation_ptr = * remaining_approximations;
         * remaining_approximations = fairness;
      }
      ++remaining_approximations;
      /* at this point: 
       *   fairness is the closest fulfilled fairness constraint 
       *   iteration is the index of the first approximation in fairness 
       *   where a successor of current can be found.
       */
      while (iteration > 0) { /* builds the path from current to the fairness */
        --iteration;
        _ComputeTransition(m, current_state, 
                           vbdd_nth(* approximation_ptr, iteration),
                           & label, & current_image_set);
        /* (found == 1) && (bdd_refs(current_image_set) > 0) && (bdd_refs(tmp) >= 0) &&
           (bdd_refs(current_state) > 0) && (bdd_refs(label) > 0) */
        model_ExplicitAppendTransitionToPath(result, label);/* adds the successive
                                                             * states leading to h
                                                             * label |=
                                                             *   E[f U (EG f) & h]
                                                             */
        bdd_free(bddm, current_state);
        /* (found == 1) && (bdd_refs(current_image_set) > 0) && (bdd_refs(tmp) >= 0) &&
           (bdd_refs(current_state) >= 0) && (bdd_refs(label) > 0) */
        current_state = bdd_identity(bddm, current_image_set);
        /* (found == 1) && (bdd_refs(current_image_set) > 0) && (bdd_refs(tmp) >= 0) &&
           (bdd_refs(current_state) > 0) && (bdd_refs(label) > 0) */
      }
      /* the states leading to a given fairness have been added to
       * the result 
       * current_state |= h 
       */
    } while (* remaining_approximations);
    /* check if it is possible to extend the path into a cycle */
    if ((tmp = bdd_and(bddm, current_state, model_CheckQSolution(loop_flag)))
         != bdd_zero(bddm)) {
      ModelPath suffix;
      bdd_free(bddm, tmp);
      _ComputeTransition(m, current_state, model_CheckQSolution(loop_flag_aux),
                         & label, & current_image_set);
      bdd_free(bddm, current_state);
      model_ExplicitAppendTransitionToPath(result, label);
      suffix = model_WitnessEU(m, current_image_set, f, start_cycle, 
                               (vbdd) model_CheckQInfo(loop_flag_aux));
      model_ExplicitAppendPaths(result, suffix);
      model_ExplicitFreePath(suffix);
      model_ExplicitLoopPath(result, start_cycle);
      
      /* indicate that we are done with looking up for the loop */
      witness_completed = 1;
      /* still in the finite prefix segment */
    } else {
      bdd_free(bddm, tmp);
      model_CheckFreeComputation(loop_flag);
      model_CheckFreeComputation(loop_flag_aux);
      
      tmp = bdd_and(bddm, EGf, model_ReachabilityStepForward(m, current_state));
      _ComputeTransition(m, current_state, tmp, & label, & current_image_set);
      bdd_free(bddm, tmp);
      
      /* add the transition */
      model_ExplicitAppendTransitionToPath(result, label);
      
      /* and keep searching for a loop */
      current_state = current_image_set;
    }
  } while (!witness_completed);

  return result;
}

