/** CFile **************************************************************

  FileName    [ modelRepresentation.c ]

  PackageName [ model ]

  Synopsis    [ Definitions of basic construction/destruction routines ]

  Description [ Provides a set of routines that permit to incrementally
  build a BDD-based model of a VHDL architecture, and to query the value
  of the various components of the model. ]

  SeeAlso     [ ModelStruct_t ]

  Author      [ David Deharbe ]

  Copyright   [ Copyright (C) 1996, Carnegie Mellon University.
                All right reserved. ]

  Revision    [ 4.2 ]

***********************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>

#include "modelInt.h"

/*--------------------------------------------------------------------*/
/* Structure declarations                                             */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Type declarations                                                  */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Variable declarations                                              */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Macro declarations                                                 */
/*--------------------------------------------------------------------*/

/**AutomaticStart******************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static bdd _RepresentationAddInit(Model m, bdd i);
static int _RepresentationQSignal(Model m, cvn decl);
static int _RepresentationQVariable(Model m, cvn decl);

/**AutomaticEnd********************************************************/

/*--------------------------------------------------------------------*/
/* Definition of exported functions                                   */
/*--------------------------------------------------------------------*/

/* ------------------------------------------------------------ */
                                           /* Memory management */

/** Function **
  Synopsis    [ Creates a new model for a VHDL architecture body ]
  Description [ Allocates memory and initializes a BDD-based model 
  for a VHDL architecture body. ]
  SideEffects [ Memory is managed with David Long's mem(3) library.  ]
  SeeAlso     [ model_RepresentationFree ModelStruct_t ]
 */
Model /* The result is a pointer to a structure */
model_RepresentationNew (
  bdd_manager bddm /* manager of subsequent BDD operations */,
  cvn design)
{
  /* declarations */
  Model result ;
  /* statements */
  result = (Model) mem_get_block(sizeof(ModelStruct_t));
  result-> bddm=bddm;
  result-> design = design;
  result-> nb_propositions=0;
  result-> nb_signals=0;
  result-> max_nb_signals=8;
  result-> signals=(ModelSignalStruct_t *) mem_get_block(8*sizeof(ModelSignalStruct_t)) ;
  result-> nb_variables=0;
  result-> max_nb_variables=8;
  result-> variables= (ModelVariableStruct_t *) mem_get_block(8*sizeof(ModelVariableStruct_t)) ;
  result-> stable= bdd_identity(bddm, bdd_one(bddm));
  result-> init= bdd_identity(bddm, bdd_one(bddm));
  result->still_inputs= 0;
  result->vars = vbdd_new();
  result->state_vars = vbdd_new();
  result->input_vars = vbdd_new();
  result->fairness_constraints = 0;
  result->fair_states = 0;
  result->invariant = bdd_identity(bddm, bdd_one(bddm));
  return result;
}

/** Function **
  Synopsis    [ Deletes a VHDL architecture body BDD-based model ]
  Description [ Disallocates memory that contains data previously
  created with model_New,  to represent a BDD-based model for a 
  VHDL architecture body. ]
  SideEffects [ Memory is managed with David Long's mem(3) library.  ]
  SeeAlso     [ model_RepresentationNew ModelStruct_t ] 
 */

void 
model_RepresentationFreeModel
(Model m /* model to delete */)
{
  mem_free_block(m->signals);
  mem_free_block(m->variables);
  mem_free_block(m);
}

/** Function **
  Synopsis    [ Adds a variable to the model of a VHDL architecture ]
  Description [ Takes as input a variable declaration, together with 
  two BDD vectors (of same length). The first vector contains 
  representations of atomic propositions of the binary coding of the
  variable. The second vector contains representations of constants 0
  and 1 that code the initial value of the variable.
  Adds the atomic propositions to the set of state variables of the
  model. Computes the condition that the variable value equals to
  initial value, and adds it to the initial state of the model.
  Returns the number of variables represented in the model. 
  The parameters decl and coding shall not be freed by the caller,
  since the routine only copies their address and not their content.
  Also the reference counters of the BDDs in coding are not incremented
  by this routine. ]
  SideEffects [ May modify the field max_nb_variables, in which case 
  additional memory is allocated to represent the variables in the 
  model.  The fields nb_propositions, state_vars, variables and init 
  of the model structure m. ]
  SeeAlso     [ ModelStruct_t ]
 */
int /* the total number of variables added so far */
model_RepresentationAddVariable (
  Model m /* model to update */,
  cvn decl  /* declaration of the VHDL variable */,
  vbdd coding /* BDD-based coding of the variable*/,
  vbdd init_coding /* BDD-based coding of the variable initial value */,
  bdd valid /* valid BDD codes */)
{
  /* declarations */
  ModelVariableStruct_t * v;
  /* statements */
  if (m->nb_variables == m->max_nb_variables) {
    m->max_nb_variables*=2;
    m->variables = (ModelVariableStruct_t *)
      mem_resize_block((pointer) m->variables, 
		       sizeof(ModelVariableStruct_t)*(m->max_nb_variables));
  }
  m->nb_propositions += vbdd_length(coding);
  v = m->variables + m->nb_variables;
  v->declaration = decl;
  v->coding = coding;
  vbdd_append(m->bddm, m->vars, coding);
  vbdd_append(m->bddm, m->state_vars, coding);
  v->init_coding = init_coding;
  v->transition = 0;
  m->nb_variables++;
  _RepresentationAddInit(m, vbdd_eq(m->bddm, coding, init_coding));
  model_RepresentationAddInvariant(m, valid);
  return m->nb_variables;
}

/** Function **
  Synopsis    [ Adds a signal to the model of a VHDL architecture ]
  Description [ Takes as input a signal declaration, together with 
  two BDD vectors (of same length). The first vector contains 
  representations of atomic propositions of the binary encoding of 
  the signal. The second vector contains representations of 
  constants 0 and 1 that code the default value of the signal.  
  Adds the atomic propositions to the set of state variables of 
  the model.  Computes the condition that the signal value equals 
  to its default value, and adds it to the initial state of the 
  model.  
  Returns the number of signals represented in the model. The 
  parameters decl and coding shall not be freed by the caller, since 
  the routine only copies their address and not their content. Also 
  the reference counters of the BDDs in coding are not incremented 
  by this routine. ]
  SideEffects [ May modify the field max_nb_signals, in which case 
  additional memory is allocated to represent the variables in the 
  model.
  The fields nb_propositions, state_vars, signals and init of the 
  model structure m. ]
  SeeAlso     [ ModelStruct_t ]
 */
int /* the total number of variables added so far */
model_RepresentationAddSignal (
  Model m /* model to update */,
  cvn decl  /* declaration of the VHDL signal */,
  vbdd coding /* BDD-based coding of the signal */,
  vbdd init_coding /* BDD-based coding of the signal's default value */,
  bdd valid /* valid BDD codes */)
{
  /* declarations */
  ModelSignalStruct_t * s;
  /* statements */
  if (m->nb_signals == m->max_nb_signals) {
    m->max_nb_signals*=2;
    m->signals= (ModelSignalStruct_t *)
      mem_resize_block((pointer) m->signals,
                        sizeof(ModelSignalStruct_t)*(m->max_nb_signals));
  }
  m->nb_propositions += vbdd_length(coding);
  s = m->signals+m->nb_signals;
  m->nb_signals++;
  s->declaration = decl;
  s->coding = coding;
  s->init_coding = init_coding;
  s->nb_drivers = 0;
  vbdd_append(m->bddm, m->vars, coding);
  if (qMode(decl) != kIn) {
    vbdd_append(m->bddm, m->state_vars, coding);
    s->max_nb_drivers = 1;
    s->drivers = mem_get_block(1*sizeof(ModelDriverStruct_t)) ;
    _RepresentationAddInit(m, vbdd_eq(m->bddm, coding, init_coding));
  } else {
    vbdd_append(m->bddm, m->input_vars, coding);
    s->max_nb_drivers = 0;
    s->drivers = 0;
  }
  s->previous = 0;
  model_RepresentationAddInvariant(m, valid);
  return m->nb_signals;
}

/** Function **
  Synopsis    [ Adds a stability condition ]
  Description [ Takes as input a BDD that represents a stability 
  condition of some concurrent statement of the architecture.  
  Computes the result stability condition (boolean conjunction) 
  and returns it.  The reference counter of the BDD st is not 
  modified by this routine. ]
  SideEffects [ Modify the field stable of the model structure m. ]
  SeeAlso     [ ModelStruct_t ]
 */
bdd
model_RepresentationAddStable (
  Model m /* model to update */,
  bdd st /* BDD of the stability condition of some process */)
{
  do {
    bdd tmp;
    tmp = bdd_and(m->bddm, st, m->stable);
    if (bdd_overflow(m->bddm)) {
      bdd_resize_manager(m->bddm);
    } else {
      bdd_free(m->bddm, m->stable);
      m->stable = tmp;
      break;
    }
  } while (1);
  return m->stable;
}

/** Function **
  Synopsis    [ Adds a signal driver ]
  Description [ Adds a driver for a given signal in a given 
  statement. The signal must have been added to the model before,
  otherwise an error occurs.  The driver is given as a vector of 
  BDD that represents the transition functions of the atomic 
  propositions that code the driven signal.  Although it is possible 
  to add several driver to a signal, this is not subsequently handled 
  correctly.  The BDD vector trans shall not be altered by the caller 
  since the routine just copies its address.  Also the reference 
  counters of the BDD in trans are not modified. ]
  SideEffects [ Modifies the internal state of the model.  If 
  boundaries are reached, may require more memory via the mem(3) 
  library. ]
  SeeAlso     [ ModelStruct_t ]
 */
void
model_RepresentationAddDriver (
  Model m /* model to update */,
  cvn decl /* declaration of the driven signal */,
  cvn stm /* statement driving the signal */,
  vbdd trans /* value assigned to the signal */)
{
  /* declarations */
  int idx ;
  /* statements */
  ModelSignalStruct_t * s;
  ModelDriverStruct_t * drv;
  idx = _RepresentationQSignal(m, decl);
  assert (idx != -1); /* s must have been put there before */
  s = m->signals+idx;
  if (s->nb_drivers == s->max_nb_drivers) {
    s->max_nb_drivers*=2;
    s->drivers= (ModelDriverStruct_t *)
      mem_resize_block((pointer) s->drivers, 
                       sizeof(ModelDriverStruct_t)*(s->max_nb_drivers));
  }
  drv = s->drivers + s->nb_drivers++;
  drv->process = stm;
  drv->transition = trans;
}

/** Function **
  Synopsis    [ Adds a variable carrying a signal previous value ]
  Description [ In order to be able to detect events on a signal, 
  the model must keep track the previous value of this signal. The 
  signal must have been added to the model before, otherwise an 
  error occurs.  This routine adds an implicit VHDL variable to the 
  model to realize this.  The initial value of the added variable 
  is set to the initial value of the signal.  The transition 
  function of the added variable is equal to the boolean coding of 
  the signal.  Also the data associated to this signal in the model 
  is updated to include a reference to the added variable and the 
  BDD of the condition corresponding to the occurence of an event 
  on the signal is computed and stored with the signal.  The BDD 
  vector coding shall not be altered by the caller since the routine 
  just copies its address.  Also the reference counters of the BDD in 
  trans are not modified. ]
  SideEffects [ Modifies the internal state of the model: adds an 
  element to the field variables, and modifies the element of the 
  field signals that correspond to the given signal parameter.  May 
  allocate additional memory via the mem(3) library. ]
  SeeAlso     [ ModelStruct_t ]
 */
void
model_RepresentationAddPrevious (
  Model m /* model to update */,
  cvn decl /* signal declaration */,
  cvn prev /* declaration of the added implicit variable */,
  vbdd coding /* boolean encoding of the added implicit variable */,
  bdd valid /* valid boolean codings */)
{
  int idx, idxp;
  ModelSignalStruct_t * s;
  idx = _RepresentationQSignal(m, decl);
  assert (idx != -1); /* s must have been put there before */
  idxp= model_RepresentationAddVariable(m, prev, coding, m->signals[idx].init_coding, valid)-1;
  s = m->signals+idx;
  m->variables[idxp].transition=vbdd_copy(m->bddm, s->coding);
  s->previous=idxp;
  s->event=vbdd_neq(m->bddm, s->coding, m->variables[idxp].coding);
}

/** Function **
  Synopsis    [ Adds the transitions of a variable ]
  Description [ Adds the transitions functions of the atomic 
  propositions of a VHDL variable to the model.  These transition 
  functions must have been built from the assignment statements to 
  the variable in some process of the VHDL description.  The 
  variable must have been added to the model before, otherwise an 
  error occurs.  The BDD vector trans shall not be altered by the 
  caller since the routine just copies its address.  Also the 
  reference counters of the BDD in trans are not modified. ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
void
model_RepresentationAddTransition (
  Model m /* model to update */,
  cvn decl /* declaration of the variable */,
  vbdd trans /* BDDs of the transition functions of the variable coding */)
{
  /* declarations */
  int idx ;
  ModelVariableStruct_t * var;
  /* statements */
  idx = _RepresentationQVariable(m, decl);
  /* assert decl has been added before */
  assert (idx != -1);
  var = m->variables+idx;
  var->transition = trans;
}

/** Function **
  Synopsis    [ Queries the BDD coding of a model signal ]
  Description [ Returns the BDD-based symbolic coding of the given 
  signal.  The signal must have previously stored into the model 
  with model_RepresentationAddSignal, otherwise an error occurs. ]
  SideEffects [ none ]
  SeeAlso     [ model_RepresentationAddSignal ]
 */
vbdd
model_RepresentationQCoding (
  Model m /* queried model */,
  cvn decl    /* signal declaration */)
{
  if (IsA(decl, kSignalDeclaration)) {
    return m->signals[_RepresentationQSignal(m, decl)].coding;
  } else {
    return m->variables[_RepresentationQVariable(m, decl)].coding;
  }
}

/** Function **
  Synopsis    [ Queries signal event condition ]
  Description [ Returns the BDD of the condition that represents an 
  event on the given signal.  The signal must have previously stored 
  into the model with model_RepresentationAddSignal, and an implicit 
  previous value for this signal must have been stored with 
  model_RepresentationAddPrevious (this can be checked with the 
  predicate model_RepresentationSignalHasPrevious), otherwise an error 
  occurs. ]
  SideEffects [ none ]
  SeeAlso     [ model_RepresentationAddSignal
                model_RepresentationAddPrevious
                model_RepresentationSignalHasPrevious ]
 */
bdd
model_RepresentationQSignalEvent (
  Model m /* queried model */,
  cvn decl)
{
  return m->signals[_RepresentationQSignal(m, decl)].event;
}

/** Function **
  Synopsis    [ Queries the initial state ]
  Description [ Returns the BDD of the initial state. ]
  SideEffects [ none ]
  SeeAlso     [ ]
 */
bdd model_RepresentationQInitialState (
  Model m /* queried model */)
{
   return m->init;
}

/** Function **
  Synopsis    [ Queries if a signal has a previous value variable ]
  Description [ Returns a positive integer if the model has a 
  variable that carries the previous value of the given signal, 0 
  otherwise.  The signal must have previously stored into the model 
  with model_RepresentationAddSignal, otherwise an error occurs. ]
  SideEffects [ none ]
  SeeAlso     [ model_RepresentationAddSignal
                model_RepresentationAddPrevious]
 */
int model_RepresentationSignalHasPrevious (
  Model m /* queried model */,
  cvn s)
{
  return m->signals[_RepresentationQSignal(m, s)].previous;
}

/** Function **
  Synopsis    [ Builds the transition functions vector ]
  Description [ Builds the transition functions vector of the model 
  out of the atomic transition vector of each variable and signal 
  previously added to the model. This routine shall only be called 
  once all signals, variables (including implicit variables) have 
  been added to the model. 
  The parameter specification_support is either 0, and is not 
  used, or is the list of free variables in the specification.
  An abstraction of the model is performed based on this list. ]
  SideEffects [ Modifies the state of the BDD manager of the
  model m. Also sets the values of the field transitions of m. ]
  SeeAlso     [ model_RepresentationAddSignal
                model_RepresentationAddPrevious
                model_RepresentationAddVariable ]
 */
int 
model_RepresentationBuildTransition
(Model m /* model to update */,
 bdd * specification_support,
 int tradeoff)
{
  bdd_manager bddm = m->bddm;
  vbdd buffer, /* all state variables and functions */
       registers,
       functions;
  int i;
  unsigned before, after;

  buffer = vbdd_new();
  registers = vbdd_new();
  functions = vbdd_new();
  before = after = 0;

  /* compute variables (the state variables) and functions */
  {
    ModelSignalStruct_t * sig;
    ModelVariableStruct_t * var;
    for (i = m->nb_variables, var= m->variables; i; --i, ++var) {
      bdd * v, * f;
      for (v= vbdd_elements(var->coding), f= vbdd_elements(var->transition);
           * v !=0 ; 
           ++v, ++f) {
        ++before;
	vbdd_add(bddm, buffer, * v);
	vbdd_add(bddm, buffer, * f);
      }
    }
    sig = m->signals;
    for (i = m->nb_signals, sig= m->signals; i; --i, ++sig) {
      if (qMode(sig->declaration) != kIn){
	if (sig->nb_drivers == 1) {
          bdd * v, * f;
	  for (v=vbdd_elements(sig->coding), 
               f=vbdd_elements(sig->drivers[0].transition) ;
               * v != 0;
               ++v, ++f) {
            ++before;
	    vbdd_add(bddm, buffer, * v);
	    vbdd_add(bddm, buffer, * f);
          }
	} else if (sig->nb_drivers == 0) {
          bdd * v;
	  for (v= vbdd_elements(sig->coding); * v != 0; ++v) {
	    vbdd_add(bddm, buffer, * v);
	    vbdd_add(bddm, buffer, * v);
          }
	} else {
	    fprintf(stderr, "multiply driven signal - feature not accepted\n");
	    exit(0);
	}
      } else {
        bdd * v;
        for (v = vbdd_elements(sig->coding); (* v) != 0; ++v) {
            ++before;
	    vbdd_add(bddm, buffer, * v);
	    vbdd_add(bddm, buffer, 0);
        }
      }
    }
  }
  {
    vbdd model_support;
    bdd * fn_support, * v;
    model_support = vbdd_new();
    if (specification_support) {
      for (v = specification_support; * v; ++v) {
        vbdd_add(bddm, model_support, * v);
      }
    } else {
      bdd * ptr;
      for (ptr = vbdd_elements(buffer); * ptr; ptr += 2) {
        ++after;
        vbdd_add(bddm, model_support, * ptr);
      }
    }
    fn_support = bdd_new_support(bddm);
    for (i = 0; i < vbdd_length(model_support); ++i) {
      bdd v = vbdd_nth(model_support, i), 
          f, * ptr;
      ptr = vbdd_elements(buffer);
      while (* ptr != v) {
        ptr+=2;
      }
      f = *(ptr+1);
      if (f) {
        vbdd_add(bddm, registers, v);
        vbdd_add(bddm, functions, f);
        if (specification_support) {
          bdd_support(bddm, f, fn_support);
        }
      } else {
        * fn_support = 0;
      }
      for (ptr = fn_support; * ptr; ++ptr) {
        bdd *find;
        find = vbdd_elements(model_support);
        while ((* find) && ((* find) != (* ptr))) ++find;
        if ((* find) == 0) {
          find = vbdd_elements(buffer);
          while ((* find) && ((* find) != (* ptr))) find += 2;
          if ((* find) != 0) {
            ++after;
            vbdd_add(bddm, model_support, *ptr);
          }
        }
      }
    }
    if (specification_support) {
      bdd_free_support(bddm, fn_support);
    }
    vbdd_free(bddm, model_support);
  }
  vbdd_free(bddm, buffer);
  m->img = img_New(bddm, registers, functions);
  m->ant = ant_New(bddm, registers, functions);
  return (before - after);
}

/** Function **
  Synopsis    [ Sets the set of fairness constraints ]
  Description [ The given set of CTL terms are taken as fairness
  constraints and stored into the model. This routine additionnally
  computes the value of each of these constraints as well as the
  set of fair states. These computations are traced onto the given
  output stream. ]
  SideEffects [ Modifies the state of the model representation.
  Output is sent to outstream. ]
 */
void
model_RepresentationSetFair
(Model m,
 FILE * outstream,
 cvn constraints)
{
  ModelCheckComputation fair_states_computation;
  m->fairness_constraints = constraints;
  fprintf(stdout, "evaluating fair states\n");
  fair_states_computation = model_CheckEG(m, outstream, bdd_one(m->bddm));
  m->fair_states = model_CheckQSolution(fair_states_computation);
  model_CheckFreeComputation(fair_states_computation);
}

/** Function **
  Synopsis    [ Sets the set of invariants ]
  Description [ The given BDD represents a function of the
  input variables of the model. This is the characteristic
  function of all possible input vectors at any given time.
  This BDD is stored into the model and affects subsequent 
  state space traversal. ]
  SideEffects [ Modifies the state of the model representation. ]
 */
void
model_RepresentationSetInvariant
(Model m,
 bdd invariant)
{
  bdd_free(m->bddm, m->invariant);
  m->invariant = bdd_identity(m->bddm, invariant);
  do {
    bdd tmp;
    tmp = bdd_and(m->bddm, invariant, m->init);
    if (bdd_overflow(m->bddm)) {
      bdd_resize_manager(m->bddm);
    } else {
      bdd_free(m->bddm, m->init);
      m->init = tmp;
      break;
    }
  } while(1);
  if (m->img) {
    img_ReduceCareSet(m->img, invariant);
  }
  if (m->ant) {
    ant_ReduceCareSet(m->ant, invariant);
  }
}

/** Function **
  Synopsis    [ Sets the set of invariants ]
  Description [ The given BDD represents a function of the
  input variables of the model. This is the characteristic
  function of all possible input vectors at any given time.
  This BDD is stored into the model and affects subsequent 
  state space traversal. ]
  SideEffects [ Modifies the state of the model representation. ]
 */
void
model_RepresentationAddInvariant
(Model m,
 bdd invariant)
{
  do {
    bdd tmp_inv, tmp_ini;
    tmp_inv = bdd_and(m->bddm, invariant, m->invariant);
    tmp_ini = bdd_and(m->bddm, invariant, m->init);
    if (bdd_overflow(m->bddm)) {
      bdd_resize_manager(m->bddm);
    } else {
      bdd_free(m->bddm, m->invariant);
      m->invariant = tmp_inv;
      bdd_free(m->bddm, m->init);
      m->init = tmp_ini;
      break;
    }
  } while(1);
  if (m->img) {
    img_ReduceCareSet(m->img, invariant);
  }
  if (m->ant) {
    ant_ReduceCareSet(m->ant, invariant);
  }
}

/** Function **
  Synopsis           [required]
  Description        [optional]
  SideEffects        [required]
  SeeAlso            [optional]
 */
void
model_RepresentationFlush
(Model m)
{
  ant_Flush(m->ant);
  img_Flush(m->img);
}

/*--------------------------------------------------------------------*/
/* Definition of internal functions                                   */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Definition of static functions                                     */
/*--------------------------------------------------------------------*/

/** Function **
  Synopsis    [ Adds a condition to the initial state condition ]
  Description [ Stores the condition i with the initial condition
  of the model m to the field init of m. ]
  SideEffects [ Modifies the field init and the state of the 
  BDD manager of m. ]
  SeeAlso     [ ]
 */
static bdd 
_RepresentationAddInit (
  Model m, 
  bdd i)
{
  do {
    bdd tmp;
    tmp = bdd_and(m->bddm, i, m->init);
    if (bdd_overflow(m->bddm)) {
      bdd_resize_manager(m->bddm);
    } else {
      bdd_free(m->bddm, m->init);
      m->init = tmp;
      break;
    }
  } while (1);
  return m->init;
}

/** Function **
  Synopsis    [ Gets the index of a signal ]
  Description [ If the signal declaration decl is present in
  the field signals of m, returns its position in this array,
  otherwise returns -1. ]
  SideEffects [ None ]
  SeeAlso     [ ]
 */
static int
_RepresentationQSignal (
  Model m,
  cvn decl)
{
  ModelSignalStruct_t * i;
  int idx = 0, sup;
  i= m->signals;
  sup = m->nb_signals;
  while ((idx < sup) && (!cv_eq(i->declaration, decl))) {++i; ++idx;}
  if (idx == sup) idx= -1;
  return idx;
}

/** Function **
  Synopsis    [ Gets the index of a variable ]
  Description [ If the variable declaration decl is present in
  the field variables of m, returns its position in this array,
  otherwise returns -1. ]
  SideEffects [ None ]
  SeeAlso     [ ]
 */
static int
_RepresentationQVariable (
  Model m,
  cvn decl)
{
  ModelVariableStruct_t * i;
  int idx = 0, sup;
  i= m->variables;
  sup = m->nb_variables;
  while ((idx < sup) && (!cv_eq(i->declaration, decl))) {++i; ++idx;}
  if (idx == sup) idx= -1;
  return idx;
}

