/** CFile **************************************************************

  FileName    [ modelReachability.c ]

  PackageName [ model ]

  Synopsis    [ Computation and manipulation of states sets ]

  Description [ This file contains a routine to compute the set
  of reachable states of a model (model_ReachabilityCompute), and
  a set of routines to get the important subsets of reachable 
  states, once these have been computed of course, namely the stable
  states (model_ReachabilityQStable), the intermediate states
  (model_ReachabilityQIntermediate) and the reachable states
  themselves (model_ReachabilityQAll). ]

  SeeAlso     [ ]

  Author      [ David Deharbe ]

  Copyright   [ Copyright (C) 1996, Carnegie Mellon University.
                All rights reserved. ]

  Revision    [ $Id$ ]

***********************************************************************/

#include "modelInt.h"

static char rcsid [] = "$Id$";

/*--------------------------------------------------------------------*/
/* Structure declarations                                             */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Type declarations                                                  */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Variable declarations                                              */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Macro declarations                                                 */
/*--------------------------------------------------------------------*/

/**AutomaticStart******************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static void _print_double_int(FILE * stream, double l);

/**AutomaticEnd********************************************************/

/*--------------------------------------------------------------------*/
/* Definition of exported functions                                   */
/*--------------------------------------------------------------------*/

/** Function **
  Synopsis    [ Set image computation ]
  Description [ Given the characteristic function set of a set of 
  states of m, returns the characterisic function of the set of the
  images of the states of this set by the transition of m. ]
  SideEffects [ Modifies the state of the BDD manager. ]
  SeeAlso     [ model_ReachabilityStepBackward ]
 */
bdd 
model_ReachabilityStepForward
(Model m,
 bdd set)
{
  bdd_manager bddm = m->bddm ;
  bdd result;
  bdd valid;
  do {
    bdd tmp = bdd_and(bddm, set, m->invariant);
    if (bdd_overflow(bddm)) {
      bdd_resize_manager(bddm);
    } else {
      set = tmp;
      break;
    } 
  } while (1);
  result= img_Compute(m->img, set);
  do {
    bdd tmp;
    if ((valid = model_ReachabilityQAll(m)) != 0) {
      tmp = bdd_and(bddm, valid, result);
    } else {
      tmp = bdd_identity(bddm, result);
    }
    tmp = bdd_andup2(bddm, m->invariant, tmp);
    if (bdd_overflow(bddm)) {
      bdd_resize_manager(bddm);
    } else {
      bdd_free(bddm, result);
      result = tmp;
      break;
    }
  } while(1);
  return result;
}

/** Function **
  Synopsis    [ Set preimage computation ]
  Description [ Given the characteristic function set of a set of 
  states of m, returns the characterisic function of the set of the
  preimages of the states of this set by the transition of m. ]
  SideEffects [ Modifies the state of the BDD manager. ]
  SeeAlso     [ model_ReachabilityStepForward ]
 */
bdd 
model_ReachabilityStepBackward
(Model m,
 bdd set)
{
  bdd_manager bddm = m->bddm ;
  bdd result, valid;
  set = bdd_and(bddm, m->invariant, set);
  result= ant_Compute(m->ant, set);
  if ((valid = model_ReachabilityQAll(m)) != 0) {
    result = bdd_andup2(bddm, valid, result);
  }
  result = bdd_andup2(bddm, m->invariant, result);
  return result;
}

/** Function **
  Synopsis    [ Frontiers sets of the reachability analysis ]
  Description [ This routine checks if the reachable states have
  already been computed. If yes, it returns the frontiers set
  vector of BDDs, otherwise, it returns 0. 
  The ith element of the frontiers set vector contains the
  characteristic function of the states that are visited for
  the first time in the reachability analysis at the (i+1)th
  step. ]
  SideEffects [ none ]
  SeeAlso     [ ]
 */
vbdd
model_ReachabilityQFrontiers
(Model m)
{
  if (m -> reachability_analysis.done) 
    return m->reachability_analysis.frontiers;
  return 0;
}

/** Function **
  Synopsis    [ Returns the set of reachable states  ]
  Description [ This routine checks if the reachable states have
                already been computed. If yes, it returns them,
                otherwise, it returns 0. ]
  SideEffects [ none ]
  SeeAlso     [ model_ReachabilityQStable 
                model_ReachabilityQIntermediate ]
 */
bdd
model_ReachabilityQAll
(Model m)
{
  if (m -> reachability_analysis.done) 
    return m->reachability_analysis.reachable;
  return 0;
}

/** Function **
  Synopsis    [ Returns the set of stable states  ]
  Description [ This routine checks if the reachable states have
                already been computed. If yes, it returns the
                stable states, otherwise, it returns 0. ]
  SideEffects [ none ]
  SeeAlso     [ model_ReachabilityQAll 
                model_ReachabilityQIntermediate ]
 */
bdd
model_ReachabilityQStable
(Model m)
{
  if (m -> reachability_analysis.done) 
    return m->reachability_analysis.stable;
  return 0;
}

/** Function **
  Synopsis    [ Returns the set of intermediate states  ]
  Description [ This routine checks if the reachable states have
                already been computed. If yes, it returns the
                intermediate states, otherwise, it returns 0. ]
  SideEffects [ none ]
  SeeAlso     [ model_ReachabilityQAll 
                model_ReachabilityQStable ]
 */
bdd
model_ReachabilityQIntermediate
(Model m)
{
  if (m -> reachability_analysis.done) 
    return m->reachability_analysis.intermediate;
  return 0;
}

/** Function **
  Synopsis    [ Computes and returns the reachable states ]
  Description [ Performs the symbolic computation of the set of
  reachable states. This routine must be called after the 
  routine model_RepresentationBuildTransition has been invoked
  to build the transition representation in the model.
  A trace is printed to the given output stream, as well as the
  total number of states and the number of stable states 
  eventually reached. ]
  SideEffects [ Modifies the state of the BDD manager
  associated to the model. Allocates memory to store
  intermediate results (the frontiers set). Sets the field 
  reachability_analysis of the model. Sends output to
  the stream outstream. ]
  SeeAlso     [ model_RepresentationBuildTransition ]
 */

bdd
model_ReachabilityCompute
(Model m,
 FILE * outstream)
{
  bdd_manager bddm = m->bddm;
  bdd reached, new, from;
  vbdd frontiers;
  int step;

  if (m -> reachability_analysis.done) 
    return m -> reachability_analysis.reachable ;

  fprintf(outstream, "The model has ");
  _print_double_int(outstream, model_MeasureNbStates(m));
  fprintf(outstream, " potential states (%u variables).\n",
          m->nb_propositions);

  frontiers = vbdd_new() ;
  new = bdd_identity(bddm, m->init);
  reached = bdd_identity(bddm, m->init);
  from = bdd_identity(bddm, m->init);
  step = 0;
  while (new != bdd_zero(bddm)) {
    bdd next ;
    fprintf(outstream, "[%u", ++step); fflush(outstream);
    vbdd_add(bddm, frontiers, new) ;
    bdd_free(bddm, new) ;
    next = model_ReachabilityStepForward(m, from) ;
    bdd_free(bddm, from) ;
    do {
      bdd next_reached, care;
      new = bdd_andup2(bddm, next, bdd_not(bddm, reached)) ;
      next_reached = bdd_or(bddm, new, reached) ;
      care = bdd_orup2(bddm, new, bdd_not(bddm, next_reached));
      from = bdd_reduce(bddm, new, care);
      bdd_free(bddm, care);
      if (bdd_overflow(bddm)) {
        bdd_free(bddm, new);
        bdd_free(bddm, from);
        bdd_free(bddm, next_reached);
        bdd_resize_manager(bddm);
      } else {
        bdd_free(bddm, reached);
        bdd_free(bddm, next);
        reached = next_reached;
        break;
      }
    } while (1);
#if TRACE_REACHABILITY
    fprintf(outstream, " %lu nodes in use\n", bdd_total_size(m->bddm));
    _print_double_int(outstream, model_MeasureSetCardinality(m, new));
    fprintf(outstream, " new configurations reached.\n");
#endif
    fprintf(outstream, "] "); fflush(outstream);
  }
  fprintf(outstream, ")\n"); fflush(outstream);
  m -> reachability_analysis.done = 1;
  m -> reachability_analysis.reachable = reached ;
  m -> reachability_analysis.stable = bdd_and(bddm, reached, m->stable);
  m -> reachability_analysis.intermediate = 
    bdd_andup2(bddm, reached, bdd_not(bddm, m->stable));
  m -> reachability_analysis.frontiers = frontiers ;
  _print_double_int(outstream, model_MeasureNbReachableStates(m));
  fprintf(outstream, " states are reachable and ");
  _print_double_int(outstream, model_MeasureNbStableStates(m));
  fprintf(outstream, " are stable.\n"); 
  fprintf(outstream, "%lu nodes represent the valid states.\n",
          bdd_size(m->bddm, reached, 1));
  model_RepresentationFlush(m);
  bdd_gc(m->bddm);
  return reached;
}

/*--------------------------------------------------------------------*/
/* Definition of internal functions                                   */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Definition of static functions                                     */
/*--------------------------------------------------------------------*/

/** Function **
  Synopsis    [ Displays a double int in a user-friendly format ]
  Description [ If the given double is small (smaller than 10000)
  it is written as a whole, other it is written using an
  mantissa and a power of 10. ]
  SideEffects [ Sends output to the stream ]
  SeeAlso     [ ]
 */
static void _print_double_int
(FILE * stream, 
 double l)
{
  if (l >= 10000) {
    fprintf(stream, "%.3E", l);
  } else {
    fprintf(stream, "%.0f", l);
  }
}
