/** CFile **************************************************************

  FileName    [ label.c ]

  PackageName [ label ]

  Synopsis    [ Implementation of the package label ]

  Author      [ David Deharbe ]

  Copyright   [ Copyright (C) 1996, Carnegie Mellon University.
                All rights reserved. ]

  Revision    [ $Id$ ]

***********************************************************************/

#include "labelInt.h"

#include <memuser.h>

static char rcsid [] = "$Id$";

/*--------------------------------------------------------------------*/
/* Structure declarations                                             */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Type declarations                                                  */
/*--------------------------------------------------------------------*/
struct labelManagerStruct_t_ {
  int counter;
  rec_mgr memory;
  char * prefix;
};

/*--------------------------------------------------------------------*/
/* Variable declarations                                              */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Macro declarations                                                 */
/*--------------------------------------------------------------------*/

/**AutomaticStart******************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/


/**AutomaticEnd********************************************************/

/*--------------------------------------------------------------------*/
/* Definition of exported functions                                   */
/*--------------------------------------------------------------------*/

/** Function **
  Synopsis    [ Creation of a label manager ]
  Description [ The string prefix shall not end with a digit. ]
  SideEffects [ Allocates memory via mem(3) ]
 */
label_manager
label_NewManager
(char * prefix)
{
  label_manager mgr;
  mgr = (label_manager) mem_get_block(sizeof(struct labelManagerStruct_t_));
  mgr->counter = 0;
  mgr->prefix = (char *) strdup(prefix);
  mgr->memory = mem_new_rec_mgr((SIZE_T) strlen(prefix) * 6);
  return mgr;
}

/** Function **
  Synopsis    [ Destroy a label manager ]
  Description [ All the memory used to represent the labels generated
  with mgr is disallocated. Do not refer to these labels after calling
  this routine. ]
  SideEffects [ Disallocates memory used to represent the manager and
  the associated label. ]
 */
void
label_FreeManager
(label_manager mgr)
{
  mem_free_rec_mgr(mgr->memory);
  mem_free_block((pointer) mgr);
}

/** Function **
  Synopsis    [ Creates a string composed of a prefix and a number ]
  Description [ Creates a hopefully unique string that consists of
  the catenation of the character argument and the current value of
  the label counter. The result must be freed with routine 
  label_FreeLabel. ]
  SideEffects [ The state of mgr is modified: memory is allocated 
  via the record manager mgr->memory and mgr->counter is incremented. ]
  SeeAlso     [ _LabelFree _labelCounter _labelManager ]
 */
char * /* result label */
label_NewLabel 
(label_manager mgr)
{
  char * label;
  label = (char *) mem_new_rec(mgr->memory);
  sprintf(label,"%s%i", mgr->prefix, mgr->counter++);
  return label;
}

/** Function **
  Synopsis    [ Frees the memory allocated to represent a label ]
  Description [ Frees the memory allocated to represent a label 
                generated by the routine _LabelNew. ]
  SideEffects [ _labelManager disallocates memory. ]
  SeeAlso     [ _LabelFree _labelCounter _labelManager ]
 */
void label_FreeLabel
(label_manager mgr,
 char * label)
{
  mem_free_rec(mgr->memory, (pointer) label);
}

/*--------------------------------------------------------------------*/
/* Definition of internal functions                                   */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Definition of static functions                                     */
/*--------------------------------------------------------------------*/

