/** CFile **************************************************************

  FileName    [ imgRangeVector.c ]

  PackageName [ img ]

  Synopsis    [ Transition vectors used in range computation ]

  Description [ A vector contains a list of couples of BDDs 
  representing a variable and its transition function. This list
  is in increasing order with respect to the address of this
  transition function, and contains no duplicates (two BDDs are
  duplicates if they are equal or the negation of each other). 
  Moreover, a vector maintains an internal BDD that records
  simplifications that were during its construction. ]

  SeeAlso     [ imgImage.c ]

  Author      [ David Deharbe ]

  Copyright   [ Copyright (C) 1996, Carnegie Mellon University.
                All rights reserved. ]

  Revision    [ 3.1 ]

***********************************************************************/

#include "imgInt.h"

/*--------------------------------------------------------------------*/
/* Structure declarations                                             */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Type declarations                                                  */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Variable declarations                                              */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Macro declarations                                                 */
/*--------------------------------------------------------------------*/

/**AutomaticStart******************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/


/**AutomaticEnd********************************************************/

/*--------------------------------------------------------------------*/
/* Definition of exported functions                                   */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Definition of internal functions                                   */
/*--------------------------------------------------------------------*/

/** Function **
  Synopsis    [ Vector allocation ]
  Description [ Creates an initially empty vector. ]
  SideEffects [ Memory is allocated with a record manager ]
  SeeAlso     [ ]
 */
RangeVector
imgRangeVectorNew
(img_manager t)
{
  RangeVector result;
  result = (RangeVector) mem_new_rec(t->RangeVectorMgr);
  result->contents = 0;
  result->info = bdd_one(t->bddm);
  return result;
}

/** Function **
  Synopsis    [ Vector deallocation ]
  Description [ Creates an initially empty vector. ]
  SideEffects [ Memory is allocated with a record manager ]
  SeeAlso     [ ]
 */
void
imgRangeVectorFree
(img_manager t,
 RangeVector vec)
{
  mem_free_rec(t->RangeVectorMgr, (pointer) vec);
}

/** Function **
  Synopsis    [ Vector constructor ]
  Description [ By hypothesis, the contents of c are such that
  there is no two couples variable, function with the same 
  function or opposite function. 
  If f is equal or the negation of a function already in c, 
  then c->info is modified to (c->info and (v x[n]or v') where
  v' is the variable associated with this function. Otherwise
  the couple v, f is added to c. ]
  SideEffects [ Allocates memory via a record manager or
  performs an operation on BDDs. While overflows occur, the
  function bdd_resize_manager is invoked. ]
 */
RangeVector
imgRangeVectorAdd
(img_manager t,
 RangeVector vec,
 bdd v,
 bdd f)
{
  bdd_manager bddm = t->bddm;
  /* if f is a 1 (resp. 0), add v (resp. not v) to c->info */
  if ((f == bdd_zero(bddm)) || (f == bdd_one(bddm))) {
    do {
      bdd var = (f == bdd_one(bddm)) ? v : bdd_not(bddm, v);
      bdd tmp = bdd_and(bddm, var, vec->info);
      if (bdd_overflow(bddm)) {
        bdd_resize_manager(bddm);
      } else {
        bdd_free(bddm, vec->info);
        vec->info = tmp;
        break;
      }
    } while(1);
  /* f is not a constant, see if c->contents contains a couple 
     <v', f'> such that f' = f (resp. f' = not f). If yes, add
     v = v' (resp v = not v') to c->range. If not insert the
     couple <v, f>. */
  } else {
    Atom prev, ptr;
    bdd f_addr = bdd_address(bddm, f);
    for (ptr = (Atom) vec->contents, prev = 0; 
         ptr && (bdd_address(bddm, ptr->f) < f_addr); 
         prev = ptr, ptr = ptr->link) {
    }
    if ((ptr == 0) || (bdd_address(bddm, ptr->f) != f_addr)) {
      Atom rec = (Atom) mem_new_rec(t->AtomMgr);
      rec->v = v;
      rec->f = f;
      rec->link = ptr;
      if (prev == 0) {
        vec->contents = rec;
      } else {
        prev->link = rec;
      }
    } else {
      do {
        bdd tmp = bdd_andup2(bddm,
                             vec->info,
                             (f == ptr->f) ? 
                              bdd_xnor(bddm, v, ptr->v):
                              bdd_xor(bddm, v, ptr->v));
        if (bdd_overflow(bddm)) {
          bdd_resize_manager(bddm);
        } else {
          bdd_free(bddm, vec->info);
          bdd_free(bddm, f);
          vec->info = tmp;
          break;
        }
      } while (1);
    }
  }
  return vec;
}

/** Function **
  Synopsis    [ Vector allocation ]
  Description [ Creates an initially empty vector. ]
  SideEffects [ Memory is allocated with a record manager ]
  SeeAlso     [ ]
 */
bdd
imgRangeVectorReachable
(img_manager t,
 RangeVector vec,
 bdd init)
{
  bdd_manager bddm = t->bddm;
  bdd new, reached, from;
  int count;
  reached = bdd_identity(bddm, init);
  from = bdd_identity(bddm, init);
  count = 0;
  setbuf(stdout, 0);
  do {
    bdd next;
    fprintf(stdout, "[%i", ++count);
    next = RangeVectorImage(t, vec, from);
    bdd_free(bddm, from);
    do {
      bdd tmp;
      new = bdd_andup2(bddm, next, bdd_not(bddm, reached));
      tmp = bdd_or(bddm, next, reached);
      if (bdd_overflow(bddm)) {
        bdd_free(bddm, new);
        bdd_free(bddm, tmp);
        bdd_resize_manager(bddm);
      } else {
        bdd_free(bddm, reached);
        reached = tmp;
        break;
      }
    } while(1);
    bdd_free(bddm, next);
    from = new;
    fprintf(stdout, "] ");
  } while (new != bdd_zero(bddm));
  fprintf(stdout, "\n");
  return reached;
}

/*--------------------------------------------------------------------*/
/* Definition of static functions                                     */
/*--------------------------------------------------------------------*/

/** Function **
  Synopsis    [ Vector allocation ]
  Description [ Creates an initially empty vector. ]
  SideEffects [ Memory is allocated with a record manager ]
  SeeAlso     [ ]
 */
bdd
RangeVectorImage
(img_manager t,
 RangeVector v,
 bdd from)
{
  bdd_manager bddm = t->bddm;
  bdd result;
  RangeVector reduced;
  Atom a;
  reduced = imgRangeVectorNew(t);
  for (a = v->contents; a; a = a->link) {
    imgRangeVectorAdd(t, reduced, a->v, imgRangeRestrict(bddm, a->f, from));
  }
  result = imgRange(t, reduced);
  return result;
}
