/** CFile **************************************************************

  FileName    [ imgRangeClass.c ]

  PackageName [ img ]

  Synopsis    [ Partition classes used in range computation ]

  Description [ A class contains a list of couples of BDDs 
  representing a variable and its transition function. This list
  is in increasing order with respect to the address of this
  transition function, and contains no duplicates (two BDDs are
  duplicates if they are equal or the negation of each other). ]

  SeeAlso     [ imgImage.c ]

  Author      [ David Deharbe ]

  Copyright   [ Copyright (C) 1996, Carnegie Mellon University.
                All rights reserved. ]

  Revision    [ 3.1 ]

***********************************************************************/

#include "imgInt.h"

/*--------------------------------------------------------------------*/
/* Structure declarations                                             */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Type declarations                                                  */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Variable declarations                                              */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Macro declarations                                                 */
/*--------------------------------------------------------------------*/

/**AutomaticStart******************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/


/**AutomaticEnd********************************************************/

/*--------------------------------------------------------------------*/
/* Definition of exported functions                                   */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Definition of internal functions                                   */
/*--------------------------------------------------------------------*/

/** Function **
  Synopsis    [ Class hash function ]
  Description [ The hash code is equal to the sum of the bdds of the
  functions in the range class, modulo 4. ]
  SideEffects [ None ]
 */
long
imgRangeClassHash
(RangeClass c,
 img_manager t)
{
  return (c->hash >> 2);
}

/** Function **
  Synopsis    [ Class equality function ]
  Description [ c1 and c2 are equal, if there is a bijective mapping m
  between their lists of couplese v, f such that if m(v, f) = v', f'
  then f = f' or f = not f'. ]
  SideEffects [ Builds a BDD association on the fly at the location
  t->support. This variable association can be later used as a 
  substitution. ]
  SeeAlso     [ range_cache ]
 */
int
imgRangeClassEq
(RangeClass c1,
 RangeClass c2, 
 img_manager t)
{
  Atom p1, p2;
  bdd_manager bddm = t->bddm;
  bdd * p = t->support;

  p1 = (Atom) c1->contents;
  p2 = (Atom) c2->contents;
  while (p1 && p2 && (bdd_address(bddm, p1->f) == bdd_address(bddm, p2->f))) {
    if (p1->f == p2->f) {
       if (p1->v != p2->v) {
         * p++ = p1->v;
         * p++ = p2->v;
       }
    } else {
       * p++ = p1->v;
       * p++ = bdd_not(bddm, p2->v);
    } 
    p1 = p1->link; p2 = p2->link;
  }
  * p = 0;
  return ((p1 == 0) && (p2 == 0));
}

/** Function **
  Synopsis    [ class allocate function ]
  SideEffects [ Allocates memory ]
 */
RangeClass
imgRangeClassNew
(img_manager t)
{
  RangeClass result;
  result = (RangeClass) mem_new_rec(t->RangeClassMgr);
  result->prelink = result->link = 0;
  result->contents = 0;
  result->hash = 0;
  result->range = 0;
  return result;
}

/** Function **
  Synopsis    [ class deallocate function ]
  SideEffects [ Decrements the reference counter of c->range,
  decrements the reference counter of each function in c->contents,
  deallocates c->contents, deallocates c. ]
 */
void
imgRangeClassFree
(RangeClass c,
 img_manager t)
{
  Atom next, p;
  rec_mgr AtomMgr = t->AtomMgr;
  bdd_manager bddm = t->bddm;
  bdd_free(bddm, c->range);
  if ((p = c->contents) != 0) {
    do {
      next = p->link;
      bdd_free(bddm, p->f);
      mem_free_rec(AtomMgr, p);
      p = next;
    } while (p);
  }
  mem_free_rec(t->RangeClassMgr, c);
}

/** Function **
  Synopsis    [ class constructor ]
  Description [ Adds a couple variable and function to the
  c->contents, such that c->contents remains in increasing
  order. c->contents shall not contain a couple variable
  function such that function and f have the same address. ]
  SideEffects [ Allocates memory. The reference counters
  of v and f are NOT changed ]
 */
RangeClass
imgRangeClassAdd
(img_manager t,
 RangeClass c,
 bdd v,
 bdd f)
{
  bdd_manager bddm = t->bddm;
  Atom prev, ptr, rec;
  bdd f_addr = bdd_address(bddm, f);
  for (ptr = c->contents, prev = 0; 
       ptr && (bdd_address(bddm, ptr->f) < f_addr); 
       prev = ptr, ptr = ptr->link) {
  }
  rec = (Atom) mem_new_rec(t->AtomMgr);
  rec->v = v;
  rec->f = f;
  rec->link = ptr;
  c->hash += (long) f;
  if (prev == 0) {
    c->contents = rec;
  } else {
    prev->link = rec;
  }
  return c;
}

/** Function **
  Synopsis    [ Merges two range classes ]
  Description [ Hypothesis: c1->contents and c2->contents are
  in strict increasing order on f, and are disjoint. ]
  SideEffects [ This routine is destructive for the parameters
  c1 and c2: the result is stored at the same location as c1,
  and the memory used for c2 is deallocated. ]
 */
RangeClass
imgRangeClassMerge
(img_manager t,
 RangeClass c1,
 RangeClass c2)
{
  Atom ptr1, ptr2, prev = 0;
  bdd a1, a2;

  if (c2 == 0) return c1;
  
  /* merge contents */
  ptr1 = c1->contents, 
  ptr2 = c2->contents,
  a1 = ptr1 ? ptr1->f : 0;
  a2 = ptr2 ? ptr2->f : 0;
  while (ptr1 && ptr2) {
    if (a1 < a2) {
      prev = ptr1;
      ptr1 = ptr1->link;
      if (ptr1) {
        a1 = ptr1->f;
      } else {
        prev->link = ptr2;
        a1 = 0;
      }
    } else {
      Atom tmp;
      tmp = ptr2->link;
      ptr2->link = ptr1;
      if (prev == 0) {
        c1->contents = ptr2;
      } else {
        prev->link = ptr2;
      }
      prev = ptr2;
      c1->hash += (long) ptr2->f;
      ptr2 = tmp;
      a2 = ptr2 ? ptr2->f : 0;
    }
  }
  mem_free_rec(t->RangeClassMgr, c2);
  return c1;
}

/*--------------------------------------------------------------------*/
/* Definition of static functions                                     */
/*--------------------------------------------------------------------*/

