/** CFile **************************************************************

  FileName    [ subtype.c ]

  PackageName [ cv ]

  Synopsis    [ Routines on VHDL scalar types ]

  Description [ This file contains routines that perform small 
                operations on the CV representation of VHDL scalar
                types. ]

  SeeAlso     [ Optional ]

  Author      [ David Deharbe ]

  Copyright   [ Copyright (C) 1996, Carnegie Mellon University.
                All rights reserved. ]

  Revision    [ $Id$ ]

***********************************************************************/

#include <Errors.h>
#include "cvInt.h"

static char rcsid [] = "$Id$";

/*--------------------------------------------------------------------*/
/* Structure declarations                                             */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Type declarations                                                  */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Variable declarations                                              */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Macro declarations                                                 */
/*--------------------------------------------------------------------*/

/**AutomaticStart******************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/


/**AutomaticEnd********************************************************/

/*--------------------------------------------------------------------*/
/* Definition of exported functions                                   */
/*--------------------------------------------------------------------*/

/** Function **
  Synopsis    [ Computes the size of a scalar type ]
  Description [ The parameter must be a CV node of the abstract class
  TypeDefinition. Moreover, the type defined must be a scalar type, as 
  defined by the Language Reference Manual. The routine returns the 
  size of the type.
  If an error occurs, a message is printed to the output, using the 
  Message routine of the reuse library and the value -1 is returned. ]
  SideEffects [ If an error occurs, sends output to stdout ]
  SeeAlso     [ cvu_ScalarTypeComputeNthValue ]
 */

int
cv_subtype_size
(const cvn subtype)
{
  cvn basetype = qBaseType(subtype);
  cvn range = qConstraint(subtype);
  int size ;

  if (!(IsA(basetype, kIntegerType) || IsA(basetype, kEnumerationType))) {
    Message("(libcv.a) cv_subtype_size: parameter type class not accepted", 
            xxError, NoPosition);
    return -1;
  }

  if ((range == 0) && IsA(basetype, kIntegerType)) {
    range = qRange(basetype);
  }
  if (IsA(basetype, kIntegerType)) {
    size = abs(qIntegerValue(qLeft(range)) - qIntegerValue(qRight(range))) + 1 ; 
  } else /* if (IsA(basetype, kEnumerationType)) */ {
    cvn ptr ;
    size = 0 ;
    if (range && (qDirection(range) == kTo)) {
      for (ptr = qElements(basetype); ptr!=qLeft(range); ptr = qNext(ptr));
      ++size;
      for (; ptr!=qRight(range); ptr = qNext(ptr), ++size);
    } else {
      for (ptr = qElements(basetype); ptr ; ptr = qNext(ptr), ++size);
    }
  }
  return size;
}

/** Function **
  Synopsis    [ Computes the nth value of a scalar type ]
  Description [ The first parameter must be a CV node of the abstract
  class TypeDefinition. Moreover, the type defined must be a scalar 
  type, as defined by the Language Reference Manual. The second
  parameter is an integer n comprised between 0 and the size of the
  given type minus 1 (see cvu_ScalarTypeComputeSize).
  The routine returns the node representing the nth value of the
  type. If the type is an enumeration type, the node returned is
  the corresponding node of the list under the qElements child of
  the type, if the type is an integer type, a new node is created.
  If an error occurs, a message is printed to the output, using the 
  Message routine of the reuse library and the value 0 is returned. ]
  SideEffects [ If an error occurs, sends output to stdout.
  If type is an integer type, memory is allocated to store a new 
  node that is returned as the result. ]
  SeeAlso     [ cvu_ScalarTypeComputeSize ]
 */
cvn
cv_subtype_nth
(const cvn subtype,
 int ordinal)
{
  cvn basetype = qBaseType(subtype);
  cvn range = qConstraint(subtype);

  if (!(IsA(basetype, kIntegerType) || IsA(basetype, kEnumerationType))) {
    Message("(libcv.a) cv_subtype_nth: parameter type class not accepted", 
            xxError, NoPosition);
    return 0;
  }
  if ((ordinal < 0) || (ordinal >= cv_subtype_size(subtype))) {
    Message("(libcv.a) cv_subtype_nth: given value has no corresponding type element", 
            xxError, NoPosition);
    return 0;
  }
  if ((range == 0) && IsA(basetype, kIntegerType)) {
    range = qRange(basetype);
  }
  switch (qKind(basetype)) {
  case kEnumerationType: {
    cvn ptr;
    ptr = qElements(basetype);
    if (range) {
      while (ptr != qLeft(range)) {
        ptr = qNext(ptr);
      }
    }
    while (ordinal--) {
      ptr = qNext(ptr);
    }
    return qValue(ptr);
  }
    break;
  case kIntegerType:
    return mIntegerValue(0, subtype, ordinal + qIntegerValue(qLeft(range)));
    break;
  default:
    Message("(libcv.a) cv_subtype_nth: inconsistency occured - please report", 
            xxError, NoPosition);
    return 0;
    break;
  }
}

/*--------------------------------------------------------------------*/
/* Definition of internal functions                                   */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Definition of static functions                                     */
/*--------------------------------------------------------------------*/

