/** CFile **************************************************************

  FileName    [ cache1.c ]

  PackageName [ cache ]

  Synopsis    [ Generic implementation of a 1-entry per line, 
  one-way cache. ]

  Description [ Definition of the basic routines providing a 
  generic implementation of a one-entry per line one-way cache.  ]

  SeeAlso     [ cache.h cacheInt.h ]

  Author      [ David Dharbe, K. McMillan and David Long ]

  Copyright   [ Copyright (C) 1996, Carnegie Mellon University.
                All rights reserved. ]

  Revision    [ $Id$ ]

***********************************************************************/

#include <string.h>

#include "cacheInt.h"

static char rcsid [] = "$Id$";

/*--------------------------------------------------------------------*/
/* Structure declarations                                             */
/*--------------------------------------------------------------------*/

/* cache data structure */
struct cache1_ {
  long size;          /* size of the cache table */
  long entries;          /* number of entries currently in the hash table */
  hash_function hash; /* hashing function */
  eq_function equal;   /* equality predicate */
  del_function delete;   /* equality predicate */
  pointer environment;       /* parameter for predicate and function */
  float threshold;      /* the rehash threshold */
  long rehash_limit;    /* maximum number of entries allowed before rehash */
  float resize_coef;    /* the resize factor */
  int size_index;  
  pointer *tab;    /* the cache table */
  long resize;     /* number of times the cache has been resized */
  long find;       /* find calls */
  long insert;     /* insert calls */
  long hit;        /* successful find operations */
  long collision;  /* colliding inserts operations */
};

/*--------------------------------------------------------------------*/
/* Type declarations                                                  */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Variable declarations                                              */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Macro declarations                                                 */
/*--------------------------------------------------------------------*/

/**AutomaticStart******************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static void _Resize(cache1_mgr cache);

/**AutomaticEnd********************************************************/

/*--------------------------------------------------------------------*/
/* Definition of exported functions                                   */
/*--------------------------------------------------------------------*/

/** Function **
  Synopsis    [ Provides a new cache manager ]
  Description [ Creates and returns a manager to cache generic data. 
  The parameters are the size of the cache table, the hashing function, 
  the equality predicate, the memory manager for the items stored in 
  the table, an integer flag that indicates if the size of the table
  is fixed or not and an integer flag that indicates the number of
  entries per line in the cache (the value of this parameter is not
  significant if the size of the cache is not fixed).
  The actual size of the table is a prime number greater or equal to
  the size parameter specified by the user.
  The equality predicate has to be C-style: returns 0 when the 
  comparison fails, a non null integer otherwise.
  The value returned by the hashing function is corrected to respect
  the table boundaries (cast to unsigned modulo table size).
  Memory manager are defined in mem(3); the given memory manager should
  have been defined to get blocks of the size of the items that the
  table will hold. ]
  SideEffects [ Allocates memory via David Long's mem(3) library. ]
  SeeAlso     [ mem(3) ]
 */
cache1_mgr /* New cache manager */
cache_1New
(long init_size          /* Size of the cache table */,
 hash_function hash  /* Hash code routine */,
 eq_function equal    /* Equality predicate */,
 del_function delete     /* Item delete procedure */,
 pointer env,
 float rehash_threshold /* Table rehash threshold - must be greater than 0 */,
 float resize_factor    /* Rehash resize factor - must be greater than 1 */)
{
  long size;
  int size_index;
  cache1_mgr res = (cache1_mgr) mem_get_block((SIZE_T) sizeof(struct cache1_));
  size_index = 0;
  while ((primes[size_index] < init_size) && (size_index < primes_size)) {
    ++size_index;
  }
  size = primes[size_index];
  res->size = size;
  res->size_index = size_index;
  res->hash = hash;
  res->equal = equal;
  res->delete = delete;
  res->environment = env;
  res->threshold = rehash_threshold > 0 ? rehash_threshold : .5;
  if (resize_factor == 1) {
    res->rehash_limit = -1;
  } else {
    res->rehash_limit = (long) (res->threshold * (float) res->size);
  }
  res->resize_coef = (resize_factor > 1) ? resize_factor : 2;
  res->tab = (pointer *) mem_get_block((SIZE_T) size * sizeof(pointer));
  res->entries = 0;
  res->find = 0;
  res->insert = 0;
  res->hit = 0;
  res->collision = 0;
  mem_zero((pointer) res->tab, (SIZE_T) size * sizeof(pointer));
  return res;
}

/**Function**
  Synopsis           [ Deallocates a cache ]
  Description        [ Resets the cache contents and deallocates the
  memory used. The cache is no longer usable. ]
  SideEffects        [ Returns memory. ]
  SeeAlso            [ cache_1New cache_1Reset ]
**/
void
cache_1Free
(cache1_mgr c)
{
  cache_1Reset(c);
  mem_free_block((pointer) c->tab);
  mem_free_block((pointer) c);
}

/** Function **
  Synopsis    [ Clears a cache manager ]
  Description [ Resets the cache manager to its initial
                state by emptying the table of its contents. ]
  SideEffects [ Memory used to represent lists of stored items
                is freed. ]
  SeeAlso     [ ]
 */
void 
cache_1Reset
(cache1_mgr cache /* Manager to clear */)
{
  long idx;
  pointer * p;
  for (idx = cache->size, p = cache->tab; idx-- > 0 ; ++p) {
    if (* p) (* cache->delete)(* p, cache->environment);
  }
  mem_zero((pointer) cache->tab, (SIZE_T) cache->size * sizeof(pointer));
  cache->entries = 0;
  cache->find = 0;
  cache->insert = 0;
  cache->hit = 0;
  cache->collision = 0;
}


/** Function **
  Synopsis    [ Prints statistics info ]
  Description [ Prints various indicators of the cache behavior
  to the given output stream ]
  SideEffects [ Sends output to stream ]
  SeeAlso     [ ]
 */
void
cache_1Stats
(cache1_mgr cache /* Cache manager */,
 FILE * outstream /* Output stream */)
{
  fprintf(outstream, 
    "cache size: %lu\n"
    "cache entries: %lu\n"
    "cache load factor: %.3f\n"
    "cache find operations: %lu\n"
    "cache hits: %lu\n"
    "cache insert operations: %lu\n"
    "cache collision: %lu\n",
    cache->size,
    cache->entries,
    (float) cache->entries / cache -> size,
    cache->find,
    cache->hit,
    cache->insert,
    cache->collision);
}

/** Function **
  Synopsis    [ Finds data in cache structure ]
  Description [ Looks for the cache in the table equal to the 
  given parameter based on the equality predicate. Returns it 
  if found, otherwise returns 0. ]
  SideEffects [ None ]
  SeeAlso     [ ]
 */
pointer
cache_1Find
(cache1_mgr cache,
 pointer ptr /* Data to look for in the cache */)
{
  pointer *p = cache->tab + (((unsigned long)(*cache->hash)(ptr, cache->environment)) % cache->size);
  ++cache->find;
  if ((* p) && ((*cache->equal)(*p, ptr, cache->environment))) {
     ++cache->hit;
     return *p;
  }
  return 0;
}

/** Function **
  Synopsis    [ Inserts data in cache structure ]
  Description [ Looks for the data in the structure table equal to 
  the given parameter according to the equality predicate defined
  for this structure and inserts it in the structure if not found.
  Be careful that it is only of the address of the location of the
  data that it inserted, not the data itself. ]
  SideEffects [ If insertion is actually performed, memory
  is allocated to store the new pointer. The cache table might
  be resized (with routine mem_get_block) if the number of entries
  in the table exceeds the four times its size. ]
  SeeAlso     [ ]
 */
void
cache_1Insert
(cache1_mgr cache,
 pointer ptr /* Data to insert in the cache table */)
{
  pointer *p = cache->tab + (((unsigned long)(*cache->hash)(ptr, cache->environment)) % cache->size);
  ++cache->insert;
  if ((* p) && ((*cache->equal)(*p, ptr, cache->environment))) {
    return;
  }
  if (*p == 0) {
    ++cache->entries;
  } else {
    ++cache->collision;
    (* cache->delete)(*p, cache->environment);
  }
  *p = ptr;
  if (cache->entries == cache->rehash_limit) {
    _Resize(cache);
  }
}


/** Function **
  Synopsis    [ Removes data from the cache structure ]
  Description [ Looks for the data in the table equal to 
  the given parameter according to the equality predicate.
  If the data is found it is removed from the table and freed,
  Otherwise value 0 is returned. ]
  SideEffects [ None ]
  SeeAlso     [ ]
 */
int
cache_1Remove
(cache1_mgr cache,
 pointer ptr /* Data to remove from the cache table */)
{
  pointer *p = cache->tab + (((unsigned long)(*cache->hash)(ptr, cache->environment)) % cache->size);
  if ((* p) && (*cache->equal)(*p, ptr, cache->environment)) {
    (* cache->delete)(*p, cache->environment);
    --cache->entries;
    *p = 0;
    return 1;
  }
  return 0;
}

/*--------------------------------------------------------------------*/
/* Definition of internal functions                                   */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Definition of static functions                                     */
/*--------------------------------------------------------------------*/

/** Function **
  Synopsis    [ Resize a table (approximately doubling its capacity) ]
  Description [ This routine resizes and rehashes the table of the 
  given hash structure. It does so by allocating a new larger table,
  copying the contents of the original table into the new table, and
  then desallocates the original table. ]
  SideEffects [ Memory is allocated (with mem_get_block) ]
  SeeAlso     [ ]
 */
static void
_Resize
(cache1_mgr cache)
{
  pointer * newtable;
  int * size_index;
  unsigned long old_size, new_size, idx, entry;
  pointer * p;
  if (cache->size_index == primes_size)
    return;
  ++cache->resize;
  old_size = cache->size;
  new_size = (unsigned long) ((float) old_size * cache->resize_coef);
  size_index = & cache->size_index;
  while ((primes[*size_index] < new_size) && (*size_index < primes_size)) {
    ++*size_index;
  }
  cache->size = primes[*size_index];
  newtable= (pointer *) mem_get_block((SIZE_T) sizeof(rec_ptr) * cache->size);
  mem_zero((pointer) newtable, (SIZE_T) sizeof(rec_ptr) * cache->size);
  for (idx = old_size, p = cache->tab; idx-- > 0 ; ++p) {
    if (* p) {
      entry = ((unsigned long)(*cache->hash)(* p, cache->environment)) % cache->size;
      newtable[entry] = * p;
    }
  }
  mem_free_block((pointer) cache->tab);
  cache->tab = newtable;
  cache->rehash_limit = (long) (cache->threshold * (float) cache->size);
}

