/* ------------------------------------------------------------ */
/* File name:                                                   */
/*    vbdd_relat.c                                              */
/*                                                              */
/* Description:                                                 */
/*    Extra routines that manipulate BDDs.                      */
/*                                                              */
/* Project:                                                     */
/*    A symbolic model checker for VHDL                         */
/* Subproject:                                                  */
/*    A program that elaborates abstract machines from VHDL     */
/*    descriptions in the internal format.                      */
/*                                                              */
/* Author:                                                      */
/*    David Deharbe                                             */
/* Affiliation:                                                 */
/*    Carnegie Mellon University (Dept Computer Science)        */
/*                                                              */
/* ------------------------------------------------------------ */

/* ------------------------------------------------------------ */
/* File management information:                                 */
/* $Revision$ */
/* $Date$ */
/* $Locker$ */
/* $Log$ */
/* ------------------------------------------------------------ */
#include "vbdd_relat.h"
#include "bdd_up.h"

#define BDDSIZE sizeof(bdd) 

/* ------------------- Relational operators ------------------- */

/* vbdd_eq returns the BDD that represent the condition of equivalence
   of two BDD vectors. */

bdd vbdd_eq_up
#if defined(__STDC__)
   (bdd_manager bddm, vbdd v1, vbdd v2)
#else
   (bddm, v1, v2)
   bdd_manager bddm;
   vbdd v1 ;
   vbdd v2 ;
#endif
{
   register bdd result ;
   result = vbdd_eq(bddm, v1, v2) ;
   vbdd_unprotect(bddm, v1) ;
   vbdd_unprotect(bddm, v2) ;
   return(result) ;
}

bdd vbdd_eq
#if defined(__STDC__)
   (bdd_manager bddm, vbdd v1, vbdd v2)
#else
   (bddm, v1, v2)
   bdd_manager bddm;
   vbdd v1 ;
   vbdd v2 ;
#endif
{
   bdd result = bdd_one(bddm) ;
   bdd * elem1, * elem2 ;

   /* v1 = v2 if 
      v1[0] = v2[0] and vbdd_Eq(v1[1..n], v2[1..n]).
      and (K1 = K2) <=> (~K1.~K2 + K1.K2)
      The following is an iterative realization of this
      recursive definition. */ 
   elem2 = vbdd_elements(v2) ;
   elem1 = vbdd_elements(v1) ;
   while (* elem1) {
      bdd Newresult, Notv2, Ite1 ;
      Notv2 = bdd_not(bddm, * elem2) ;
      Ite1 = bdd_ite(bddm, * elem1, * elem2, Notv2) ;
      bdd_free(bddm, Notv2) ;
      Newresult = bdd_and(bddm, result, Ite1) ;
      bdd_free(bddm, Ite1) ;
      bdd_free(bddm, result) ;
      result = Newresult ;
      elem1++;
      elem2++;
   }
   return(result) ;
}

/* vbdd_neq returns the BDD that represent the condition of non-
   equivalence of two BDD vectors. */
bdd vbdd_neq_up
#if defined(__STDC__)
   (bdd_manager bddm, vbdd v1, vbdd v2)
#else
   (bddm, v1, v2)
   bdd_manager bddm;
   vbdd v1 ;
   vbdd v2 ;
#endif
{
   bdd result ;
   result = vbdd_neq(bddm, v1, v2) ;
   vbdd_unprotect(bddm, v1) ;
   vbdd_unprotect(bddm, v2) ;
   return(result) ;
}

bdd vbdd_neq
#if defined(__STDC__)
   (bdd_manager bddm, vbdd v1, vbdd v2)
#else
   (bddm, v1, v2)
   bdd_manager bddm;
   vbdd v1 ;
   vbdd v2 ;
#endif
{
   bdd result, Eqv1v2 ;

   Eqv1v2 = vbdd_eq(bddm, v1, v2) ;
   result = bdd_not(bddm, Eqv1v2) ;
   bdd_free(bddm, Eqv1v2) ;
   return(result) ;
}

/* vbdd_le returns the BDD that represent the condition "Lower
   or equal" of two BDD vectors. */
bdd vbdd_le_up
#if defined(__STDC__)
   (bdd_manager bddm, vbdd v1, vbdd v2)
#else
   (bddm, v1, v2)
   bdd_manager bddm;
   vbdd v1 ;
   vbdd v2 ;
#endif
{
   register bdd result ;
   result = vbdd_le(bddm, v1, v2) ;
   vbdd_unprotect(bddm, v1) ;
   vbdd_unprotect(bddm, v2) ;
   return(result) ;
}

bdd vbdd_le
#if defined(__STDC__)
   (bdd_manager bddm, vbdd v1, vbdd v2)
#else
   (bddm, v1, v2)
   vbdd v1 ;
   vbdd v2 ;
#endif
{
   register bdd result = bdd_one(bddm) ;
   bdd * elem1, * elem2 ;

   /* vbdd_le(v1, v2) iff 
      - n > 1:
        v1[0] < v2[0] or 
        v1[0] = v2[0] and vbdd_le(v1[1..n], v2[1..n])
      - n = 1:
        v1[0] <= v2[0]
      and (K1 <= K2) <=> (~K1 + K1.K2)
      The following is an iterative realization of this
      recursive definition. */ 
   for (elem1 = vbdd_elements(v1), elem2 = vbdd_elements(v2) ;
        (* elem1) ; 
       ++elem1) ++elem2 ;
   --elem1; --elem2;
   result = bdd_ite(bddm, * elem1, * elem2, bdd_one(bddm)) ;
   while (elem1 != vbdd_elements(v1)) {
      bdd Newresult, Ite1, Notv2, Ite2, AndResIte2 ;
      --elem1; --elem2;
      Notv2 = bdd_not(bddm, * elem2) ;
      Ite2 = bdd_ite(bddm, * elem1, * elem2, Notv2) ;
      bdd_free(bddm, Notv2) ;
      AndResIte2 = bdd_and(bddm, result, Ite2) ;
      bdd_free(bddm, Ite2) ;
      Ite1 = bdd_ite(bddm, *elem1, bdd_zero(bddm), *elem2) ;
      Newresult = bdd_or(bddm, Ite1, AndResIte2) ;
      bdd_free(bddm, result) ;
      result = Newresult ;
   }
   return(result) ;
}

/* vbdd_lt returns the BDD that represent the condition "Striclty lower"
   of two BDD vectors. */
bdd vbdd_lt_up
#if defined(__STDC__)
   (bdd_manager bddm, vbdd v1, vbdd v2)
#else
   (bddm, v1, v2)
   bdd_manager bddm;
   vbdd v1 ;
   vbdd v2 ;
#endif
{
   register bdd result ;
   result = vbdd_lt(bddm, v1, v2) ;
   vbdd_unprotect(bddm, v1) ;
   vbdd_unprotect(bddm, v2) ;
   return(result) ;
}

bdd vbdd_lt
#if defined(__STDC__)
   (bdd_manager bddm, vbdd v1, vbdd v2)
#else
   (bddm, v1, v2)
   bdd_manager bddm ;
   vbdd v1 ;
   vbdd v2 ;
#endif
{
   register bdd result ;
   register bdd * elem1, * elem2 ;
   bdd * start ;

   /* v1 < v2 if 
      - n > 1:
        v1[0] < v2[0] or 
        v1[0] = v2[0] and vbdd_lt(v1[1..n], v2[1..n])
      - n = 1:
        v1[0] < v2[0].
      and (K1 < K2) <=> (~K1.K2)
      The following is an iterative realization of this
      recursive definition. */
   start = vbdd_elements(v1) ;
   for (elem1 = start, elem2 = vbdd_elements(v2) ;
        (* elem1) ; 
       ++elem1) ++elem2 ;
   --elem1; --elem2;
   result = bdd_ite(bddm, * elem1, bdd_zero(bddm), * elem2);
   while (elem1 != vbdd_elements(v1)) {
      bdd Newresult, Ite1, Ite2, AndResIte2 ;
      --elem1; --elem2;
      Ite1 = bdd_ite(bddm, * elem1, bdd_zero(bddm), * elem2) ;
      Ite2 = bdd_ite(bddm, * elem1, * elem2, bdd_one(bddm)) ;
      AndResIte2 = bdd_and(bddm, result, Ite2) ;
      bdd_free(bddm, Ite2) ;
      Newresult = bdd_or(bddm, Ite1, AndResIte2) ;
      bdd_free(bddm, result) ;
      result = Newresult ;
   }
   return(result) ;

}

/* vbdd_ge returns the BDD that represent the condition "Greater or
   equal" of two BDD vectors. */
bdd vbdd_ge_up
#if defined(__STDC__)
   (bdd_manager bddm, vbdd v1, vbdd v2)
#else
   (bddm, v1, v2)
   bdd_manager bddm;
   vbdd v1 ;
   vbdd v2 ;
#endif
{
   register bdd result ;
   result = vbdd_ge(bddm, v1, v2) ;
   vbdd_unprotect(bddm, v1) ;
   vbdd_unprotect(bddm, v2) ;
   return(result) ;
}

bdd vbdd_ge
#if defined(__STDC__)
   (bdd_manager bddm, vbdd v1, vbdd v2)
#else
   (bddm, v1, v2)
   vbdd v1 ;
   vbdd v2 ;
#endif
{
   bdd result ;
   bdd Ltv1v2 ;

   Ltv1v2 = vbdd_lt(bddm, v1, v2) ;
   result = bdd_not(bddm, Ltv1v2) ;
   bdd_free(bddm, Ltv1v2) ;
   return(result) ;
}

/* vbdd_gt returns the BDD that represent the condition "Strictly
   greater" of two BDD vectors. */
bdd vbdd_gt_up
#if defined(__STDC__)
   (bdd_manager bddm, vbdd v1, vbdd v2)
#else
   (bddm, v1, v2)
   bdd_manager bddm;
   vbdd v1 ;
   vbdd v2 ;
#endif
{
   register bdd result ;
   result = vbdd_gt(bddm, v1, v2) ;
   vbdd_unprotect(bddm, v1) ;
   vbdd_unprotect(bddm, v2) ;
   return(result) ;
}

bdd vbdd_gt
#if defined(__STDC__)
   (bdd_manager bddm, vbdd v1, vbdd v2)
#else
   (bddm, v1, v2)
   vbdd v1 ;
   vbdd v2 ;
#endif
{
   bdd result ;
   bdd Lev1v2 ;

   Lev1v2 = vbdd_le(bddm, v1, v2) ;
   result = bdd_not(bddm, Lev1v2) ;
   bdd_free(bddm, Lev1v2) ;
   return(result) ;
}

