/* ------------------------------------------------------------ */
/* File name:                                                   */
/*    vbdd.c                                                    */
/*                                                              */
/* Description:                                                 */
/*    Extra routines that manipulate BDDs.                      */
/*                                                              */
/* Project:                                                     */
/*    A symbolic model checker for VHDL                         */
/* Subproject:                                                  */
/*    A program that elaborates abstract machines from VHDL     */
/*    descriptions in the internal format.                      */
/*                                                              */
/* Author:                                                      */
/*    David Deharbe                                             */
/* Affiliation:                                                 */
/*    Carnegie Mellon University (Dept Computer Science)        */
/*                                                              */
/* ------------------------------------------------------------ */

#include <assert.h>
#include <memuser.h>

#include "vbdd.h"

#define BDDSIZE sizeof(bdd) 

struct vbdd_ {
  int size;
  int max_size;
  bdd *elements;
};

#define VBDDSIZE (sizeof(struct vbdd_))

static rec_mgr vbdd_mem_mgr;

void vbdd_init
#if defined(__STDC__)
(void)
#else
()
#endif
{
   vbdd_mem_mgr = mem_new_rec_mgr(VBDDSIZE);
}

vbdd vbdd_new
#if defined(__STDC__)
(void)
#else
()
#endif
{
  vbdd result;
  unsigned chunk_size;
  result=(vbdd)mem_new_rec(vbdd_mem_mgr);
  result->size=0;
  result->max_size=8;
  chunk_size = BDDSIZE * 9;
  result->elements=(bdd *) mem_get_block(chunk_size);
  mem_zero(result->elements, chunk_size);
  return (result);
}

vbdd vbdd_newn
#if defined(__STDC__)
(int size)
#else
(size)
int size;
#endif
{
  vbdd result;
  unsigned chunk_size;

  if (size < 0) return 0;

  result=(vbdd)mem_new_rec(vbdd_mem_mgr);
  result->size=size;
  result->max_size=size;
  chunk_size = BDDSIZE*(size+1);
  result->elements=(bdd *) mem_get_block(chunk_size);
  mem_zero(result->elements, chunk_size);
  return (result);
}

void vbdd_free
#if defined(__STDC__)
(bdd_manager bddm, vbdd v)
#else
(bddm, v)
bdd_manager bddm;
vbdd v;
#endif
{
  vbdd_apply(bdd_free, bddm, v);
  mem_free_block(v->elements);
  mem_free_rec(vbdd_mem_mgr, v);
}


bdd * vbdd_elements
#if defined(__STDC__)
(vbdd v)
#else
(v)
vbdd v;
#endif
{
  return (v->elements);
}


bdd * vbdd_position
#if defined(__STDC__)
(vbdd v, bdd e)
#else
(v, e)
vbdd v;
bdd e;
#endif
{
  int i;

  for (i=0; i < v->size; ++i)
    if (v->elements[i] == e)
      return (v->elements+i);
  return 0;
}


bdd vbdd_add
#if defined(__STDC__)
(bdd_manager bddm, vbdd v, bdd e)
#else
(bddm, v, e)
bdd_manager bddm;
vbdd v;
bdd e;
#endif
{
  if (v->size == v->max_size) {
    v->max_size*=2;
    v->elements= 
      (bdd *) mem_resize_block(v->elements, 
                               sizeof(pointer)*(v->max_size+1));
  }
  v->elements[v->size] = (e ? bdd_identity(bddm, e) : 0);
  v->size++;
  v->elements[v->size] = 0;
  return (e);
}


void vbdd_delete
#if defined(__STDC__)
(bdd_manager bddm, vbdd v)
#else
(bddm, v)
bdd_manager bddm;
vbdd v;
#endif
{
  bdd e;

  v->size--;
  if (v->size) {
    e=v->elements[v->size];
    v->elements[v->size]=0;
    bdd_free(bddm, e);
  }
}


bdd vbdd_last
#if defined(__STDC__)
(vbdd v)
#else
(v)
vbdd v;
#endif
{
  if (v->size)
    return (v->elements[v->size-1]);
  return ((bdd)0);
}


void vbdd_set
#if defined(__STDC__)
(bdd_manager bddm, vbdd v, int idx, bdd e)
#else
(bddm, v, idx, e)
bdd_manager bddm;
vbdd v;
int idx ;
bdd e ;
#endif
{
  if (idx < v->size) {
    if (v->elements[idx])
      bdd_free(bddm, v->elements[idx]);
    v->elements[idx] = bdd_identity(bddm, e) ;
  }
}

bdd vbdd_nth
#if defined(__STDC__)
(vbdd v, int idx)
#else
(v, idx)
vbdd v;
int idx ;
#endif
{
   if (idx < v->size)
     return v->elements[idx];
   else
     return 0;
}

int vbdd_length
#if defined(__STDC__)
(vbdd v)
#else
(v)
vbdd v;
#endif
{
  return (v->size);
}


vbdd vbdd_copy
#if defined(__STDC__)
(bdd_manager bddm, vbdd v)
#else
(bddm, v)
bdd_manager bddm;
vbdd v;
#endif
{
  vbdd result;
  int i;

  result=vbdd_newn(v->size);
  for (i=0; i < v->size; ++i)
    vbdd_set(bddm, result, i, v->elements[i]);
  return result;
}

vbdd vbdd_protect
#if defined(__STDC__)
(bdd_manager bddm, vbdd v)
#else
(bddm, v)
bdd_manager bddm;
vbdd v;
#endif
{
  return vbdd_apply1(bdd_identity, bddm, v) ;
}

void vbdd_unprotect
#if defined(__STDC__)
(bdd_manager bddm, vbdd v)
#else
(bddm, v)
bdd_manager bddm;
vbdd v;
#endif
{
  register bdd * ptr ;
  for (ptr = vbdd_elements(v) ; * ptr ; ptr++) bdd_free(bddm, * ptr) ;
}

void vbdd_append
#if defined(__STDC__)
(bdd_manager bddm, vbdd v1, vbdd v2)
#else
(bddm, v1, v2)
bdd_manager bddm;
vbdd v1;
vbdd v2;
#endif
{
  bdd * ptr = vbdd_elements(v2);
  while (* ptr) {
    vbdd_add(bddm, v1, *ptr);
    ++ptr;
  }
}

vbdd vbdd_interleave
#if defined(__STDC__)
(bdd_manager bddm, vbdd v1, vbdd v2)
#else
(bddm, v1, v2)
bdd_manager bddm;
vbdd v1;
vbdd v2;
#endif
{
  int length, cpt ;
  vbdd result ;
  length = vbdd_length(v1) ;
  if (length != vbdd_length(v2))
    return 0;
  result = vbdd_newn(2 * length) ;
  cpt = 0;
  while (cpt < length) {
    vbdd_set(bddm, result, 2*cpt, vbdd_nth(v1, cpt));
    vbdd_set(bddm, result, 1 + 2*cpt, vbdd_nth(v2, cpt));
    ++ cpt;
  }
  return(result) ;
}

vbdd vbdd_apply1_up
#if defined(__STDC__)
   (bdd (* fn)(bdd_manager, bdd), bdd_manager bddm, vbdd v)
#else
   (fn, bddm, v)
   bdd (* fn)() ;
   bdd_manager bddm ;
   vbdd v ;
#endif
{
   vbdd result ;
   
   result = vbdd_apply1(fn, bddm, v) ;
   vbdd_unprotect(bddm, v) ;
   return(result) ;
}

void vbdd_apply
#if defined(__STDC__)
   (void (* fn)(bdd_manager, bdd), bdd_manager bddm, vbdd v)
#else
   (fn, bddm, v)
   void (* fn)() ;
   bdd_manager bddm ;
   vbdd v ;
#endif
{
   bdd * ptr ;

   for (ptr = vbdd_elements(v) ; *ptr ; ++ptr) 
     if (* ptr) 
       (* fn) (bddm, * ptr) ;
}

vbdd vbdd_apply1
#if defined(__STDC__)
   (bdd (* fn)(bdd_manager, bdd), bdd_manager bddm, vbdd v)
#else
   (fn, bddm, v)
   bdd (* fn)() ;
   bdd_manager bddm ;
   vbdd v ;
#endif
{
   vbdd result ;
   register bdd * ptr ;

   result = vbdd_copy(bddm, v) ;
   for (ptr = vbdd_elements(result) ; * ptr ; ++ptr) 
     * ptr = (* fn) (bddm, * ptr) ;
   return(result) ;
}

vbdd vbdd_apply2_up
#if defined(__STDC__)
   (bdd (* fn)(bdd_manager, bdd, bdd), bdd_manager bddm, vbdd v1, vbdd v2)
#else
   (fn, bddm, v1, v2)
   bdd (* fn)() ;
   bdd_manager bddm ;
   vbdd v1 ;
   vbdd v2 ;
#endif
{
   register vbdd result ;
   
   result = vbdd_apply2(fn, bddm, v1, v2) ;
   vbdd_unprotect(bddm, v1) ;
   vbdd_unprotect(bddm, v2) ;
   return(result) ;
}

vbdd vbdd_apply2
#if defined(__STDC__)
   (bdd (* fn)(bdd_manager, bdd, bdd), bdd_manager bddm, vbdd v1, vbdd v2)
#else
   (fn, bddm, v1, v2)
   bdd (* fn)() ;
   bdd_manager bddm ;
   vbdd v1 ;
   vbdd v2 ;
#endif
{
   vbdd result ;
   register bdd * ptr , * ptr2 ;

   result = vbdd_copy(bddm, v1) ;
   ptr = vbdd_elements(result) ;
   ptr2 = vbdd_elements(v2) ;

   for (ptr = vbdd_elements(result), ptr2=vbdd_elements(v2);
	* ptr;
	++ptr, ++ptr2)
     * ptr = (* fn) (bddm, * ptr, * ptr2) ;

   return(result) ;
}

vbdd vbdd_distribute_up
#if defined(__STDC__)
   (bdd (* fn)(bdd_manager, bdd, bdd), bdd_manager bddm, bdd f, vbdd v,
    int to_the_right)
#else
   (fn, bddm, f, v, to_the_right)
   bdd (* fn)() ;
   bdd_manager bddm ;
   bdd f ;
   vbdd v ;
   int Totheright
#endif
{
   register vbdd result ;
   result = vbdd_distribute(fn, bddm, f, v, to_the_right) ;
   bdd_free(bddm, f) ;
   vbdd_unprotect(bddm, v) ;
   return(result) ;
}

vbdd vbdd_distribute_up2
#if defined(__STDC__)
   (bdd (* fn)(bdd_manager, bdd, bdd), bdd_manager bddm, bdd f, vbdd v,
    int to_the_right)
#else
   (fn, bddm, f, v, to_the_right)
   bdd (* fn)() ;
   bdd_manager bddm ;
   bdd f ;
   vbdd v ;
   int to_the_right ;
#endif
{
   register vbdd result ;
   result = vbdd_distribute(fn, bddm, f, v, to_the_right) ;
   vbdd_unprotect(bddm, v) ;
   return(result) ;
}

vbdd vbdd_distribute
#if defined(__STDC__)
   (bdd (* fn)(bdd_manager, bdd, bdd), bdd_manager bddm, bdd f, vbdd v, 
    int to_the_right)
#else
   (fn, bddm, f, v, to_the_right)
   bdd (* fn)() ;
   bdd_manager bddm ;
   bdd f ;
   bdd * v ;
   int to_the_right ;
#endif
{
   vbdd result ;
   register bdd * ptr ;

   result = vbdd_copy(bddm, v) ;
   ptr = vbdd_elements(result) ;

   while (* ptr) {
      if (to_the_right) * ptr = (* fn) (bddm, f, * ptr) ;
      else * ptr = (* fn) (bddm, * ptr, f) ;
      ++ptr;
    }
   return(result) ;
}


