/**CFile***********************************************************************

  FileName    [ antCompute.c ]

  PackageName [ ant ]

  Synopsis    [ Set antecedent computation routines ]

  Description [ Antecedent of a set is computed by replacing each
  occurence of a variable in the Shannon's expansion of its
  characteristic function by the corresponding transition function. 
  Variables that do not have a transition function (input variables)
  are existentially quantified out. ]

  SeeAlso     [ ]

  Author      [ David D&eacuteharbe ]

  Copyright   [ Carnegie Mellon University - 1996 ]

******************************************************************************/
#include <stdio.h>
#include "antInt.h" 

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Structure declarations                                                    */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/


/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static bdd _Aux(const ant_manager t, bdd set);

/**AutomaticEnd***************************************************************/


/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function**
  Synopsis           [ Antecedent computation ]
  Description        [ Returns the BDD of the characteristic function for
  the set of states in the inverse image of the set of states characterized
  by set. ]
  SideEffects        [ Stores intermediate computations in the internal
  cache of t and allocates memory. If a BDD overflow  occurs, the function 
  bdd_resize_manager is invoked. The result is correct, as well as all
  computations stored in the cache regardless of BDD overflows happening
  or not. ]
**/
bdd
ant_Compute
(const ant_manager t,
 bdd set)
{
  bdd proj;
  /* checks that the map from variables to functions is defined for
     all variables of the bdd_manager */

  /* existential quantification of all variables that do not have
     an associated transition function */
  bdd_assoc(t->bddm, t->var_assoc);
  proj = bdd_project(t->bddm, set);

  /* computation */
  return _Aux(t, proj);
}


/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

/**Function**
  Synopsis           [ Actual recursive computation ]
  Description        [ Substitution of each node of set with the
  transition function of the corresponding variable. This is 
  performed by a recursive implementation of a DFS traversal of
  the graph structure. This function relies on the fact that all
  nodes in the graph are mapped to a function. Intermediate results 
  are stored in the cache. ]
  SideEffects        [ Memory is allocated and the internal cache
  is modified. The function bdd_resize_manager is called whenever
  a bdd_overflow is detected. ]
  SeeAlso            [ ant_Compute ]
**/

static bdd
_Aux
(const ant_manager t, 
 bdd set)
{
  bdd_manager bddm = t->bddm;
  bdd result, topvar, topfun;
  bdd not_topvar, not_set, set_in_variable, set_out_of_variable;
  cache_entry ptr;

  if ((set == bdd_zero(bddm)) || (set == bdd_one(bddm))) {
    return set;
  }

  {
    cache_entry_rec rec;

    rec.set = set;
    if ((ptr = cache_XFind(t->cache, (pointer) & rec)) != 0) {
      return bdd_identity(bddm, ptr->ant);
    }
  }

  {
    long topid;
    topid = bdd_if_id(bddm, set);
    topvar = bdd_var_with_id(bddm, topid);
    topfun = antGetFunction(t, topid);
  }

  do {
    not_topvar = bdd_not(bddm, topvar) ;
    not_set = bdd_not(bddm, set) ;
    set_in_variable = bdd_or(bddm, not_set, topvar) ;
    set_out_of_variable = bdd_or(bddm, not_set, not_topvar) ;
  
    if (set_in_variable == bdd_one(bddm)) {
      bdd tmp;
      tmp = bdd_then(bddm, set);
      result = bdd_andup2(bddm, topfun, _Aux(t, tmp));
      bdd_free(bddm, tmp);
    } else if (set_out_of_variable == bdd_one(bddm)) {
      bdd tmp;
      tmp = bdd_else(bddm, set);
      result = bdd_andup(bddm, bdd_not(bddm, topfun), _Aux(t, tmp)) ;
      bdd_free(bddm, tmp);
    } else {
      bdd tmp, low, high ;
      /* not set_out_of_variable <==> not (set => not variable) 
         <==> not (not set or not variable)
	 <==> set and variable [CQFD]
	 for efficiency, negation is performed instead of conjunction */
      tmp = bdd_then(bddm, set) ;
      low = bdd_andup2(bddm, topfun, _Aux(t, tmp));
      bdd_free(bddm, tmp) ;
      /* not set_in_variable <==> not (set => variable) 
         <==> not (not set or variable)
	 <==> set and not variable [CQFD]
	 for efficiency, negation is performed instead of conjunction */
      tmp = bdd_else(bddm, set) ;
      high = bdd_andup(bddm, bdd_not(bddm, topfun), _Aux(t, tmp));
      bdd_free(bddm, tmp) ;
      result = bdd_orup(bddm, low, high) ;
    }
    bdd_free(bddm, not_topvar) ;
    bdd_free(bddm, not_set) ;
    bdd_free(bddm, set_in_variable) ;
    bdd_free(bddm, set_out_of_variable) ;
    if (bdd_overflow(bddm)) {
      bdd_free(bddm, result);
      bdd_resize_manager(bddm);
    } else {
      break;
    }
  } while (1);

  ptr = mem_new_rec(t->memory);
  ptr->set = bdd_identity(bddm, set);
  ptr->ant = bdd_identity(bddm, result);
  cache_XInsert(t->cache, (pointer) ptr);

  return result;
}

