/* ------------------------------------------------------------ */
/* File name:                                                   */
/*    utilities.c                                               */
/*                                                              */
/* Description:                                                 */
/*    Miscellaneous routines that do simple jobs.               */
/*                                                              */
/* Project:                                                     */
/*    A symbolic model checker for VHDL                         */
/*                                                              */
/* Author:                                                      */
/*    David Deharbe                                             */
/* Affiliation:                                                 */
/*    Carnegie Mellon University (Dept Computer Science)        */
/*                                                              */
/* ------------------------------------------------------------ */

#include <string.h>
#include <memuser.h>

#include "cvc.h"

/* ------------------------------------------------------------ */
void add_identifier 
  (Model m, cvn id)
{
  cvn decl = qDeclaration(id);
  add_declaration(m, decl);
}

void add_declarations 
  (Model m, cvn decls)
{
  while (decls) {
    add_declaration(m, decls);
    decls = qNext(decls);
  }
}

void add_declaration 
  (Model m, cvn decl)
{
  bdd_object obj;
  if (IsA(decl, kSignalDeclaration)) {
    obj= encode_decl(decl);
    model_RepresentationAddSignal(m, decl, obj->coding, obj->default_coding,
                                  obj->valid);
  } else if (IsA(decl, kVariableDeclaration)) {
    obj= encode_decl(decl);
    model_RepresentationAddVariable(m, decl, obj->coding, obj->default_coding,
                                  obj->valid);
  } 
}

void add_previous
  (Model m, cvn decl)
{
  if (!model_RepresentationSignalHasPrevious(m, decl)) {
    char * name= mem_get_block(sizeof(char) * (9 + strlen(qName(decl))));
    cvn prev_decl;
    bdd_object prev_obj;
    sprintf(name, "%s'delayed", qName(decl));
    prev_decl= mVariableDeclaration(0, 0, decl, 0,
				    name, qSubtype(decl), kInternal,
				    qDefaultValue(decl));
    prev_obj= encode_decl(prev_decl);
    model_RepresentationAddPrevious(m, decl, prev_decl,
                                    prev_obj->coding, prev_obj->valid);
  }
}

int is_previous
(cvn decl)
{
  char * c = qName(decl);
  while (* c && (* c != '\'')) ++c;
  return (strcmp(c, "'delayed") == 0);
}

/* --                 ----------------                          */
/* -- Routine:        ReduceExpression                          */
/* --                 ----------------                          */
/* -- Description:    */

cvn ReduceExpression
   (cvn Node)
{
   register cvn Result ;
   if (NullNode(Node)) {
      Result = 0;
   } else if (IsA(Node, kIdentifier)) {
      cvn Declaration = qDeclaration(Node) ;
      if (IsA(Declaration, kSignalDeclaration)) {
         return Node;
      } else {
         if (qToolInfo(Declaration) == 0) {
            Result = Node ;
         } else {
            Result = qToolInfo(Declaration) ;
         }
      }
   } else if (IsA(Node, kUnary)) {
      Result = cvn_copy(Node) ;
      sOperand(Result, ReduceExpression(qOperand(Node))) ;
   } else if (IsA(Node, kBinary)) {
      Result = cvn_copy(Node) ;
      sLeft(Result, ReduceExpression(qLeft(Node)));
      sRight(Result, ReduceExpression(qRight(Node)));
   } else if ((IsA(Node, kIntegerValue)) ||
              (IsA(Node, kEnumeratedLiteral)) ||
              (IsA(Node, kCharacterLiteral))) {
      Result = Node;
   } else {
      Result = 0;
   }
   return(Result) ;
}

/*                 -------------                             */
/* Routine:        CountBranches                             */
/*                 -------------                             */
/* Description: Count the number of branches of an if        */
/* statement. Examples:                                      */
/*   if C then <stm> end if ; -> 2  (implicit else counts)   */
/*   if C then <stm1> else <stm2> end if ; -> 2              */

unsigned CountBranches
   (cvn stm)
{
  if (IsA(stm, kIfStatement)) {
    cvn Previous = qBranches(stm),
        Branch = qNext(Previous),
        Condition ;
    unsigned Cpt = 1;
    while (Branch) {
      ++Cpt;
      Previous = Branch ;
      Branch = qNext(Branch) ;
    }
    Condition = qCondition(Previous) ;
    /* There is no explicit else part, add one branch for it */
    if (Condition) { 
      ++Cpt;
    }
    return Cpt;
    /* There is at least one branch */
  } else if (IsA(stm, kCaseStatement)) {
    cvn alternative;
    unsigned result;
    for (alternative = qAlternatives(stm), result = 0;
         !NullNode(alternative);
         alternative = qNext(alternative), ++result);
    return result;
  }
  return 0;
}

