/* ------------------------------------------------------------ */
/* File name:                                                   */
/*    encode.c                                                  */
/*                                                              */
/* Description:                                                 */
/*    Routines that encode VHDL data types and values into      */
/*    a symbolic representation, by means of BDDs.              */
/*                                                              */
/* Project:                                                     */
/*    A symbolic model checker for VHDL                         */
/* Subproject:                                                  */
/*    A program that elaborates abstract machines from VHDL     */
/*    descriptions in the internal format.                      */
/*                                                              */
/* Author:                                                      */
/*    David Deharbe                                             */
/* Affiliation:                                                 */
/*    Carnegie Mellon University (Dept Computer Science)        */
/*                                                              */
/* ------------------------------------------------------------ */

#include <stdlib.h>
#include <assert.h>
#include <stdio.h>
#include <sys/types.h>
#include <string.h>

#include <Errors.h>

#include "cvc.h"

/* ------------------------------------------------------------ */

bdd_manager bddm ;

/* ------------------------------------------------------------ */
static cvn InitIdentifier ;
static cvn InitVarDecl ;

/* ------------------------------------------------------------ */
#define minimum(a, b) (a <= b) ? a : b
#define maximum(a, b) (a >= b) ? a : b

/* ------------------------------------------------------------ */
static char * vhdl_type_str (vhdl_type);
static void vhdl_type_print (vhdl_type);
static char * coded_type_str (coded_type);
static void coded_type_print (coded_type);
static void bdd_object_print (bdd_object);
static void bdd_value_print (bdd_value);

/* ------------------------------------------------------------ */
/* InitEncode
      args: _none_
      value: void
      Initializes the encode and bdd data structures.
*/

void encode_init ()
{
  /* Initialize the bdd library */
  vbdd_init() ;

  {
    InitVarDecl = 
      mVariableDeclaration(0, 0, 0, 0, (char *) strdup("_init_"),
			   mSubtype(0, 0, boolean_type, 0), kInternal,
                           true_literal) ;
    InitIdentifier = mIdentifier(0, (char *) strdup("_init_"),
				 InitVarDecl) ;
       
    InitCondition = InitIdentifier ;
  }
}

void
encode_close
(void)
{
  bdd_short_stats(bddm, stdout) ;
  bdd_quit(bddm) ;
}

/* ------------------------------------------------------------ */
/* encode_subtype
   args: cvn subtype
   value: coded_type
   We assume that Type is a node of the class TypeDefinition,
   that represents the definition of a subtype.
   If the attribute aToolInfo is null, the BDD encoding info
   is stored under this attribute, otherwise, it is retrieved
   from the attribute.
   */

coded_type
encode_subtype
(cvn subtype)
{
  int cpt;
  int size, encode_size ;
  vbdd variables, default_value ;
  coded_type attribute ;
   
  /* If the computation has already been done, stop the routine */
  if (qToolInfo(subtype) != 0) {
    return((coded_type) qToolInfo(subtype)) ;
  }
  /* Allocate memory to store the elaborated information */
  assert ((attribute = malloc(sizeof(struct coded_type_))) != 0) ;

  /* Compute the number of elements in the data type, and the
     number of binary variables needed to represent this number
     of elements */
  size = cv_subtype_size(subtype) ;
  attribute->size = size ;
  encode_size = vbdd_code_length(size) ;
  
  /* Allocate a set of binary variables */ 
  variables = vbdd_newn(encode_size) ;
  cpt = 0 ;
  while (cpt<encode_size) {
    vbdd_set(bddm, variables, cpt++, bdd_new_var_last(bddm)) ;
  }
  attribute->variables = variables ;

  /* Compute the condition under which a valuation of the
     variables actually represents a value of the type */
  attribute->valid = vbdd_code_valid(bddm, variables, size) ;

  /* Compute the default value associated to the type: it
     is always the first element, it is encoded by a all-zeroes
     cube. */
  default_value = vbdd_newn(encode_size) ;
  cpt = 0;
  while (cpt < encode_size) vbdd_set(bddm, default_value, cpt++, bdd_zero(bddm)) ;
  attribute->default_coding = default_value ;
  
  /* Add the elaborated info to the intermediate format */ 
  sToolInfo(subtype, attribute) ;
  
  return attribute;
}

/* ------------------------------------------------------------ */
/* encode_constant
   args: cvn Constant
   value: bdd_value
   We assume that Object is a node of the class IntegerValue,
   CharacterLiteral, or EnumeratedLiteral that represents a
   constant.
   */

bdd_value
encode_constant
(cvn constant)
{
  coded_type subtype_info ;
  bdd_value res ;
  int encode_size ;
  vbdd variables ;
  cvn subtype = qSubtype(constant) ;
  cvn basetype = qBaseType(subtype) ;
  cvn range = qConstraint(subtype);
  int ord ;

  if (qToolInfo(constant) != 0) {
    return qToolInfo(constant);
  }
   
  subtype_info = encode_subtype(subtype) ;
  if ((range == 0) && IsA(basetype, kIntegerType)) {
    range = qRange(basetype);
  }
  /* :- IsA(basetype, kIntegerType) || qConstraint(subtype) != 0
   *    => range != 0 
   */

  /* Allocate memory to store elaborated information */
  assert ((res = malloc(sizeof(struct bdd_value_))) != 0) ;

  if (IsA(basetype, kIntegerType)) {
    ord = (int) qIntegerValue(constant) - qIntegerValue(qLeft(range)) ;
  } else {
    if (range) {
      ord = (int) qOrdinalValue(constant) - qOrdinalValue(qLeft(range));
    } else {
      ord = (int) qOrdinalValue(constant) ;
    }
  }
   
  encode_size = vbdd_length(subtype_info->variables) ;

  variables = vbdd_code_ordinal(bddm, encode_size, ord) ;
  res->valid = bdd_one(bddm) ;
  res->coding = variables ;

  /* Store the result in the intermediate format representation,
     but not for integers, that can really belong to different
     subtypes. */
  if (IsA(basetype, kEnumerationType)) {
    (res->type).enum_t.kind = kEnumType ;
    (res->type).enum_t.node = subtype;
    sToolInfo(constant, res) ;
  } else {
    (res->type).int_t.kind = kSRType ;
    (res->type).int_t.node = subtype;
    (res->type).int_t.offset = qIntegerValue(qLeft(range)) ;
    (res->type).int_t.range = encode_size ;
  }

  return res;
}

/* ------------------------------------------------------------ */
/* encode_object
   args: cvn Object
   value: bdd_value
   We assume that Object is a node of the class Identifier,
   that represents the identifier of a variable or signal.
   */

bdd_value
encode_object
(cvn object)
{
  bdd_object attribute ;
  coded_type subtype_info ;
  bdd_value res = malloc(sizeof(struct bdd_value_));
  cvn subtype = qSubtype(object) ;
  cvn basetype = qBaseType(subtype) ;
  cvn range = qConstraint(subtype);
  int encode_size ;

  attribute= encode_decl(qDeclaration(object));
  subtype_info = (coded_type) encode_subtype(subtype) ;
  encode_size = vbdd_length(subtype_info->variables) ;

  /* Compute the different fields of the result */
  if (IsA(basetype, kIntegerType)) {
    res->type.int_t.kind = kSRType ;
    res->type.int_t.node = subtype;
    if (range) {
      res->type.int_t.offset = qIntegerValue(qLeft(range));
    } else {
      res->type.int_t.offset = qIntegerValue(qLeft(qRange(basetype))) ;
    }
    res->type.int_t.range = encode_size ;
  } else if (IsA(basetype, kEnumerationType)) {
    res->type.enum_t.kind = kEnumType ;
    res->type.enum_t.node = subtype;
  }
  res->coding = attribute->coding ;
  res->valid = attribute->valid ;
  
  return res;
}

/* ------------------------------------------------------------ */
/* encode_decl
   args: cvn decl
   value: bdd_object
   We assume that decl is a node of the class kSignalDeclaration
   or kVariableDeclaration.
   */

bdd_object
encode_decl
(cvn decl)
{
  bdd_object res ;
  coded_type subtype_info ;
  cvn subtype = qSubtype(decl) ;
  cvn basetype = qBaseType(subtype) ;
  int encode_size ;

  /* check assumptions */
  assert (IsA(decl, kSignalDeclaration) || IsA(decl, kVariableDeclaration));

  if (qToolInfo(decl) != 0) {
    res = (bdd_object) qToolInfo(decl) ;
  } else {
    vbdd variables ;
    bdd valid ;

    if (IsA(basetype, kIntegerType) || IsA(basetype, kEnumerationType)) {
      subtype_info = encode_subtype(subtype) ;
    } else {
      fprintf(stderr, "User error: type of declaration %s not allowed yet.\n", 
	      qName(decl)) ;
      exit(ERROR_EXIT_CODE) ;
    }

    /* Allocate memory to store elaborated information */
    assert ((res = malloc(sizeof(struct bdd_object_))) != 0);

    /* Allocate the correct number of binary variables needed to 
       represent the object */
    /* Allocate the correct number of binary variables needed to 
       represent the object */
    encode_size = vbdd_length(subtype_info -> variables) ;
    variables = vbdd_new() ;
    /* heuristic 1: put init_var_decl at the top of the BDDs */
    if (cv_eq(decl, InitVarDecl)) {
      vbdd_add(bddm, variables, bdd_new_var_first(bddm)) ;
    /* heuristic 2: interleave the coding of a signal with the coding of 
       its previous value */
    } else if (is_previous(decl)) {
      bdd_object sig = encode_decl(qNext(decl));
      bdd * v;
      for (v = vbdd_elements(sig->coding); * v; ++v) {
        vbdd_add(bddm, variables, bdd_new_var_after(bddm, * v));
        bdd_new_var_block(bddm, * v, 2);
      }
    } else {
      int cpt;
      for (cpt = 0 ; cpt  < encode_size ; ++cpt) 
	vbdd_add(bddm, variables, bdd_new_var_last(bddm)) ;
    }
    /* Compute the valid under which the object represents a value of its type */
    {
      vbdd substitution ;
      valid = subtype_info->valid ;
      substitution = vbdd_interleave(bddm, subtype_info->variables, variables) ;
      bdd_temp_assoc(bddm, vbdd_elements(substitution), 1);
    }
    res->valid= bdd_substitute(bddm, subtype_info->valid) ;
    if (IsA(decl, kSignalDeclaration)) 
      res->default_coding = encode_constant(qDefaultValue(decl))->coding;
    else
      res->default_coding = encode_constant(qInitialValue(decl))->coding;
    res->coding = variables ;
  
    /* Store the result in the intermediate format representation */
    sToolInfo(decl, res) ;
  }
  return res;
}

/* ------------------------------------------------------------ */
/* adjust_values
   args: bdd_value v1, v2
   result: bdd_value
   */
bdd_value *
adjust_values
(bdd_value v1,
 bdd_value v2)
{
  bdd_value * res ;

  res = calloc(2, sizeof(bdd_value)) ;
  res[0] = malloc(sizeof(struct bdd_value_)) ;
  res[1] = malloc(sizeof(struct bdd_value_)) ;
  if ((v1->type.int_t.kind == kSRType) && (v2->type.int_t.kind == kSRType)) {
    int offset1, offset2, offsetMin ;
    int range1, range2, rangeNew ;
    vhdl_type TypeR ;
    int SizeR ;
    vbdd coding1, coding2, codingR1, codingR2 ;

    offset1 = v1->type.int_t.offset;
    offset2 = v2->type.int_t.offset;

    range1 = v1->type.int_t.range ;
    range2 = v2->type.int_t.range ;

    coding1 = v1->coding ;
    coding2 = v2->coding ;

    offsetMin = minimum(offset1, offset2) ;
    rangeNew = maximum(offset1 + range1, offset2 + range1) - offsetMin ;

    SizeR = vbdd_code_length(rangeNew) ;
    TypeR.int_t.kind = kSRType ;
    TypeR.int_t.node = 0 ;
    TypeR.int_t.offset = offsetMin ;
    TypeR.int_t.range = rangeNew ;

    if (offset1 == offsetMin) {
      codingR1 = coding1;
    } else {
      codingR1 = 
	vbdd_plus(bddm,
		  coding1,
		  vbdd_code_ordinal(bddm, (int) vbdd_length(coding1), 
				   (int) offset2 - offset1)) ;
    }
    if (offset2 == offsetMin) {
      codingR2 = coding2;
    } else {
      codingR2 = 
	vbdd_plus(bddm,
		  coding2,
		  vbdd_code_ordinal(bddm, (int) vbdd_length(coding2), 
				   (int) offset1 - offset2)) ;
    }

    res[0]->type = TypeR ;
    res[0]->coding = codingR1 ;
    res[0]->valid = v1->valid ;
    
    res[1]->type = TypeR ;
    res[1]->coding = codingR2 ;
    res[1]->valid = v2->valid ;

  } else {

    res[0] = v1 ;
    res[1] = v2 ;

  } 
  return res;
}


/* ------------------------------------------------------------ */
/* vhdl_type_equal
   args: */

int
vhdl_type_equal
(vhdl_type T1,
 vhdl_type T2)
{
  return T1.int_t.node == T2.int_t.node;
}

/* ---------------------------------------- bdd_to_bdd_value -- */
bdd_value
bdd_to_bdd_value
(bdd f)
{
  bdd_value res;

  /* Allocate memory to store elaborated information */
  assert ((res = malloc(sizeof(struct bdd_value_))) != 0) ;

  res->valid = bdd_one(bddm) ;
  res->coding = vbdd_newn(1);
  vbdd_set(bddm, res->coding, 0, f);
  res->type.enum_t.kind = kEnumType ;
  res->type.enum_t.node = boolean_type;

  return res;
}

/* ------------------------------------------------------------ */
static char *
vhdl_type_str
(vhdl_type type)
{
  char * res ;
  res = malloc(100 * sizeof(char)) ;

  if (type.int_t.kind == kSRType) {
    sprintf(res,
	    "{kind: tSRKind; offset: %i; range: %u}",
	    type.int_t.offset, type.int_t.range) ;
  } else {
    sprintf(res, "{kind: tEnumKind}") ;
  }
  return(res) ;
}

static void
vhdl_type_print
(vhdl_type type)
{
  char * Message ;
  Message = vhdl_type_str(type) ;
  printf("vhdl_type type:\n") ;
  printf("  %s\n", Message) ;
  free(Message) ;
}

static char *
coded_type_str
(coded_type type)
{
  char * res ;
  res = malloc(100 * sizeof(char)) ;
  sprintf(res, "{size: %u; encode size: %u}", 
          type->size , vbdd_length(type->variables)) ;
  return(res) ;
}

static void
coded_type_print
(coded_type type)
{
  char * Message ;
  Message = coded_type_str(type) ;
  printf("coded_type type:\n") ;
  printf("   %s\n", Message) ;
  free(Message) ;
  printf("   valid:\n") ;
  bdd_print_sop(bddm, type->valid, bdd_naming_fn_none, 0) ;
  printf("   variables:\n") ;
  vbdd_print_sop(bddm, type->variables, bdd_naming_fn_none, 0);
  printf("   default coding:\n") ;
  vbdd_print_sop(bddm, type->default_coding, bdd_naming_fn_none, 0);
}

void
bdd_object_print
(bdd_object Object)
{
  printf("bdd_object Object:\n") ;
  printf("   valid:\n") ;
  bdd_print(bddm, Object->valid) ;
  printf("   coding:\n") ;
  vbdd_print_sop(bddm, Object->coding, bdd_naming_fn_none, 0) ;
  printf("   default coding:\n") ;
  vbdd_print_sop(bddm, Object->default_coding, bdd_naming_fn_none, 0) ;
}

void
bdd_value_print
(bdd_value aValue)
{
  printf("bdd_value Value:\n") ;
  vhdl_type_print(aValue->type),
  printf("   size: %u}\n", vbdd_length(aValue->coding)) ;
  printf("   valid:\n") ;
  bdd_print_sop(bddm, aValue->valid, bdd_naming_fn_none, 0) ;
  printf("   coding:\n") ;
  vbdd_print_sop(bddm, aValue->coding, bdd_naming_fn_none, 0) ;
}

/* ------------------------------------------------------------ */
void
bdd_value_free
(bdd_value val)
{
  bdd_free(bddm, val->valid);
  vbdd_free(bddm, val->coding);
}

