/* ------------------------------------------------------------ */
/* File name:                                                   */
/*   design-entity.c                                            */
/*                                                              */
/* Description:                                                 */
/*   Routines that elaborate the symbolic model of a design     */
/*   entity.                                                    */
/*                                                              */
/* Project:                                                     */
/*    A symbolic model checker for VHDL                         */
/* Subproject:                                                  */
/*    A program that elaborates symbolic models from VHDL       */
/*    descriptions in the internal format.                      */
/*                                                              */
/* Author:                                                      */
/*    David Deharbe                                             */
/* Affiliation:                                                 */
/*    Carnegie Mellon University (Dept Computer Science)        */
/*                                                              */
/* ------------------------------------------------------------ */

/* ------------------------------------------------------------ */
#include "cvc.h"

static void elaborate_architecture_body (cvc_context c, cvn body) ;

/* ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ 
    . routine: 
       elaborate_design_entity
              
    . arguments: 
       Model m
       cvn design
    . result   : 
       void (see description)
    . description : 
       Computes the bdd-based model for the secondary VHDL
       entity "design".
       Presently, the unique type of design entity accepted
       is the architecture body.
    . side effects: 
       The result is passed by the parameter "m".
    . invokes     :                             . module:
       printf                                      stdio
       qKind, qLine                                canif
       elaborate_architecture_body                 design_entity
    . invoked by  :                             . module:
    . date: 
    . modifications: 
    . author: 
       David Deharbe.
   ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */

void
elaborate_design_entity
(cvc_context c,
 cvn design)
{
  switch(qKind(design)) {
     case kArchitectureBody :
       elaborate_architecture_body(c, design) ;
       break ;
     default :
       fprintf(c->outstream,
               "Elaboration of design entity line %u not implemented yet.\n",
               qLine(design)) ;
     exit(ERROR_EXIT_CODE);
  }
}

/* ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ 
    . routine: 
       elaborate_architecture_body
              
    . arguments: 
       Model m
       cvn body
    . result   : 
       void (see description)
    . description : 
       Computes the bdd-based Model for the secondary VHDL
       entity "body".
       Additional comments are included in the code.
    . side effects: 
       The result is passed by the parameter "m".
    . invokes     :                             . module:
       qDeclaration, qPorts, qPrimary               canif
       bdd_reorder                                 bdd
       vbdd_zero                                   bdd+
       elaborate_block_statement                   concurrent_statement
       add_declaration, add_declarations           utilities
    . invoked by  :                             . module:
       elaborate_design_entity                     design_entiy
    . date: 
       1995
    . modifications: 
    . author: 
       David Deharbe.
   ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */

static void
elaborate_architecture_body
(cvc_context c,
 cvn body)
{
  /* add the proposition InitCondition as a variable of m */
  {
    cvn decl= qDeclaration(InitCondition);
    vbdd transition= vbdd_zero(bddm, 1);
    add_declaration(c->model, decl);
    model_RepresentationAddTransition(c->model, decl, transition); 
  }
  /* add the ports of the entity to the signals of the model */
  add_declarations(c->model, qPorts(qPrimary(body)));

  /* Elaborate the abstract machine from the statement part */
  elaborate_block_statement(c, body) ;
  if (c->reorder_coeff >= 1) {
    bdd_reorder(bddm) ;
  }
}

/* ------------------------------------------------------------ */
