/* ------------------------------------------------------------ */
/* File name:                                                   */
/*   concurrent-statement.c                                     */
/*                                                              */
/* Description:                                                 */
/*   Routines that elaborate the symbolic model out of a block  */
/*   statement.                                                 */
/*                                                              */
/* Project:                                                     */
/*    A symbolic model checker for VHDL                         */
/* Subproject:                                                  */
/*    A program that elaborates abstract machines from VHDL     */
/*    descriptions in the internal format.                      */
/*                                                              */
/* Author:                                                      */
/*    David Deharbe                                             */
/* Affiliation:                                                 */
/*    Carnegie Mellon University (Dept Computer Science)        */
/*                                                              */
/* ------------------------------------------------------------ */

#include <stdio.h>
#include "cvc.h"

/* ------------------------------------------------------------ */
static void elaborate_signal_assignment_statement
   (cvc_context c, cvn stm);
static void elaborate_conditional_signal_assignment_statement
   (cvc_context c, cvn stm);
static void elaborate_selected_signal_assignment_statement
   (cvc_context c, cvn stm);

/* ------------------------------------------------------------ */
void
elaborate_concurrent_statement
(cvc_context c,
 cvn stm)
{
   switch(qKind(stm)) {
      case kBlockStatement : 
         elaborate_block_statement(c, stm) ;
	 break;
      case kProcessStatement :
         elaborate_process_statement(c, stm) ;
	 break;
      case kSignalAssignmentStatement :
         elaborate_signal_assignment_statement(c, stm) ;
	 break;
      case kConditionalSignalAssignmentStatement :
         elaborate_conditional_signal_assignment_statement(c, stm) ;
	 break;
      case kSelectedSignalAssignmentStatement :
         elaborate_selected_signal_assignment_statement(c, stm) ;
	 break;
      default :
	fprintf(stderr, 
		"statement line %u not implemented.\n",
		qLine(stm)) ;
         exit(ERROR_EXIT_CODE) ;
   }
}

/* ------------------------------------------------------------ */
void
elaborate_block_statement
(cvc_context c,
 cvn blk)
{
  cvn stm ;

  /* add the signals of the declarative part to the signals of the model */
  add_declarations(c->model, qDeclarations(blk));

  /* Elaborate the abstract machine from the statement part */
  for (stm = qStatements(blk) ; stm ; stm = qNext(stm)) {
    elaborate_concurrent_statement(c, stm) ;
    if (c->reorder_coeff == 4) {
      bdd_reorder(bddm) ;
    }
  }
  
  if (((c->reorder_coeff >= 2) && !(IsA(blk, kArchitectureBody))) ||
      (c->reorder_coeff >= 3)){
    bdd_reorder(bddm) ;
  }
}

/* ------------------------------------------------------------ */
static void
elaborate_signal_assignment_statement
(cvc_context c,
 cvn stm)
{
  cvn tgt, src, decl ;
  vbdd coding;
  bdd stable, resume ;
  cvn support = 0 ;

  tgt= qTarget(stm);
  decl= qDeclaration(tgt);
  coding= model_RepresentationQCoding(c->model, decl);
  src= qValue(qSource(stm));
  support= cv_signal_support(src);

  /* the statement activity is sensitive to each signal that appears in 
     the rhs of the assignment,
     thus check if a variable carrying the previous value exists, if not
     create it */
  {
    cvn elem ; 
    bdd init;
    init = condition_code(InitCondition, c->expr_mgr);
    stable = bdd_identity(bddm, bdd_one(bddm));
    resume = bdd_identity(bddm, init);
    for (elem = support ; elem ; elem = qNext(elem)) {
      bdd event;
      cvn decl = qDeclaration(qValue(elem));
      add_previous(c->model, decl);
      event= model_RepresentationQSignalEvent(c->model, decl);
      resume= bdd_orup2(bddm, event, resume);
      if (qMode(decl) != kIn) {
        stable = bdd_andup2(bddm, stable, bdd_not(bddm, event));
      }
    }
    model_RepresentationAddStable(c->model, stable);
  }
  
  /* the assignment being:
       tgt<=src;
     the transition is:
        src when some_event(support(src)) else
        tgt */
  {
    vbdd transition ;
    bdd_value src_value;
    src_value = expression_value(c->expr_mgr, src);
    transition =
      vbdd_ite(bddm, resume, src_value->coding, coding);
    model_RepresentationAddDriver(c->model, decl, stm, transition);
   }
}

/* ------------------------------------------------------------ */
static void
elaborate_conditional_signal_assignment_statement
(cvc_context c,
 cvn stm)
{
  cvn tgt, src, decl ;
  vbdd coding;
  bdd resume;
  cvn support = 0 ;

  tgt= qTarget(stm);
  decl= qDeclaration(tgt);
  coding= model_RepresentationQCoding(c->model, decl);
  src= qSource(stm);
  support= cv_signal_support(src);

  /* the statement activity is sensitive to each signal that appears in 
     the rhs of the assignment,
     thus check if a variable carrying the previous value exists, if not
     create it */
  {
    cvn elem ; 
    bdd stable;
    bdd init;
    init = condition_code(InitCondition, c->expr_mgr);
    stable = bdd_identity(bddm, bdd_one(bddm));
    resume = bdd_identity(bddm, init);
    for (elem = support ; elem ; elem = qNext(elem)) {
      cvn decl = qDeclaration(qValue(elem));
      bdd event;
      add_previous(c->model, decl);
      event= model_RepresentationQSignalEvent(c->model, decl);
      resume= bdd_orup2(bddm, event, resume);
      if (qMode(decl) != kIn) {
	stable = bdd_andup2(bddm, stable, bdd_not(bddm, event));
      }
    }
    model_RepresentationAddStable(c->model, stable);
  }
  
   /* . S <= <conditional_waveform> ;
        <suspension>
               ----->
        new value of S
          if <suspension> then S else f(<conditional_waveform>) ;
      . E1 when C1 else          [form a)]
        ...
        En when Cn else
        En+1
               --f-->
        [C1 and E1] or ... or [(Cn and not (Cn-1 or ... C1) and En] 
                    or [not (C1 or ... Cn) and En+1]
      . E1 when C1 else          [form b)]
        ...
        En when Cn
               --f-->
        [C1 and E1] or ... or [Cn and not (Cn-1 or ... C1) and En] 
                    or [not (C1 or ... Cn) and S]
		    */
  {
    bdd conditions= 
      bdd_identity(bddm, bdd_zero(bddm)) ; /* sum of conditions */
    vbdd transition =                   /* zero is bottom for sum */
      vbdd_protect(bddm, vbdd_zero(bddm, vbdd_length(coding))) ;
    cvn wave;
    bdd_value wave_value;
    /* treats all branches but the last */
    for (wave = src ; qNext(wave) ; wave = qNext(wave)) {
      bdd this_condition, condition;
      this_condition= condition_code(qCondition(wave), c->expr_mgr);
      condition= bdd_andup2(bddm, this_condition, bdd_not(bddm, conditions));
      wave_value= expression_value(c->expr_mgr, qValue(wave));
      transition= 
	vbdd_apply2(bdd_orup2,
		    bddm, vbdd_distribute(bdd_and, bddm, condition,
					  wave_value->coding, 1),
		    transition);
      bdd_free(bddm, condition);
      conditions = bdd_orup2(bddm, this_condition, conditions) ;
    }
    /* treats the last branch */
    if (qCondition(wave)) {            /* form a) */
      bdd this_condition, condition ;
      this_condition = condition_code(qCondition(wave), c->expr_mgr);
      condition= bdd_andup2(bddm, this_condition, bdd_not(bddm, conditions));
      wave_value= expression_value(c->expr_mgr, qValue(wave));
      transition = 
	vbdd_apply2(bdd_orup2,
		    bddm, vbdd_distribute(bdd_and, bddm, condition, 
					  wave_value->coding,1),
		    transition);
      bdd_free(bddm, condition);
      conditions = bdd_orup2(bddm, this_condition, conditions) ;
      condition = bdd_notup(bddm, conditions);
      transition =
	vbdd_apply2(bdd_orup2, bddm, 
		    vbdd_distribute(bdd_and, bddm, condition, coding, 1), 
                    transition);
      bdd_free(bddm, condition);
    } else {                        /* form b) */
      wave_value= expression_value(c->expr_mgr, qValue(wave));
      transition =
	vbdd_apply2(bdd_orup2, bddm, 
		    vbdd_distribute(bdd_and, bddm, 
				    bdd_not(bddm, conditions), 
				    wave_value->coding, 1), transition) ;
    }
    transition = vbdd_ite(bddm, resume, transition, coding) ;
    model_RepresentationAddDriver(c->model, decl, stm, transition);
  }
}

/* ------------------------------------------------------------ */
static void
elaborate_selected_signal_assignment_statement
(cvc_context c,
 cvn stm)
{
  cvn tgt, decl, selector, alternatives ;
  vbdd coding, transition;
  bdd resume;
  cvn support = 0 ;

  tgt= qTarget(stm);
  decl= qDeclaration(tgt);
  selector = qSelector(stm);
  coding= model_RepresentationQCoding(c->model, decl);
  alternatives= qAlternatives(stm);
  support = cv_signal_support(stm);

  /* the statement activity is sensitive to each signal that appears in 
     the rhs of the assignment,
     thus check if a variable carrying the previous value exists, if not
     create it */
  {
    cvn elem ; 
    bdd stable;
    bdd init;
    init = condition_code(InitCondition, c->expr_mgr);
    stable = bdd_identity(bddm, bdd_one(bddm));
    resume = bdd_identity(bddm, init);
    for (elem = support ; elem ; elem = qNext(elem)) {
      cvn decl = qDeclaration(qValue(elem));
      bdd event;
      add_previous(c->model, decl);
      event= model_RepresentationQSignalEvent(c->model, decl);
      resume= bdd_orup2(bddm, event, resume);
      if (qMode(decl) != kIn) {
	stable = bdd_andup2(bddm, stable, bdd_not(bddm, event));
      }
    }
    model_RepresentationAddStable(c->model, stable);
  }
  
   /* . S <= <selected_waveform> ;
        <suspension>
               ----->
        new value of S
          if <suspension> then S else f(<conditional_waveform>) ;
      . with E select          [form a)]
          target <= wave1 when choices1,
                    wave2 when choices2,
                    ...
                    waven when choicesn;
               --f-->
          [E in choices1 and wave1] or [E in choices2 and wave2] or...
          [E in choicesn and waven]
      . with E select          [form b)]
          target <= wave1 when choices1,
          target <= wave2 when choices2,
                    ...
                    waven when others;
               --f-->
          [E in choices1 and wave1] or [E in choices2 and wave2] or...
          [not E in (choices1 or choices2 or... choicesn-1) and waven]
		    */
  {
    bdd sum_choices; 
    cvn alternative;
    int has_other_alt;
    transition =
      vbdd_protect(bddm, vbdd_zero(bddm, vbdd_length(coding))) ;
    for (alternative = alternatives;
         alternative && qChoices(alternative);
         alternative = qNext(alternative));
    has_other_alt = alternative ? 1 : 0;
    if (has_other_alt)
      sum_choices = bdd_identity(bddm, bdd_zero(bddm)) ;
    for (alternative = alternatives;
         alternative && qChoices(alternative);
         alternative = qNext(alternative)) {
      bdd choices_cond = bdd_zero(bddm);
      cvn condition;
      cvn choice;
      bdd_value src_value;
      for (choice = qChoices(alternative); choice; choice = qNext(choice)) {
        condition = mBinary(0, boolean_type, kEq, selector, qValue(choice));
	choices_cond = 
	  bdd_orup2(bddm, condition_code(condition, c->expr_mgr), choices_cond);
      }
      src_value= expression_value(c->expr_mgr, qValue(qStatements(alternative)));
      transition = 
	vbdd_apply2(bdd_orup2,
		    bddm, vbdd_distribute(bdd_and, bddm, choices_cond,
					  src_value->coding, 1),
		    transition);
      if (has_other_alt)
	sum_choices = bdd_orup(bddm, choices_cond, sum_choices);
    }
    if (has_other_alt) {
      bdd other_cond;
      bdd_value src_value;
      other_cond= bdd_notup(bddm, sum_choices);
      src_value= expression_value(c->expr_mgr, qValue(qStatements(alternative)));
      transition= 
	vbdd_apply2(bdd_orup2,
		    bddm, vbdd_distribute(bdd_and, bddm, other_cond,
					  src_value->coding, 1),
		    transition);
      bdd_free(bddm, other_cond);
      
    } else {
      bdd_free(bddm, sum_choices);
    }
    transition = vbdd_ite(bddm, resume, transition, coding) ;
  }
  model_RepresentationAddDriver(c->model, decl, stm, transition);
}


