/* ------------------------------------------------------------ */
/* File name:                                                   */
/*    misc.c                                                    */
/*                                                              */
/* Description:                                                 */
/*                                                              */
/* Project:                                                     */
/*    A symbolic model checker for VHDL                         */
/* Subproject:                                                  */
/*    A program that compiles VHDL text descriptions into the   */
/*    internal format.                                          */
/*                                                              */
/* Author:                                                      */
/*    David Deharbe                                             */
/* Affiliation:                                                 */
/*    Carnegie Mellon University (Dept Computer Science)        */
/*                                                              */
/* ------------------------------------------------------------ */

#include <memuser.h>
#include <string.h>
#include "actions.h"

/* ---------------------------------------------- name_kind --- */
/* When a concurrent statement starts with a name, it can be:
      - the target of a signal assignment statement,
      - the label of the statement.
   the following routines are used to disambiguate this. */

static int name_kind ;

int
is_assignment_statement
(void)
{
  return(name_kind) ;
}

#define kind_label 0
#define kind_target 1

void
name_is_label
(void)
{
  name_kind = kind_label ;
}

void
name_is_target
(void)
{
  name_kind = kind_target ;
}

/* ------------------------------------------------- append --- */
/* cvn append (cvn end, cvn list)

      . assumes that end and list are nodes of a class Listed;
      . end is appended to the _end_ of list */

cvn
append
(cvn end,
 cvn list)
{
  cvn current, previous ;
  current = list ;
  if (NullNode(current)) {
    return end;
  } 
  while (current) {
    previous = current ;
    current = qNext(current) ;
  }
  if (previous) sPrevious(end, previous);
  sNext(previous, end) ;

  return list;
}

/* ---------------------------------------- check_end_id --- */
void check_end_id
(char * begin_id,
 char * end_id,
 int * nb_errors,
 tPosition end_pos) 
{
   if (end_id && (strcasecmp(end_id, begin_id) != 0)) {
     char * message;
     message = (char *) mem_get_block((SIZE_T) sizeof(char) *
       (strlen(begin_id) + strlen(end_id) + 1 + 
        strlen("identifier () mismatches identifier ()")));

     sprintf(message, 
             "identifier \042%s\042 mismatches identifier \042%s\042",
             end_id, begin_id) ;
     Message(message, xxError, end_pos) ;
     mem_free_block(message) ;
     if (++*nb_errors > max_nb_errors) exit(MAX_NB_ERRORS_EXIT_CODE);
   }
}

/* ------------------------------------------------------------ */
int has_sensitivity_list;

void
reset_has_sensitivity_list
(void)
{
  has_sensitivity_list = 0;
}

void
set_has_sensitivity_list
(void)
{
  has_sensitivity_list = 1;
}

/* ------------------------------------------------------------ */
void production_not_implemented
(char * name,
 int * nb_errors,
 tPosition pos)
{  
   Message(name, xxError, pos) ;
   if (++*nb_errors > max_nb_errors) exit(MAX_NB_ERRORS_EXIT_CODE);
}

