/* ------------------------------------------------------------ */
/* File name:                                                   */
/*    expression.c                                              */
/*                                                              */
/* Description:                                                 */
/*                                                              */
/* Project:                                                     */
/*    A symbolic model checker for VHDL                         */
/* Subproject:                                                  */
/*    A program that compiles VHDL text descriptions into the   */
/*    internal format.                                          */
/*                                                              */
/* Author:                                                      */
/*    David Deharbe                                             */
/* Affiliation:                                                 */
/*    Carnegie Mellon University (Dept Computer Science)        */
/*                                                              */
/* ------------------------------------------------------------ */

#include <Errors.h>

#include "actions.h"

/* ------------------------------------------------------------ */

#if !defined(ARGS)
#   if defined(__STDC__)
#      define ARGS(args) args
#   else
#      define ARGS(args) ()
#   endif
#endif

/* -------------------------------------------- binary_node --- */

/* cvn binary_node
      (tOperator op, cvn left, cvn right, int * nb_errors, tPosition pos)

      . creates a node of class Binary that represent a binary
      expression;
      . performs semantic checks;
      . tries to recover from errors;
      . returns created node. */

cvn
binary_node
(tOperator op,
 cvn left,
 cvn right,
 int * nb_errors,
 tPosition pos)
{
  if (NullNode(left) || NullNode(right)) {
    ++nb_errors;
    Message("internal error occured, please report us your VHDL code",
	    xxFatal, pos) ;
    return 0;
  }

  switch (op) {
    /* for the relations, the result is boolean, and the arguments
       must be of the same type */
  case kEq : case kNeq : case kGt : case kGe : case kLt : case kLe :
    if (! type_compatible(qSubtype(left), qSubtype(right))) {
            Message("relation between expression of different types",
                    xxError, pos) ;
            if (++*nb_errors > max_nb_errors) exit(MAX_NB_ERRORS_EXIT_CODE);
	    return mBinary(0, qSubtype(left), op, left, right);
    } else
      return(mBinary(0, mSubtype(0, 0, boolean_type, 0), op, left, right)) ;
    /* for the logical operations, the arguments must be of the same
       type and the result type is this type. Whether the type actually
       has an associated "and" operator is not checked, for we only
       accept BIT and BOOLEAN types. */
  case kAnd : case kOr : case kNand : case kNor : case kXor : case kXnor :
    if (! type_compatible(qSubtype(left), qSubtype(right))) {
      Message("incompatible types in logical operation",xxError, pos) ;
      if (++*nb_errors > max_nb_errors) exit(MAX_NB_ERRORS_EXIT_CODE);
    }
    return mBinary(0, qSubtype(left), op, left, right);
    /* for the other operations, issue an error message and return
       a node of the type of the left operand */
  default :
    Message("invalid operator in expression", xxError, pos) ;
    if (++*nb_errors > max_nb_errors) exit(MAX_NB_ERRORS_EXIT_CODE);
    return mBinary(0, qSubtype(left), op, left, right);
  }
}

/* --------------------------------------------- unary_node --- */

/* cvn unary_node
      (tOperator op, cvn arg, int * nb_errors, tPosition pos)

      . creates a node of class Unary that represent a unary
      expression;
      . performs semantic checks;
      . tries to recover from errors;
      . returns created node. */

cvn
unary_node
(tOperator op,
 cvn arg,
 int * nb_errors,
 tPosition pos)
{
  if (arg == 0) {
    Message("internal error occured, please report us your VHDL code", 
            xxFatal, pos) ;
    if (++*nb_errors > max_nb_errors) exit(MAX_NB_ERRORS_EXIT_CODE);
    return 0;
  }
  switch (op) {
    /* for the logical operations, the arguments must be of the same
       type and the result type is this type. Whether the type actually
       has an associated "and" operator is not checked, for we only
       accept BIT and BOOLEAN types. */
  case kNot:
     return mUnary(0, qSubtype(arg), op, arg);
     break ;
     /* for the other operations, issue an error message and return
	a node of the type of the left operand */
   default :
     Message("invalid operator", xxError, pos) ;
     if (++*nb_errors > max_nb_errors) exit(MAX_NB_ERRORS_EXIT_CODE);
     return mUnary(0, qSubtype(arg), op, arg);
     break ;
   }
}

/* -------------------------------------------- is_condition --- */

/* int is_condition (cvn Expression)
   
      . returns a non null value if exp is of type boolean. */

int
is_condition
(cvn exp)
{
   return(type_compatible(qSubtype(exp), boolean_type)) ;
}

/* --------------------------------------- is_locally_static --- */

/* int is_condition (cvn Expression)

      . returns a non null value if exp is of locally static. */

int
is_locally_static
(cvn exp)
{
  switch (qKind(exp)) {
    case kEnumeratedLiteral : 
    case kCharacterLiteral : 
    case kIntegerValue :
      return 1;
    case kUnary :
      return is_locally_static(qOperand(exp));
    case kBinary :
      return is_locally_static(qLeft(exp)) && is_locally_static(qRight(exp));
    default :
      return 0;
  }
}


/* -------------------------------- eval_locally_static_exp --- */

/* cvn eval_locally_static_exp (cvn exp)

      . exp being is a locally static exp, returns the node that 
      represents the constant value this expression is equal to;
      . it is actually divided into two subprograms: 
      eval_unary_exp_ and eval_binary_exp_;
      . actually only works if the result is of type BIT or
      BOOLEAN.
      . the result is not duplicated. */

static cvn eval_unary_exp_ (tOperator, cvn);
static cvn eval_binary_exp_ (tOperator, cvn, cvn);

cvn
eval_locally_static_exp
(cvn exp)
{
  switch (qKind(exp)) {
    case kEnumeratedLiteral : 
    case kCharacterLiteral : 
    case kIntegerValue :
      return exp;
    case kUnary :
      return eval_unary_exp_(qOperator(exp), 
                             eval_locally_static_exp(qOperand(exp)));
    case kBinary :
      return eval_binary_exp_(qOperator(exp), 
                              eval_locally_static_exp(qLeft(exp)),
                              eval_locally_static_exp(qRight(exp)));
    default :
      return 0;
  }
}

cvn
eval_unary_exp_
(tOperator op, cvn arg)
{
  /* only one unary operator, kNot, which applies either to
     a BIT or a BOOLEAN parameter */
  if (qBaseType(qSubtype(arg)) == boolean_type) {
    if (strcmp(qName(arg), "TRUE") == 0)
      return false_literal;
    else
      return true_literal;
  } else {
    if (strcmp(qName(arg), "'1'") == 0)
      return bit0_literal;
    else
      return bit1_literal;
  }
}

cvn
eval_binary_exp_
(tOperator op,
 cvn left,
 cvn right)
{
  switch(op) {
    case kAnd :
      if (qBaseType(qSubtype(left)) == boolean_type) {
        if ((strcmp(qName(left), "TRUE") == 0) &&
            (strcmp(qName(right), "TRUE") == 0)) return true_literal;
        else return false_literal;
      } else {
        if ((strcmp(qName(left), "'1'") == 0) &&
            (strcmp(qName(right), "'1'") == 0)) return bit1_literal;
        else return bit0_literal;
      }
    case kOr :
      if (qBaseType(qSubtype(left)) == boolean_type) {
        if ((strcmp(qName(left), "TRUE") == 0) ||
            (strcmp(qName(right), "TRUE") == 0)) return true_literal;
        else return false_literal;
      } else {
        if ((strcmp(qName(left), "'1'") == 0) ||
            (strcmp(qName(right), "'1'") == 0)) return bit1_literal;
        else return bit0_literal;
      }
    case kNand :
      if (qBaseType(qSubtype(left)) == boolean_type) {
        if ((strcmp(qName(left), "FALSE") == 0) ||
            (strcmp(qName(right), "FALSE") == 0)) return true_literal;
        else return false_literal;
      } else {
        if ((strcmp(qName(left), "'0'") == 0) ||
            (strcmp(qName(right), "'0'") == 0)) return bit1_literal;
        else return bit0_literal;
      }
    case kNor :
      if (qBaseType(qSubtype(left)) == boolean_type) {
        if ((strcmp(qName(left), "FALSE") == 0) &&
            (strcmp(qName(right), "FALSE") == 0)) return true_literal;
        else return false_literal;
      } else {
        if ((strcmp(qName(left), "'0'") == 0) &&
            (strcmp(qName(right), "'0'") == 0)) return bit1_literal;
        else return bit0_literal;
      }
    case kXor :
      if (qBaseType(qSubtype(left)) == boolean_type) {
        if (strcmp(qName(left), qName(right)) != 0) return true_literal;
        else return false_literal;
      } else {
        if (strcmp(qName(left), qName(right)) != 0) return bit1_literal;
        else return bit0_literal;
      }
    case kXnor :
      if (qBaseType(qSubtype(left)) == boolean_type) {
        if (strcmp(qName(left), qName(right)) == 0) return true_literal;
        else return false_literal;
      } else {
        if (strcmp(qName(left), qName(right)) == 0) return bit1_literal;
        else return bit0_literal;
      }
    case kEq :
      if (qOrdinalValue(left) == qOrdinalValue(right)) return true_literal;
      else return false_literal;
    case kNeq :
      if (qOrdinalValue(left) != qOrdinalValue(right)) return true_literal;
      else return false_literal;
    case kGt :
      if (qOrdinalValue(left) > qOrdinalValue(right)) return true_literal;
      else return false_literal;
    case kGe :
      if (qOrdinalValue(left) >= qOrdinalValue(right)) return true_literal;
      else return false_literal;
    case kLt :
      if (qOrdinalValue(left) < qOrdinalValue(right)) return true_literal;
      else return false_literal;
    case kLe :
      if (qOrdinalValue(left) <= qOrdinalValue(right)) return true_literal;
      else return false_literal;
    default :
      return 0;
  }
}
/* ------------------------------------------------------------ */

/* void put_sign(tOperator op, cvn t)
   Put sign on the t->IntegerValue */

cvn
put_sign
(tOperator op,
 cvn t,
 int * nb_errors,
 tPosition position)
{
  if (IsA(t, kIntegerValue)) {
    if (op == kMinus) {
      sIntegerValue(t, qIntegerValue(t) * (-1));
    }
    return t;
  }
  switch (qKind(qBaseType(qSubtype(t)))) {
    case kIntegerType:
      if (op == kMinus) {
        return mUnary(0, qSubtype(t), kMinus, t);
      } else {
        return t;
      }
    default:
      Message("sign not applied to an integer expression", 
              xxError, position);
      if (++*nb_errors > max_nb_errors) exit(MAX_NB_ERRORS_EXIT_CODE);
      return t;
  }
}

