
structure Parser
   :> PARSER
   =
   struct

      exception Error

      open EL

      type pos = int

      fun identity x = x
      fun lift x () = x

      fun null () = []
      fun single x = [x]
      fun double (x, y) = [x, y]

      structure Arg =
         struct

            type string = string
            type int = int
            type char = char

            type string = string
            val ident = identity

            type int = int
            val number = identity

            type longid = string list
            val single_longid = single
            val cons_longid = op ::

            type tp = tp

            fun ident_tp longid =
               (case longid of
                   ["unit"] =>
                      Cprod []
                 | ["bool"] =>
                      Cbool
                 | ["int"] =>
                      Cint
                 | ["char"] =>
                      Cchar
                 | ["string"] =>
                      Cstring
                 | _ =>
                      Cident longid)

            val tp_atom_tp = identity
            val prod_tp = Cprod
            val arrow_tp = Carrow

            type tp_list = tp list
            val double_tp_prod = double
            val cons_tp_prod = op ::

            type dt = (id * tp) list
            val single_dt = single
            fun cons_dt (ident, tp, tail) = (ident, tp) :: tail

            type spec = spec
            val val_spec = Sval
            val type_spec = Stype
            val type_eq_spec = Stypeeq
            val structure_spec = Smodule
            val datatype_spec = Sdata

            type specs = spec list
            val nil_specs = null
            val cons_specs = op ::

            type sg = sg
            val sig_sg = identity

            type pattern = pattern
            val ident_pattern = Pident
            val paren_pattern = identity
            val unit_pattern = lift (Ptuple [])
            val tuple_pattern = Ptuple
            val number_pattern = Pint
            val atom_pattern = identity
            val app_pattern = Papp

            type patterns = pattern list
            val double_patterns = double
            val cons_patterns = op ::

            type term = term
            fun ident_term longid =
               (case longid of
                   ["true"] =>
                      Tbool true
                 | ["false"] =>
                      Tbool false
                 | ["print"] =>
                      Tprim ELPrim.Print
                 | ["Int", "toString"] =>
                      Tprim ELPrim.IntToString
                 | _ =>
                      Tvar longid)
            val paren_term = identity
            val unit_term = lift (Ttuple [])
            val tuple_term = Ttuple
            val let_term = Tlet
            val number_term = Tint
            val string_term = Tstring
            val char_term = Tchar
            val neg_term = lift (Tprim ELPrim.Neg)
            val atom_term = identity
            val app_term = Tapp
            val term_app_term = identity
            fun plus_term (e1, e2) = Tapp (Tprim ELPrim.Plus, Ttuple [e1, e2])
            fun minus_term (e1, e2) = Tapp (Tprim ELPrim.Minus, Ttuple [e1, e2])
            fun times_term (e1, e2) = Tapp (Tprim ELPrim.Times, Ttuple [e1, e2])
            fun concat_term (e1, e2) = Tapp (Tprim ELPrim.Concat, Ttuple [e1, e2])
            val fn_term = Tlam
            val case_term = Tcase

            type terms = term list
            val double_terms = double
            val cons_terms = op ::

            type match = (pattern * term) list
            val single_match = single
            fun cons_match (p, e, tail) = (p, e) :: tail
            
            type module = module
            val ident_module = Mident
            val paren_module = identity
            val struct_module = Mstruct
            fun trans_seal_module (m, sg) = Mseal (m, Transparent, sg)
            fun opaque_seal_module (m, sg) = Mseal (m, Opaque, sg)

            type decl = decl
            val val_decl = Dval
            val fun_decl = Dfun
            val type_decl = Dtype
            val datatype_decl = Ddata
            val structure_decl = Dmodule
            fun trans_structure_decl (ident, sg, module) = Dmodule (ident, Mseal (module, Transparent, sg))
            fun opaque_structure_decl (ident, sg, module) = Dmodule (ident, Mseal (module, Opaque, sg))
            val open_decl = Dopen
            val local_decl = Dlocal

            type decls = decl list
            val nil_decls = null
            val cons_decls = op ::

            datatype terminal = datatype Token.token

            fun error s =
               (case Stream.front s of
                   Stream.Nil =>
                      (
                      print "Syntax error at end of file.\n";
                      Error
                      )
                 | Stream.Cons ((_, pos), _) =>
                      (
                      print "Syntax error at ";
                      print (Int.toString pos);
                      print ".\n";
                      Error
                      ))
         end

      structure StreamWithPos =
         CoercedStreamable (structure Streamable = StreamStreamable
                            type 'a item = 'a * pos
                            fun coerce (x, _) = x)

      structure ParseMain =
         ParseMainFun
         (structure Streamable = StreamWithPos
          structure Arg = Arg)

      fun parse s =
         #1 (ParseMain.parse (Lexer.lex s))

      fun parseFile fname =
         let
            val ins = TextIO.openIn fname
            val decls = parse (Stream.fromTextInstream ins)
            val () = TextIO.closeIn ins
         in
            decls
         end

   end
