
structure InspectModule :> INSPECT_MODULE =
   struct

      open InspectionValue
      open ILModule

      exception TagKind of kind
      exception TagCon of con
      exception TagName of name
      exception TagSg of sg
      exception TagTerm of term
      exception TagModule of module
      exception TagPrim of Prim.prim

      val TK = Abs o TagKind
      val TC = Abs o TagCon
      val TN = Abs o TagName
      val TS = Abs o TagSg
      val TT = Abs o TagTerm
      val TM = Abs o TagModule

      fun Data0 str = Data (str, [])

      val nope = ref false  (* don't modify this *)
      val suppressCon = ref false

      fun kindTool ex =
         (case ex of
             TagKind k =>
                SOME
                ((case k of
                     Ktype =>
                        Data0 "Ktype"
                   | Ksing c =>
                        Data ("Ksing", [TC c])
                   | Kpi (k1, k2) =>
                        Data ("Kpi", [TK k1, TK k2])
                   | Ksigma (k1, k2) =>
                        Data ("Ksigma", [TK k1, TK k2])
                   | Kunit =>
                        Data0 "Kunit"),
                 "kind", nope)
           | _ =>
                NONE)

      fun conTool ex =
         (case ex of
             TagCon c =>
                SOME
                ((case c of
                     Cvar (i, jo) =>
                        Data ("Cvar", [Int i, Option (Option.map Int jo)])
                   | Clam (k, c) =>
                        Data ("Clam", [TK k, TC c])
                   | Capp (c1, c2) =>
                        Data ("Capp", [TC c1, TC c2])
                   | Cpair (c1, c2) =>
                        Data ("Cpair", [TC c1, TC c2])
                   | Cpi1 c =>
                        Data ("Cpi1", [TC c])
                   | Cpi2 c =>
                        Data ("Cpi2", [TC c])
                   | Cunit =>
                        Data0 "Cunit"
                   | Carrow (c1, c2) =>
                        Data ("Carrow", [TC c1, TC c2])
                   | Cprod cl =>
                        Data ("Cprod", [List (map TC cl)])
                   | Csum cl =>
                        Data ("Csum", [List (map TC cl)])
                   | Crec c =>
                        Data ("Crec", [TC c])
                   | Ctag c =>
                        Data ("Ctag", [TC c])
                   | Cref c =>
                        Data ("Cref", [TC c])
                   | Cexn =>
                        Data0 "Cexn"
                   | Cbool =>
                        Data0 "Cbool"
                   | Cint =>
                        Data0 "Cint"
                   | Cchar =>
                        Data0 "Cchar"
                   | Cstring =>
                        Data0 "Cstring"),
                 "con", suppressCon)
           | _ =>
                NONE)

      fun nameTool ex =
         (case ex of
             TagName name =>
                SOME
                ((case name of
                     HIDE =>
                        Data0 "HIDE"
                   | VAL str =>
                        Data ("VAL", [String str])
                   | CON str =>
                        Data ("CON", [String str])
                   | MOD str =>
                        Data ("MOD", [String str])
                   | DT str =>
                        Data ("DT", [String str])
                   | DCON =>
                        Data0 "DCON"),
                 "name", nope)
           | _ =>
                NONE)

      fun sgTool ex =
         (case ex of
             TagSg sg =>
                SOME
                ((case sg of
                     Sval c =>
                        Data ("Sval", [TC c])
                   | Scon k =>
                        Data ("Scon", [TK k])
                   | Ssigma (sg1, sg2) =>
                        Data ("Ssigma", [TS sg1, TS sg2])
                   | Spi (sg1, sg2) =>
                        Data ("Spi", [TS sg1, TS sg2])
                   | Sunit =>
                        Data0 "Sunit"
                   | Snamed (name, sg) =>
                        Data ("Snamed", [TN name, TS sg])),
                 "sg", nope)
           | _ =>
                NONE)

      fun termTool ex =
         (case ex of
             TagTerm e =>
                SOME
                ((case e of
                     Tvar v =>
                        Data ("Tvar", [Int v])
                   | Tlam (v, c, e) =>
                        Data ("Tlam", [Int v, TC c, TT e])
                   | Tapp (e1, e2) =>
                        Data ("Tapp", [TT e1, TT e2])
                   | Ttuple el =>
                        Data ("Ttuple", [List (map TT el)])
                   | Tproj (e, i) =>
                        Data ("Tproj", [TT e, Int i])
                   | Tinj (e, i, c) =>
                        Data ("Tinj", [TT e, Int i, TC c])
                   | Tcase (e, match) =>
                        Data ("Tcase", [TT e,
                                        List (map
                                                 (fn (vi, ei) =>
                                                     Tuple [Int vi, TT ei])
                                                 match)])
                   | Troll (e, c) =>
                        Data ("Troll", [TT e, TC c])
                   | Tunroll e =>
                        Data ("Tunroll", [TT e])
                   | Ttag (e1, e2) =>
                        Data ("Ttag", [TT e1, TT e2])
                   | Tiftag (e1, e2, v, e3, e4) =>
                        Data ("Tiftag", [TT e1, TT e2, Int v, TT e3, TT e4])
                   | Tnewtag c =>
                        Data ("Tnewtag", [TC c])
                   | Traise (e, c) =>
                        Data ("Traise", [TT e, TC c])
                   | Thandle (e1, v, e2) =>
                        Data ("Thandle", [TT e1, Int v, TT e2])
                   | Tref e =>
                        Data ("Tref", [TT e])
                   | Tderef e =>
                        Data ("Tderef", [TT e])
                   | Tassign (e1, e2) =>
                        Data ("Tassign", [TT e1, TT e2])
                   | Tbool b =>
                        Data ("Tbool", [Bool b])
                   | Tif (e1, e2, e3) =>
                        Data ("Tif", [TT e1, TT e2, TT e3])
                   | Tint i =>
                        Data ("Tint", [Int i])
                   | Tchar ch =>
                        Data ("Tchar", [Char ch])
                   | Tstring str =>
                        Data ("Tstring", [String str])
                   | Tlet (v, e1, e2) =>
                        Data ("Tlet", [Int v, TT e1, TT e2])
                   | Tletm (v, m, e, c) =>
                        Data ("Tletm", [Int v, TM m, TT e, TC c])
                   | Tprim (prim, el) =>
                        Data ("Tprim", [Abs (TagPrim prim), List (map TT el)])
                   | Tsnd m =>
                        Data ("Tsnd", [TM m])),
                 "term", nope)
           | _ =>
                NONE)

      fun moduleTool ex =
         (case ex of
             TagModule m =>
                SOME
                ((case m of
                     Mvar v =>
                        Data ("Mvar", [Int v])
                   | Mval e =>
                        Data ("Mval", [TT e])
                   | Mcon c =>
                        Data ("Mcon", [TC c])
                   | Munit =>
                        Data0 "Munit"
                   | Mpair (m1, m2) =>
                        Data ("Mpair", [TM m1, TM m2])
                   | Mdpair (v, m1, m2) =>
                        Data ("Mdpair", [Int v, TM m1, TM m2])
                   | Mpi1 m =>
                        Data ("Mpi1", [TM m])
                   | Mpi2 m =>
                        Data ("Mpi2", [TM m])
                   | Mlam (v, sg, m) =>
                        Data ("Mlam", [Int v, TS sg, TM m])
                   | Mapp (m1, m2) =>
                        Data ("Mapp", [TM m1, TM m2])
                   | Min (name, m) =>
                        Data ("Min", [TN name, TM m])
                   | Mout m =>
                        Data ("Mout", [TM m])
                   | Mlet (v, m1, m2, sg) =>
                        Data ("Mlet", [Int v, TM m1, TM m2, TS sg])
                   | Mletd (v, m1, m2) =>
                        Data ("Mletd", [Int v, TM m1, TM m2])
                   | Mlete (v, e, m) =>
                        Data ("Mlete", [Int v, TT e, TM m])
                   | Mseal (m, sg) =>
                        Data ("Mseal", [TM m, TS sg])),
                 "module", nope)
           | _ =>
                NONE)

      val tools = [kindTool, conTool, nameTool, sgTool, termTool, moduleTool]

      structure Inspect = InspectFun (val tools = tools)
      open Inspect

      val loadKind = Inspect.load o TagKind
      val loadCon = Inspect.load o TagCon
      val loadName = Inspect.load o TagName
      val loadSg = Inspect.load o TagSg
      val loadTerm = Inspect.load o TagTerm
      val loadModule = Inspect.load o TagModule

      fun extractKind () =
         (case Inspect.extract () of
             TagKind x => x
           | _ => raise Inspect.Extract)

      fun extractCon () =
         (case Inspect.extract () of
             TagCon x => x
           | _ => raise Inspect.Extract)

      fun extractName () =
         (case Inspect.extract () of
             TagName x => x
           | _ => raise Inspect.Extract)

      fun extractSg () =
         (case Inspect.extract () of
             TagSg x => x
           | _ => raise Inspect.Extract)

      fun extractTerm () =
         (case Inspect.extract () of
             TagTerm x => x
           | _ => raise Inspect.Extract)

      fun extractModule () =
         (case Inspect.extract () of
             TagModule x => x
           | _ => raise Inspect.Extract)

   end

