
structure HashCons :> HASH_CONS =
   struct

      type label = word
      type tag = word

      datatype desc =
         D0 of label
       | D1 of label * tag
       | D2 of label * tag * tag
       | D3 of label * tag * tag * tag
       | Dn of label * tag list

      structure DescHashable =
         struct
            type t = desc
            
            fun eq d1_d2 =
               (case d1_d2 of
                   (D0 lab, D0 lab') =>
                      lab = lab'
                 | (D1 (lab, tag), D1 (lab', tag')) =>
                      lab = lab' andalso tag = tag'
                 | (D2 (lab, tag1, tag2), D2 (lab', tag1', tag2')) =>
                      lab = lab' andalso tag1 = tag1' andalso tag2 = tag2'
                 | (D3 (lab, tag1, tag2, tag3), D3 (lab', tag1', tag2', tag3')) =>
                      lab = lab' andalso tag1 = tag1' andalso tag2 = tag2' andalso tag3 = tag3'
                 | (Dn (lab, l), Dn (lab', l')) =>
                      lab = lab'
                      andalso
                      ListPair.allEq (op =) (l, l')
                 | _ => false)

            val hashInc = MJHash.hashInc

            fun hash d =
               (case d of
                   D0 lab => lab
                 | D1 (lab, tag) =>
                      hashInc lab tag
                 | D2 (lab, tag1, tag2) =>
                      hashInc (hashInc lab tag1) tag2
                 | D3 (lab, tag1, tag2, tag3) =>
                      hashInc (hashInc (hashInc lab tag1) tag2) tag3
                 | Dn (lab, l) =>
                      foldl (fn (tag, h) => hashInc h tag) lab l)
         end

      structure Table = HashTable (structure Key = DescHashable)
      structure StringTable = HashTable (structure Key = StringHashable)

      val initialSize = 10007

      type 'a object = 'a * tag

      type 'a table = 'a object Table.table

      fun table () = Table.table initialSize

      fun destroyTable tb = Table.reset tb initialSize

      val nextTag = ref 0w1

      fun newTag () =
         let
            val tag = !nextTag
         in
            if tag = 0w0 then
               raise Overflow
            else
               (
               nextTag := tag + 0w1;
               tag
               )
         end

      fun cons0 table lab c =
         Table.lookupOrInsert
            table (D0 lab)
            (fn () => (c, newTag ()))

      fun cons1 table lab c (v, tag) =
         Table.lookupOrInsert
            table (D1 (lab, tag))
            (fn () => (c v, newTag ()))

      fun cons2 table lab c ((v1, tag1), (v2, tag2)) =
         Table.lookupOrInsert
            table (D2 (lab, tag1, tag2))
            (fn () => (c (v1, v2), newTag ()))

      fun cons3 table lab c ((v1, tag1), (v2, tag2), (v3, tag3)) =
         Table.lookupOrInsert
            table (D3 (lab, tag1, tag2, tag3))
            (fn () => (c (v1, v2, v3), newTag ()))

      fun consList table lab c l =
         let
            val (vl, tagl) = ListPair.unzip l
         in
            Table.lookupOrInsert
               table (Dn (lab, tagl))
               (fn () => (c vl, newTag ()))
         end



      fun consInt i = (i, Word.fromInt i)

      fun consChar ch = (ch, Word.fromInt (Char.ord ch))

      val stringTable : string object StringTable.table = StringTable.table initialSize

      fun consString str =
         StringTable.lookupOrInsert
            stringTable str
            (fn () => (str, newTag ()))

      fun destroyStringTable () = StringTable.reset stringTable 1

   end
