
structure InterpretAlloc :> INTERPRET_ALLOC =
   struct
      open Variable
      open ILAlloc
      open ResultAlloc

      structure PrimEval =
         PrimEvalFun (structure Param =
                         struct
                            open ResultAlloc

                            val Runit = Rptr (Array.array (0, Rjunk))

                            fun destRbool k =
                                (case k of
                                    Rbool x => SOME x
                                  | _ => NONE)
                                
                            fun destRint k =
                                (case k of
                                    Rint x => SOME x
                                  | _ => NONE)
                                
                            fun destRchar k =
                                (case k of
                                    Rchar x => SOME x
                                  | _ => NONE)

                            fun destRstring k =
                                (case k of
                                    Rstring x => SOME x
                                  | _ => NONE)
                         end)

      structure D = VariableDict

      type env = result D.dict

      fun eval env v =
          (case v of
              Vvar x =>
                 (case D.find env x of
                     SOME r =>
                        r
                   | NONE =>
                        raise Wrong)
            | Vbool b =>
                 Rbool b
            | Vint i =>
                 Rint i
            | Vchar ch =>
                 Rchar ch)

      fun perform env oper =
          (case oper of
              Olet (x, v) =>
                 D.insert env x (eval env v)
            | Oalloc (x, i) =>
                 D.insert env x (Rptr (Array.array (i, Rjunk)))
            | Oread (x, y, i) =>
                 (case D.find env y of
                     SOME (Rptr a) =>
                        (D.insert env x (Array.sub (a, i))
                         handle Subscript => raise Wrong)
                   | _ =>
                        raise Wrong)
            | Owrite (y, i, v) =>
                 let val r = eval env v
                 in
                    (case D.find env y of
                        SOME (Rptr a) =>
                           ((
                            Array.update (a, i, r);
                            env
                            ) handle Subscript => raise Wrong)
                      | _ =>
                           raise Wrong)
                 end
            | Ostring (x, str) =>
                 D.insert env x (Rstring str)
            | Oprim (x, prim, vl) =>
                 let val rl = map (fn v => eval env v) vl
                 in
                    D.insert env x (PrimEval.primeval prim rl)
                 end
            | Optreq (x, y, z) =>
                 (case (D.find env y, D.find env z) of
                     (SOME (Rptr a), SOME (Rptr b)) =>
                        D.insert env x (Rbool (a = b))
                   | _ =>
                        raise Wrong))

      fun execExp env e =
          (case e of
              Edo (ops, e') =>
                 let val env' =
                        foldl (fn (oper, env) => perform env oper) env ops
                 in
                    execExp env' e'
                 end
            | Eapp (x, v) =>
                 (case D.find env x of
                     SOME (Rfn f) =>
                        f (eval env v)
                   | _ =>
                        raise Wrong)
            | Ecase (x, l) =>
                 (case D.find env x of
                     SOME (Rint i) =>
                        (execExp env (List.nth (l, i))
                         handle Subscript => raise Wrong)
                   | _ =>
                        raise Wrong)
            | Eif (x, e1, e2) =>
                 (case D.find env x of
                     SOME (Rbool b) =>
                        if b then
                           execExp env e1
                        else
                           execExp env e2
                   | _ =>
                        raise Wrong)
            | Ehalt =>
                 ())

      fun execProgram env p =
          (case p of
              Pbody e =>
                 execExp env e
            | Pfun (f, x, e, p) =>
                 execProgram (D.insert env f (Rfn (fn r => execExp (D.insert env x r) e))) p)

   end
