
structure InterpretClosure :> INTERPRET_CLOSURE =
   struct

      open ILClosure
      open ResultClosure

      structure D = VariableDict

      type env = result D.dict

      structure PrimEval =
         PrimEvalFun
         (structure Param =
             struct
                open ResultClosure

                val Runit = Rtuple []

                fun destRbool k =
                    (case k of
                        Rbool x => SOME x
                      | _ => NONE)
          
                fun destRint k =
                    (case k of
                        Rint x => SOME x
                      | _ => NONE)
          
                fun destRchar k =
                    (case k of
                        Rchar x => SOME x
                      | _ => NONE)
          
                fun destRstring k =
                    (case k of
                        Rstring x => SOME x
                      | _ => NONE)
             end)

      fun exec env e =
         (case e of
             Eapp (v1, v2) =>
                (case eval env v1 of
                    Rcont f =>
                       f (eval env v2)
                  | _ =>
                       raise Wrong)

           | Eunpack (x, v, e) =>
                (case eval env v of
                    Rpack r =>
                       exec (D.insert env x r) e
                  | _ =>
                       raise Wrong)

           | Eproj (x, v, i, e) =>
                (case eval env v of
                    Rtuple rl =>
                       let
                          val r =
                             List.nth (rl, i)
                             handle Subscript => raise Wrong
                       in
                          exec (D.insert env x r) e
                       end
                  | _ =>
                       raise Wrong)

           | Ecase (v, arms) =>
                (case eval env v of
                    Rinj (r, i) =>
                       let
                          val (xi, ei) =
                             List.nth (arms, i)
                             handle Subscript => raise Wrong
                       in
                          exec (D.insert env xi r) ei
                       end
                  | _ =>
                       raise Wrong)

           | Eiftag (v1, v2, x, e1, e2) =>
                (case (eval env v1, eval env v2) of
                    (Rtag tag, Rexn (tag', r)) =>
                       if tag = tag' then
                          exec (D.insert env x r) e1
                       else
                          exec env e2
                  | _ =>
                       raise Wrong)

           | Enewtag (x, _, e) =>
                exec (D.insert env x (Rtag (ref ()))) e

           | Eref (x, v, e) =>
                exec (D.insert env x (Rref (ref (eval env v)))) e

           | Ederef (x, v, e) =>
                (case eval env v of
                    Rref (ref r) =>
                       exec (D.insert env x r) e
                  | _ =>
                       raise Wrong)

           | Eassign (v1, v2, e) =>
                (case eval env v1 of
                    Rref rr =>
                       (
                       rr := eval env v2;
                       exec env e
                       )
                  | _ =>
                       raise Wrong)

           | Eif (v, e1, e2) =>
                (case eval env v of
                    Rbool true =>
                       exec env e1
                  | Rbool false =>
                       exec env e2
                  | _ =>
                       raise Wrong)

           | Elet (x, v, e) =>
                exec (D.insert env x (eval env v)) e

           | Eprim (x, prim, vl, e) =>
                let
                   val r =
                      PrimEval.primeval prim (map (eval env) vl)
                in
                   exec (D.insert env x r) e
                end

           | Ehalt => ())

      and eval env v =
         (case v of
             Vvar x =>
                (case D.find env x of
                    SOME r => r
                  | NONE =>
                       raise Wrong)

           | Vlam (x, _, e) =>
                Rcont (fn r => exec (D.insert D.empty x r) e)

           | Vpack (_, v, _) =>
                Rpack (eval env v)

           | Vtuple vl =>
                Rtuple (map (eval env) vl)

           | Vinj (v, i, _) =>
                Rinj (eval env v, i)

           | Vroll (v, _) =>
                Rroll (eval env v)

           | Vunroll v =>
                (case eval env v of
                    Rroll r => r
                  | _ =>
                       raise Wrong)

           | Vtag (v1, v2) =>
                (case eval env v1 of
                    Rtag tag =>
                       Rexn (tag, eval env v2)
                  | _ =>
                       raise Wrong)

           | Vbool b =>
                Rbool b

           | Vint i =>
                Rint i

           | Vchar ch =>
                Rchar ch

           | Vstring str =>
                Rstring str)

   end
