
structure DebugClosure :> DEBUG_CLOSURE =
   struct

      open ILClosure

      exception IndexError of int list

      exception IndexErrorKind of kind * int option * int list
      exception IndexErrorCon of con * int option * int list
      exception IndexErrorExp of exp * int option * int list
      exception IndexErrorValue of value * int option * int list


      val bottom = Depth.bottom
      fun dec path d = Depth.dec (IndexError path) d
      fun join path d1_d2 = Depth.join (IndexError path) d1_d2

      fun mapi f l =
         rev (#1 (foldl (fn (x, (l, i)) => (f (x, i) :: l, i+1)) ([], 0) l))

      fun depthKind path k =
         (case k of
             Ktype => bottom
           | Ksing c =>
                depthCon (0 :: path) c
           | Kpi (k1, k2) =>
                join path (depthKind (0 :: path) k1, dec path (depthKind (1 :: path) k2))
           | Ksigma (k1, k2) =>
                join path (depthKind (0 :: path) k1, dec path (depthKind (1 :: path) k2))
           | Kunit => bottom)

      and depthCon path c =
         (case c of
             Cvar (i, NONE) =>
                Depth.AtLeast i
           | Cvar (i, SOME j) =>
                if i < j then
                   Depth.Exactly j
                else
                   raise (IndexError (rev path))
           | Clam (k, c) =>
                join path (depthKind (0 :: path) k, dec path (depthCon (1 :: path) c))
           | Capp (c1, c2) =>
                join path (depthCon (0 :: path) c1, depthCon (1 :: path) c2)
           | Cpair (c1, c2) =>
                join path (depthCon (0 :: path) c1, depthCon (1 :: path) c2)
           | Cpi1 c =>
                depthCon (0 :: path) c
           | Cpi2 c =>
                depthCon (0 :: path) c
           | Cunit => bottom
           | Cnot c =>
                depthCon (0 :: path) c
           | Cexists (k, c) =>
                join path (depthKind (0 :: path) k, dec path (depthCon (1 :: path) c))
           | Cprod cl =>
                #1 (foldl (fn (c, (d, i)) => (join path (depthCon (i :: 0 :: path) c, d), i+1)) (bottom, 0) cl)
           | Csum cl =>
                #1 (foldl (fn (c, (d, i)) => (join path (depthCon (i :: 0 :: path) c, d), i+1)) (bottom, 0) cl)
           | Crec c =>
                dec path (depthCon (0 :: path) c)
           | Ctag c =>
                depthCon (0 :: path) c
           | Cref c =>
                depthCon (0 :: path) c
           | Cexn => bottom
           | Cbool => bottom
           | Cint => bottom
           | Cchar => bottom
           | Cstring => bottom)

      fun checkKind k =
         (
         ((depthKind [] k) handle IndexError path => raise IndexErrorKind (k, NONE, path));
         ()
         )

      fun checkCon c =
         (
         ((depthCon [] c) handle IndexError path => raise IndexErrorCon (c, NONE, path));
         ()
         )

      fun imposeKindMain path n k =
         (case k of
             Ktype => k
           | Ksing c =>
                Ksing (imposeConMain (0 :: path) n c)
           | Kpi (k1, k2) =>
                Kpi (imposeKindMain (0 :: path) n k1, imposeKindMain (1 :: path) (n+1) k2)
           | Ksigma (k1, k2) =>
                Ksigma (imposeKindMain (0 :: path) n k1, imposeKindMain (1 :: path) (n+1) k2)
           | Kunit => k)

      and imposeConMain path n c =
         (case c of
             Cvar (i, NONE) =>
                if i < n then
                   Cvar (i, SOME n)
                else
                   raise (IndexError (rev path))
           | Cvar (i, SOME j) =>
                if n = j andalso i < j then
                   c
                else
                   raise (IndexError (rev path))
           | Clam (k, c) =>
                Clam (imposeKindMain (0 :: path) n k, imposeConMain (1 :: path) (n+1) c)
           | Capp (c1, c2) =>
                Capp (imposeConMain (0 :: path) n c1, imposeConMain (1 :: path) n c2)
           | Cpair (c1, c2) =>
                Cpair (imposeConMain (0 :: path) n c1, imposeConMain (1 :: path) n c2)
           | Cpi1 c =>
                Cpi1 (imposeConMain (0 :: path) n c)
           | Cpi2 c =>
                Cpi2 (imposeConMain (0 :: path) n c)
           | Cunit => c
           | Cnot c =>
                Cnot (imposeConMain (0 :: path) n c)
           | Cexists (k, c) =>
                Cexists (imposeKindMain (0 :: path) n k, imposeConMain (1 :: path) (n+1) c)
           | Cprod cl =>
                Cprod (mapi (fn (c, i) => imposeConMain (i :: 0 :: path) n c) cl)
           | Csum cl =>
                Csum (mapi (fn (c, i) => imposeConMain (i :: 0 :: path) n c) cl)
           | Crec c =>
                Crec (imposeConMain (0 :: path) (n+1) c)
           | Ctag c =>
                Ctag (imposeConMain (0 :: path) n c)
           | Cref c =>
                Cref (imposeConMain (0 :: path) n c)
           | Cexn => c
           | Cbool => c
           | Cint => c
           | Cchar => c
           | Cstring => c)

      fun imposeExpMain path n e =
         (case e of
             Eapp (v1, v2) =>
                Eapp (imposeValueMain (0 :: path) n v1, imposeValueMain (1 :: path) n v2)
           | Eunpack (x, v, e) =>
                Eunpack (x, imposeValueMain (1 :: path) n v, imposeExpMain (2 :: path) (n+1) e)
           | Eproj (x, v, i, e) =>
                Eproj (x, imposeValueMain (1 :: path) n v, i, imposeExpMain (3 :: path) n e)
           | Ecase (v, arms) =>
                Ecase (imposeValueMain (0 :: path) n v, mapi (fn ((xi, ei), i) => (xi, imposeExpMain (1 :: i :: 1 :: path) n ei)) arms)
           | Eiftag (v1, v2, x, e1, e2) =>
                Eiftag (imposeValueMain (0 :: path) n v1, imposeValueMain (1 :: path) n v2, x, imposeExpMain (3 :: path) n e1, imposeExpMain (4 :: path) n e2)
           | Enewtag (x, t, e) =>
                Enewtag (x, imposeConMain (1 :: path) n t, imposeExpMain (2 :: path) n e)
           | Eref (x, v, e) =>
                Eref (x, imposeValueMain (1 :: path) n v, imposeExpMain (2 :: path) n e)
           | Ederef (x, v, e) =>
                Ederef (x, imposeValueMain (1 :: path) n v, imposeExpMain (2 :: path) n e)
           | Eassign (v1, v2, e) =>
                Eassign (imposeValueMain (0 :: path) n v1, imposeValueMain (1 :: path) n v2, imposeExpMain (2 :: path) n e)
           | Eif (v, e1, e2) =>
                Eif (imposeValueMain (0 :: path) n v, imposeExpMain (1 :: path) n e1, imposeExpMain (2 :: path) n e2)
           | Elet (x, v, e) =>
                Elet (x, imposeValueMain (1 :: path) n v, imposeExpMain (2 :: path) n e)
           | Eprim (x, prim, vl, e) =>
                Eprim (x, prim, mapi (fn (e, i) => imposeValueMain (i :: 2 :: path) n e) vl, imposeExpMain (3 :: path) n e)
           | Ehalt =>
                e)

      and imposeValueMain path n v =
         (case v of
             Vvar _ =>
                v
           | Vlam (x, c, e) =>
                Vlam (x, imposeConMain (1 :: path) n c, imposeExpMain (2 :: path) n e)
           | Vpack (c1, v', c2) =>
                Vpack (imposeConMain (0 :: path) n c1, imposeValueMain (1 :: path) n v', imposeConMain (2 :: path) n c2)
           | Vtuple vl =>
                Vtuple (mapi (fn (v, i) => imposeValueMain (i :: 0 :: path) n v) vl)
           | Vinj (v', i, c) =>
                Vinj (imposeValueMain (0 :: path) n v', i, imposeConMain (2 :: path) n c)
           | Vroll (v', c) =>
                Vroll (imposeValueMain (0 :: path) n v', imposeConMain (1 :: path) n c)
           | Vunroll v' =>
                Vunroll (imposeValueMain (0 :: path) n v')
           | Vtag (v1, v2) =>
                Vtag (imposeValueMain (0 :: path) n v1, imposeValueMain (1 :: path) n v2)
           | Vbool _ => v
           | Vint _ => v
           | Vchar _ => v
           | Vstring _ => v)

      fun imposeKind n k =
         ((imposeKindMain [] n k) handle IndexError path => raise IndexErrorKind (k, SOME n, path))

      fun imposeCon n c =
         ((imposeConMain [] n c) handle IndexError path => raise IndexErrorCon (c, SOME n, path))

      fun imposeExp n e =
         ((imposeExpMain [] n e) handle IndexError path => raise IndexErrorExp (e, SOME n, path))

      fun imposeValue n e =
         ((imposeValueMain [] n e) handle IndexError path => raise IndexErrorValue (e, SOME n, path))

   end
