
structure CheckCps :> CHECK_CPS =
   struct

      open ILCps
      open SubstCps
      open ContextCps
      open EquivCps

      exception TypeError = Misc.TypeError

      fun checkKind ctx k =
         (case k of
             Ktype => ()

           | Ksing c =>
                checkCon ctx c Ktype

           | Kpi (k1, k2) =>
                (
                checkKind ctx k1;
                checkKind (extendKind ctx k1) k2
                )

           | Ksigma (k1, k2) =>
                (
                checkKind ctx k1;
                checkKind (extendKind ctx k1) k2
                )

           | Kunit => ())

      and inferCon ctx c =
         (case c of
             Cvar (i, _) =>
                selfify c (lookupKind ctx i)

           | Clam (k, c) =>
                (
                checkKind ctx k;
                Kpi (k, inferCon (extendKind ctx k) c)
                )

           | Capp (c1, c2) =>
                (case inferCon ctx c1 of
                    Kpi (dom, cod) =>
                       (
                       checkCon ctx c2 dom;
                       substKind c2 cod
                       )
                  | _ =>
                       raise TypeError)

           | Cpair (c1, c2) =>
                Ksigma (inferCon ctx c1, liftKind 1 (inferCon ctx c2))

           | Cpi1 c' =>
                (case inferCon ctx c' of
                    Ksigma (k1, k2) =>
                       k1
                  | _ =>
                       raise TypeError)

           | Cpi2 c' =>
                (case inferCon ctx c' of
                    Ksigma (k1, k2) =>
                       substKind (Cpi1 c') k2
                  | _ =>
                       raise TypeError)

           | Cunit => Kunit

           | Cnot c' =>
                (
                checkCon ctx c' Ktype;
                Ksing c
                )
            
           | Cexists (k, c') =>
                (
                checkKind ctx k;
                checkCon (extendKind ctx k) c' Ktype;
                Ksing c
                )

           | Cprod cl =>
                (
                List.app (fn c' => checkCon ctx c' Ktype) cl;
                Ksing c
                )

           | Csum cl =>
                (
                List.app (fn c' => checkCon ctx c' Ktype) cl;
                Ksing c
                )

           | Crec c' =>
                (
                checkCon (extendKind ctx Ktype) c' Ktype;
                Ksing c
                )

           | Ctag c' =>
                (
                checkCon ctx c' Ktype;
                Ksing c
                )

           | Cref c' =>
                (
                checkCon ctx c' Ktype;
                Ksing c
                )

           | Cexn => Ksing Cexn
           | Cbool => Ksing Cbool
           | Cint => Ksing Cint
           | Cchar => Ksing Cchar
           | Cstring => Ksing Cstring)

      and checkCon ctx c k =
         subkind ctx (inferCon ctx c) k

      fun whnfAnnot ctx t =
         (
         checkCon ctx t Ktype;
         whnf ctx t
         )

      structure PrimType =
         PrimTypeFun (structure Param =
                         struct
                            open ILCps
                            val Cunittype = Cprod []
                         end)

      fun checkExp ctx e =
         (case e of
             Eapp (v1, v2) =>
                (case inferValueWhnf ctx v1 of
                    Cnot dom =>
                       checkValue ctx v2 dom
                  | _ =>
                       raise TypeError)

           | Eunpack (x, v, e) =>
                (case inferValueWhnf ctx v of
                    Cexists (k, t) =>
                       checkExp (extendType (extendKind ctx k) x t) e
                  | _ =>
                       raise TypeError)

           | Eproj (x, v, i, e) =>
                (case inferValueWhnf ctx v of
                    Cprod tl =>
                       let
                          val t = 
                             List.nth (tl, i)
                             handle Subscript => raise TypeError
                       in
                          checkExp (extendType ctx x t) e
                       end
                  | _ =>
                       raise TypeError)

           | Ecase (v, arms) =>
                (case inferValueWhnf ctx v of
                    Csum tl =>
                       (ListPair.appEq
                           (fn ((xi, ei), ti) =>
                               checkExp (extendType ctx xi ti) ei)
                           (arms, tl)
                        handle ListPair.UnequalLengths => raise TypeError)
                  | _ =>
                       raise TypeError)

           | Eiftag (v1, v2, x, e1, e2) =>
                (case inferValueWhnf ctx v1 of
                    Ctag t =>
                       (
                       checkValue ctx v2 Cexn;
                       checkExp (extendType ctx x t) e1;
                       checkExp ctx e2
                       )
                  | _ =>
                       raise TypeError)

           | Enewtag (x, t, e) =>
                (
                checkCon ctx t Ktype;
                checkExp (extendType ctx x (Ctag t)) e
                )

           | Eref (x, v, e) =>
                let
                   val t = inferValue ctx v
                in
                   checkExp (extendType ctx x (Cref t)) e
                end

           | Ederef (x, v, e) =>
                (case inferValueWhnf ctx v of
                    Cref t =>
                       checkExp (extendType ctx x t) e
                  | _ =>
                       raise TypeError)

           | Eassign (v1, v2, e) =>
                (case inferValueWhnf ctx v1 of
                    Cref t =>
                       (
                       checkValue ctx v2 t;
                       checkExp ctx e
                       )
                  | _ =>
                       raise TypeError)

           | Eif (v, e1, e2) =>
                (
                checkValue ctx v Cbool;
                checkExp ctx e1;
                checkExp ctx e2
                )

           | Elet (x, v, e) =>
                let
                   val t = inferValue ctx v
                in
                   checkExp (extendType ctx x t) e
                end

           | Eprim (x, prim, vl, e) =>
                let
                   val (tl, t) = PrimType.primtype prim
                in
                   (ListPair.appEq
                       (fn (vi, ti) => checkValue ctx vi ti)
                       (vl, tl)
                    handle ListPair.UnequalLengths => raise TypeError);
                   checkExp (extendType ctx x t) e
                end

           | Ehalt => ())
                   
      and inferValue ctx v =
         (case v of
             Vvar x =>
                lookupType ctx x

           | Vlam (x, dom, e) =>
                (
                checkCon ctx dom Ktype;
                checkExp (extendType ctx x dom) e;
                Cnot dom
                )

           | Vpack (c, v, annot) =>
                (case whnfAnnot ctx annot of
                    Cexists (k, t) =>
                       (
                       checkCon ctx c k;
                       checkValue ctx v (substCon c t);
                       annot
                       )
                  | _ =>
                       raise TypeError)

           | Vtuple vl =>
                Cprod (map (inferValue ctx) vl)

           | Vinj (v, i, annot) =>
                (case whnfAnnot ctx annot of
                    Csum tl =>
                       let
                          val t =
                             List.nth (tl, i)
                             handle Subscript => raise TypeError
                       in
                          checkValue ctx v t;
                          annot
                       end
                  | _ =>
                       raise TypeError)

           | Vroll (v, annot) =>
                (case whnfAnnot ctx annot of
                    Crec t =>
                       (
                       checkValue ctx v (substCon annot t);
                       annot
                       )
                  | _ =>
                       raise TypeError)

           | Vunroll v =>
                (case inferValueWhnf ctx v of
                    t as Crec t' =>
                       substCon t t'
                  | _ =>
                       raise TypeError)

           | Vtag (v1, v2) =>
                (case inferValueWhnf ctx v1 of
                    Ctag t =>
                       (
                       checkValue ctx v2 t;
                       Cexn
                       )
                  | _ =>
                       raise TypeError)
                                 
           | Vbool b => Cbool
           | Vint _ => Cint
           | Vchar _ => Cchar
           | Vstring _ => Cstring)

      and inferValueWhnf ctx v =
         whnf ctx (inferValue ctx v)

      and checkValue ctx v t =
         equiv ctx (inferValue ctx v) t Ktype



      fun checkProgram e =
         checkExp empty e

   end
