#ifndef cDECKHEADER
#define cDECKHEADER

#include <vector>
#include <deque>
#include <cassert>
#include <iostream>
#include <algorithm>
#include <cmath>
#include <iomanip>
#include "Constant.h"

using namespace std;

typedef int cCard;

double max(double a, double b);
double max(double a, double b, double c);

//This data structure stores the expected payoff (of 1 dollar bet) of various player choices given...
//  1) Dealer's up card
//  2) Player's current total
struct cExp
{
	double exp; // the expectation for best choice (if dec is DOUBLEORHIT, or DOUBLEORSTAND, this equals expdouble)
	cDecision dec; // the best choice
	
	// the expectation for double, stand and hit
	double expdouble;
	double expstand;
	double exphit;
	
	double expsoft; //the expection for best choice when the total is soft
	cDecision decsoft; //the best choice when total is soft
	double expdoublesoft;
	double expstandsoft; //Note: this always equal to expstand, (it's here for ease of reading and debuging)
	double exphitsoft;

	bool split; // true if split is the best decision
	double expsplit; // the expected payoff for split
};

class cDeck
{
public:
	
	//Initialize nDeck decks of card
	//  if bTen = true, then J, Q and K will be counted as 10
	cDeck(int nDeck = NUMDECK, bool ten = true);
	
	//Returns 1 = A, 2 = 2, 3 = 3 .... 11 = J, 12 = Q, 13 = K
	//  Note: This function REMOVES a card from the deck.
	cCard Deal();

	//Reshuffle the deck
	void Reset(void);

	//Output the distribution of the current deck.  (i.e. how many of each card left?)
	void Output(void);

	//Returns the number of cards left.
	int NumLeft(){return Total;};
	
	//Returns the number of n left.
	//  n = 1 or 11, returns number of Ace
	//  n = 10 && bTen, returns total number of 10, J, Q, K
	int NumLeft(int n);

	//Return the number of initial deck.
	int NumDeck(){return nDeck;};

	//Two simple estimator of the current distribution.
	//  Note: returns Count/Deck
	double HighLowCount(void);
	double MyCount(void);

	//Input:
	//  nTotal = Dealer's current total
	//  bSoft = Is this dealer's soft total?
	//  s = Dealer's strategy
	//  Current distribution of the deck
	//Output:
	//  vector<double> Result[i] = Probability that dealer will stop with a total of i, for i = 0 to MAXTOTAL+11
	//Note:
	//  This outputs an estimation, but it's fairly accurate.
	//  (Because I assumed the distribution of deck does not change in the same deal.)
	vector<double> Prob(const int nTotal, const bool bSoft = false, cStrat s = sHITSOFT17);

	//figure out the expectation at every point given the deck
	//expection of 1 mean definitely win, 0 means tie, -1 means lose

	//Input:
	//  upcard = Dealer's upcard
	//  s = Dealer's strategy
	//  Current distribution of the deck
	//Output:
	//  vector<cExp> Result[i] = The expected payoff when player has total i.
	//  Note: Payoff computation assumes player bet 1 dollars.
	//        cExp contains the expected payoff for all possible player choices.
	vector<cExp> Exp(const int upcard, const cStrat s = sHITSOFT17);
	
	//Compute the expected payoff (of 1 dollar bet) given the current deck configuration.
	double OverallExp(cStrat strat = sHITSOFT17);

	//Remove a specific card (useful for testing)
	//  Note: Slow.  i.e. O(# of card in deck)
	//        bTen doesn't affect this function
	int ForceDeal(cCard card);

private:
	//The number of 52-cards-deck used to create this deck.
	int nDeck;

	//Should J, Q, K be treated as 10?
	bool bTen;

	//This vector stores the distribution of the current deck
	vector<cCard> Card;
 
	//Total number of cards left
	int Total;

	//An estimator of the deck's distribution
	int HighLowIndex;

	//Store the whole deck
	deque<cCard> Deck;
};

struct cHand
{
	vector<cCard> cards;
	double bet;
	bool bDouble;
	bool bSplit;

	//default
	cHand(){Reset();};

	//split, if any = false, that means player can not split or double after split
	cHand(int card, bool any = false){Reset(); bDouble = any; bSplit = any;};

	void Reset()
	{
		bet = 1.0;
		bDouble = true;
		bSplit = true;
		cards.erase(cards.begin(), cards.end());
	}

	void Add(cCard card); 	//Add a card
	void Double(){assert(bDouble); bet *= 2.0; bDouble = false;};

	//Return the total of current hand (and if the total is soft)
	int GetTotal(){bool bTemp; return GetTotal(bTemp);};
	int GetTotal(bool& soft);
	
	//Check if current hand is Blackjack
	bool IsBlackjack();

	//output
	void Output();
};

//Various strategy for the player
typedef cDecision (fDecision) (cDeck&, cHand, cHand);
cDecision HitSoft17(cDeck& deck, cHand dealer, cHand player);
cDecision Hit16(cDeck& deck, cHand dealer, cHand player);
cDecision Optimal(cDeck& deck, cHand dealer, cHand player);
cDecision Basic(cDeck& deck, cHand dealer, cHand player);

//Various betting method for the player
typedef double (fBet) (cDeck& deck);
double AlwaysOne(cDeck& deck);
double TwoBet(cDeck& deck);
double MultiBet(cDeck& deck);
double CountBet(cDeck& deck);

class cGame
{
public:
	
	cGame(fDecision dealer = &HitSoft17, fDecision player = &Basic, fBet bet = &AlwaysOne, bool bDoubleAfterSplit = false, bool bSplitAfterSplit = false)
	{
		//Set up player and dealer's strategies
		fDealer = dealer;
		fPlayer = player;
		fMakeBet = bet;

		//Set up rules
		bDAS = bDoubleAfterSplit;
		bSAS = bSplitAfterSplit;

		//Statistics (divided into earning for each count)
		money.erase(money.begin(), money.end());
		money.resize(MAXCOUNT*2+1, 0);

		times.erase(times.begin(), times.end());
		times.resize(MAXCOUNT*2+1, 0);
		
		//Start a new game
		NewRound();
	}
	
	//Output the statistics of earning for each count
	void OutStat();

	//Start a new game
	void NewRound();

	//Player for nGame, Show each hand if bDetail = true
	void Play(long nGame, bool bDetail);

	//Compute the changes in money since I played.
	double GetMoney()
	{
		double res = 0;
		for(int i=0; i<money.size(); i++)
		{
			res += money[i];
		}
		return res;
	}
	
	void UpdateMoney(int count, double diff);
	double GetMoney(int count);
	double GetTimes(int count);

private:
	
	cCard Deal(){return deck.Deal();};

	cHand dealer;
	vector<cHand> player;

	//Decision function for both dealer and player
	fDecision* fDealer;
	fDecision* fPlayer;
	//Player's betting strategy
	fBet* fMakeBet;

	bool bDAS; //double after split
	bool bSAS; //split after split

	cDeck deck;

	//Earning/Frequency for each count.
	vector<double> money;
	vector<int> times;

};

#endif