!+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
! File rocketS.f90
!
!  The original problem formulation:
!        min            t1
!        subject to:
!                       y1'   = y2
!                       y2'   = 2*u1 - 1
!
!                       y1(0) = 0
!                       y2(0) = 0
!
!                       y1(tf) = 100
!                       u1 = 1 for t in [t0,t1]
!                       u1 = 0 for t in [t1,tf]
!
!  The problem input to SNCTRL:
!        min            y3
!        subject to:
!                       y1'   = y2*p1
!                       y2'   = (2*u1 - 1)*p1
!                       y3'   = p1
!
!                       y1(0) = 0
!                       y2(0) = 0
!                       y3(0) = 0
!
!                       y1(tf) = 100
!                       u1 = 1 for t in [t0,t1]
!                       u1 = 0 for t in [t1,tf]
!                       p1 >= 0
!
!+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

program rocketS
  use precision,   only : ip, rp
  use ctUsrModule, only : ctProb
  use control
  implicit none

  !=====================================================================

  ! SNOPT variables and workspaces
  character(20) :: lfile
  real(rp)      :: sInf
  integer(ip)   :: Start, INFO, iPrint, iSpecs, nOut, &
                   iSumm, mincw, miniw, minrw, nS, nInf, &
                   lencu, leniu, lenru, &
                   lencw, leniw, lenrw
  parameter ( lenrw = 50000, leniw = 50000, lencw = 500, &
              lenru = 1,     leniu = 1,     lencu = 1)
  integer(ip)  :: iu(leniu), iw(leniw)
  real(rp)     :: ru(lenru), rw(lenrw)
  character(8) :: cu(lencu), cw(lencw)

  ! Structures, variables, constants, etc.
  integer(ip) :: nY, nU, nP, nC, nPhs, j
  integer(ip), pointer :: hs(:), objL(:), ytype(:,:), npInt(:), neJ(:)
  real(rp),    pointer :: phsPt(:), x(:)
  character(8) :: name
  external     :: odecon, algcon, varbds
  type(ctProb) :: prob

  !=====================================================================

  iSpecs =  4  ! equivalenced to .spc
  iPrint =  9  ! equivalenced to .out
  iSumm  =  6  ! summary file goes to standard output...
  nOut   =  6  ! ... as do messages from this program.

  ! Open the Specs and print files.
  lfile = 'rocketS.spc'
  open( iSpecs, file=lfile, status='OLD',     err=800 )
  lfile = 'rocketS.out'
  open( iPrint, file=lfile, status='UNKNOWN', err=800 )


  ! Set options to default values.
  call sncInit ( iPrint, iSumm, prob, cw, lencw, iw, leniw, rw, lenrw )


  ! Read a Specs file.
  call sncSpec ( iSpecs, INFO, cw, lencw, iw, leniw, rw, lenrw )

  if (INFO /= 101  .AND.  INFO /= 107) go to 910


  ! Set up problem
  name(1:8) = 'rocketS '
  nY = 3       ! number of states
  nU = 1       ! number of controls
  nP = 1       ! number of parameters
  nC = 0       ! number of algebraic constraints
  nPhs = 2     ! number of phases

  ! Derivatives
  allocate ( neJ(2) )
  neJ(1) = 5
  neJ(2) = 5

  ! Objective
  allocate ( objL(2) )
  objL(1) = 3
  objL(2) = 0

  ! Interval
  allocate ( phsPt(3), npInt(2) )
  phsPt(1) = 0.0d+0
  phsPt(2) = 0.5d+0
  phsPt(3) = 1d+0

  npInt(1) = 5
  npInt(2) = 5


  ! Linearity
  allocate ( ytype(3,2) )
  do j = 1, 2
     ytype(1,j) = 0
     ytype(2,j) = 0
     ytype(3,j) = 0
  end do


  ! Set the ctProb structure
  prob%probName = name
  prob%nY   = nY
  prob%nU   = nU
  prob%nP   = nP
  prob%nC   = nC
  prob%nPhs = nPhs

  prob%objL  => objL
  prob%ytype => ytype
  prob%npInt => npInt
  prob%phsPt => phsPt
  prob%neJ   => neJ


  !---------------------------------------------------------------------
  ! Solve the problem, call snctrl.
  !---------------------------------------------------------------------
  Start = 0
  call snctrl ( Start, prob, x, hs, odecon, algcon, varbds, &
                INFO, mincw, miniw, minrw, nS, nInf, sInf, &
                cu, lencu, iu, leniu, ru, lenru, &
                cw, lencw, iw, leniw, rw, lenrw )

  if (INFO == 82 .OR. INFO == 83 .OR. INFO == 84) then
     go to 910
  end if

  write(nOut, *) ' '
  write(nOut, *) 'snctrl finished.'
  write(nOut, *) 'INFO          =', INFO
  write(nOut, *) 'nInf          =', nInf
  write(nOut, *) 'sInf          =', sInf

  deallocate ( objL, phsPt, npInt, ytype, x, hs, neJ )

  if (INFO >= 30) go to 900

  stop

  !---------------------------------------------------------------------
  ! Error exit.
  !---------------------------------------------------------------------
800 write(nOut, 4000) 'Error while opening file', lfile
  stop

900 write(nOut, *) ' '
  write(nOut, *) 'STOPPING because of error condition'
910 stop

4000 format(/  a, 2x, a  )

end program rocketS

!+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

subroutine varbds ( curPhs, nPhs, nY, nU, nP, nC, nNodes, lbds, ubds, x, &
                    plbds, pubds, p, clbds, cubds )
  use precision, only : ip, rp
  implicit none
  integer(ip), intent(in)  :: curPhs, nPhs, nY, nU, nP, nC, nNodes
  real(rp),    intent(out) :: lbds(nY+nU,nNodes), ubds(nY+nU,nNodes), &
                              x(nY+nU,nNodes), plbds(nP), pubds(nP), p(nP), &
                              clbds(nC), cubds(nC)
  !=============================================================================
  integer(ip) :: j
  real(rp)    :: bplus, bminus
  parameter    ( bplus  = 1.0d+20, bminus = -bplus )

  lbds = bminus
  ubds = bplus
  x    = 0.0

  if ( curPhs == 1 ) then
     ! At t0
     lbds(1,1) = 0.0
     lbds(2,1) = 0.0

     ubds(1,1) = 0.0
     ubds(2,1) = 0.0

     do j = 1, nNodes
        lbds(3,j) = 0.0

        lbds(4,j) = 1.0
        ubds(4,j) = 1.0
     end do
  end if

  if ( curPhs == nPhs ) then
     ! At tf
     lbds(1,nNodes) = 100.0
     ubds(1,nNodes) = 100.0

     do j = 1, nNodes
        lbds(3,j) = 0.0

        lbds(4,j) = 0.0
        ubds(4,j) = 0.0
     end do

  end if

  ! Parameter
  plbds(1) = 0.0
  pubds(1) = bplus
  p(1)     = 1.0

end subroutine varbds

!+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

subroutine odecon ( snStat, curPhs, nPhs, nY, nU, nP, nNodes, F, &
                    Jrow, Jval, Jcol, lenJ, dvar, pvar, needF, needJ, &
                    cu, lencu, iu, leniu, ru, lenru )
  use precision, only : ip, rp
  implicit none
  integer(ip), intent(in) :: snStat, curPhs, nPhs, nY, nU, nP, nNodes, lenJ, &
                             needF, needJ, lencu, leniu, lenru
  real(rp),    intent(in) :: dvar(nY+nU,nNodes), pvar(nP)

  integer(ip), intent(inout) :: iu(leniu)
  real(rp),    intent(inout) :: ru(lenru)
  character(8),intent(inout) :: cu(lencu)

  integer(ip), intent(out) :: Jrow(lenJ), Jcol(nY+nU+nP)
  real(rp),    intent(out) :: F(nY,nNodes), Jval(lenJ,nNodes)

  !=============================================================================
  integer(ip) :: jt, neJ

  if ( needF > 0 ) then
     do jt = 1, nNodes
        F(1,jt) = dvar(2,jt)*pvar(1)
        F(2,jt) = (2*dvar(1+nY,jt) - 1) * pvar(1)
        F(3,jt) = pvar(1)
     end do
  end if

  if ( needJ > 0 ) then
     do jt = 1, nNodes
        neJ = 1
        ! Column 1
        Jcol(1) = neJ

        ! Column 2
        Jcol(2) = neJ

        Jrow(neJ) = 1
        Jval(neJ,jt) = pvar(1)
        neJ = neJ + 1

        ! Column 3
        Jcol(3) = neJ

        ! Column 4
        Jcol(4) = neJ

        Jrow(neJ) = 2
        Jval(neJ,jt) = 2.0*pvar(1)
        neJ = neJ + 1

        ! Column 5
        Jcol(5) = neJ

        Jrow(neJ) = 1
        Jval(neJ,jt) = dvar(2,jt)
        neJ = neJ + 1

        Jrow(neJ) = 2
        Jval(neJ,jt) = 2.0*dvar(1+nY,jt) - 1.0
        neJ = neJ + 1

        Jrow(neJ) = 3
        Jval(neJ,jt) = 1.0
        neJ = neJ + 1

        ! Finish off column pointers
        Jcol(6) = neJ
        neJ = neJ - 1

     end do
  end if

end subroutine odecon

!+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

subroutine algcon ( snStat, curPhs, nPhs, nC, nY, nU, nP, nNodes, C, &
                    Grow, Gval, Gcol, lenG, dvar, pvar, &
                    needC, needG, cu, lencu, iu, leniu, ru, lenru )
  use precision, only : ip, rp
  implicit none
  integer(ip), intent(in) :: snStat, curPhs, nPhs, nC, nY, nU, nP, nNodes, &
                             lenG, needC, needG, lencu, leniu, lenru
  real(rp),    intent(in) :: dvar(nY+nU,nNodes), pvar(nP)

  integer(ip), intent(inout) :: iu(leniu)
  real(rp),    intent(inout) :: ru(lenru)
  character(8),intent(inout) :: cu(lencu)

  integer(ip), intent(out) :: Grow(lenG), Gcol(nY+nU+nP+1)
  real(rp),    intent(out) :: C(nC,nNodes), Gval(lenG,nNodes)

end subroutine algcon

!+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
