/*****************************************************************************/

/* Driver for routine POWELL */

#include <stdio.h>
#include <math.h>
#include "nr-short.h"
#include "nrutil.h"

/*****************************************************************************/
/* Distance metric stuff */

#define N_INPUTS 6
#define N_OUTPUTS 2
#define N_DIMENSIONS (N_INPUTS + N_OUTPUTS)
#define N_FIT_INPUTS 28
#define N_POINTS 1000

#define N_POINTS_TO_PREDICT 1000

float metric[N_INPUTS];
	/* metric[i] = p[i+1] with the last element of metric = 1.0 */
float *points[N_POINTS];
double input_array[N_FIT_INPUTS];
double output_array[N_OUTPUTS];
double altered_point[N_INPUTS+1];

/* Order of points in data file is p1, v1, a1, t1, p2, v2, a2, t2 */
/* int indices[]={0,1,2,6,3,4,5,7}; */
int indices[]={1,4,5,7,0,3,2,6};

extern double **ldlt_x;

float *malloc_float_array();

/*****************************************************************************/
/* Powell stuff */

#define NDIM (N_INPUTS-1)
#define FTOL 1.0e-6

int count = 0;

float p[NDIM+1]=
  {0.0,1.0,1.0,1.0,1.0,1.0};
/* {0.0,286.0,1.0,1.0,1.0,1.0}; */
/* {0.0, 13862, 1.854, 0.079044, 0.042166, 0.787967 }; */
/* {0.0, 13862.0, 1.0, 1.0, 1.0, 1.0 }; */
/* {0.0, 8971.0, 205.0, 57.0, 0.0525, 113.0 }; */
/* {0.0, 6000.0, 1.0, 1.0, 1.0, 1.0 }; */

/* Why don't I just copy p into metric. Why bother limiting magnitude of
   metric to 1?
   YES: limiting magnitude doesn't do us any good!
*/

/* With indices reversed misses optimum at count = 120, iter = 1.
   int indices[]={3,4,5,7,0,1,2,6};
   float p[NDIM+1]= {0.0,1.0,1.0,1.0,1.0,1.0};
   This was done with a bug: value -> sqrt(value)
   Happened again with:
   int indices[]={1,4,5,7,0,3,2,6}; and value bug fixed.
*/

float func();

float **xi;

int iter;

/*****************************************************************************/

main()
{
  int i, j;
  float fret;

  allocate_points();
  read_points( "points.r2" );
  init_ldlt( N_FIT_INPUTS, N_OUTPUTS );

  xi = matrix( 1, NDIM, 1, NDIM );
  for ( i = 1; i <= NDIM; i++ )
    for ( j = 1; j <= NDIM; j++ )
      xi[i][j] = (i == j ? 1.0 : 0.0 );
  powell( p, xi, NDIM, FTOL, &iter, &fret, func );
  printf( "Iterations: %d (%d)\n\n", iter, count );
  printf( "Minimum found at: \n" );
  for ( i = 1; i <= NDIM; i++ )
    printf( "%12.6f", p[i] );
  printf( "\nMetric is: \n" );
  for ( i = 0; i <= NDIM; i++ )
    printf( "%12.6f", metric[i] );
  printf( "\n\nMinimum function value = %12.6f \n\n", fret );
  free_matrix( xi, 1, NDIM, 1, NDIM );
}

/*****************************************************************************/

allocate_points()
{
  int i;

  for( i = 0; i < N_POINTS; ++i )
    {
      points[i] = malloc_float_array( N_DIMENSIONS );
    }
}

/*****************************************************************************/

read_points( file_name )
     char *file_name;
{
  FILE *fopen(), *pstream;
  int i, j;
  float a1, a2, t1, t2;

  printf( "Reading points from file %s:\n", file_name );
  pstream = fopen( file_name, "r" );
  if ( pstream == NULL )
    {
      fprintf( stderr, "Cannot open file %s for reading.\n", file_name );
      exit( -1 );
    }

  for( i=0; i < N_POINTS; ++i )
    {
      for( j=0; j < N_DIMENSIONS; ++j )
	{
	  fscanf( pstream, "%f", &(points[i][indices[j]]) );
	}
    }

  fclose( pstream );

  /*
  for( i=0; i < N_POINTS; ++i )
    {
      two_joint_forward_dynamics( points[i][0], points[i][3],
				 points[i][1], points[i][4],
				 points[i][6], points[i][7],
				 &a1, &a2 );
      two_joint_inverse_dynamics( points[i][0], points[i][3],
				 points[i][1], points[i][4],
				 points[i][2], points[i][5],
				 &t1, &t2 );
      printf( "%d: %f %f %f %f\n", i,
	     points[i][2] - a1, points[i][5] - a2,
	     points[i][6] - t1, points[i][7] - t2 );
	     
    }
    */
}

/*****************************************************************************/

float *malloc_float_array( size )
     int size;
{
  float *p;

  p = (float *) malloc( size * sizeof(float) );
  if ( p == NULL )
    {
      fprintf( stderr, "Memory allocation failed for float array of size %d",
	      size );
      exit( -1 );
    }
  return p;
}

/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/

#define SQUARE(a) ((a)*(a))

float func(x)
float x[];
{
  int i,j,k,index;
  double value, cost_function;
  int point;
  double distance, weight;

  /* NOPE, don't bother to normalize length. Just drops out in the wash. */
  for( i = 1; i <= NDIM; ++i )
    {
      metric[i-1] = x[i];
    }
  metric[ NDIM ] = 1.0;
  
  printf( "Metric is: \n" );
  for ( i = 0; i <= NDIM; i++ )
    printf( "%12.6f", metric[i] );
  printf( "\n" );

  cost_function = 0.0;

  for( point = 0; point < N_POINTS_TO_PREDICT; point++ )
    {   /* Pick a point and try to predict its value. */
      /* skip making fake points. */

      /* Need to find closest point to scale weights against
	 (if (> (car distances) min-min-distance)
	 (setq min-distance (car distances))
	 (setq min-distance min-min-distance))
	 (setq weight 1.0) ; default weighting.
	 */

      zero_ata_atb();

      for( i = 0; i < N_POINTS; i++ )
	{
	  if ( i == point )
	    continue;

	  /* compute the distance */
	  distance = 0;
	  for ( j = 0; j < N_INPUTS; j++ )
	    {
	      distance +=
		SQUARE(metric[j] * (points[i][j] - points[point][j]));
	    }
	  if ( distance < 1e-20 )   /* HACK: Fix later. */
	    distance = 1e-20;
	  /* compute a weight */
	  weight = 1.0/(distance*distance);

	  /* Linear model: 
	  input_array[0] = 1.0*weight;
	  for ( j = 0; j < N_INPUTS; j++ )
	    {
	      input_array[j+1] = (points[i][j] - points[point][j]) * weight;
	    }
	  */
	  /* Quadratic model: */
	  altered_point[0] = 1.0;
	  for ( j = 0; j < N_INPUTS; j++ )
	    {
	      altered_point[j+1] = (points[i][j] - points[point][j]);
	    }
	  index = 0;
	  for ( j = 0; j <= N_INPUTS; ++j )
	    {
	      for ( k = j; k <= N_INPUTS; ++k )
		{
		  input_array[index++] =
		    altered_point[j] * altered_point[k] * weight;
		}
	    }
	  if ( index != N_FIT_INPUTS )
	    {
	      fprintf( stderr, "BUG IN QUADRATIC MODEL: %d %d\n",
		      index, N_FIT_INPUTS );
	      exit( -1 );
	    }

	  for ( j = 0; j < N_OUTPUTS; j++ )
	    {
	      output_array[j] =
		(points[i][j+N_INPUTS] - points[point][j+N_INPUTS]) * weight;
	    }
	  add_point( input_array, output_array );
	}

      ldlt_decompose();
      ldlt_solve_1();
      ldlt_solve_2();
      value = SQUARE(ldlt_x[0][0]) + SQUARE(ldlt_x[0][1]);
      cost_function += value;
    
      count++;

      /*
      for( i = 0; i < N_INPUTS+1; i++ )
	{
	  for( j = 0; j < N_OUTPUTS; j++ )
	    {
	      printf( "%lf ", ldlt_x[i][j] );
	    }
	  printf( "\n" );
	}
      printf( "\n%d: function value = %lg \n\n", count, value );
      */
    }

  printf( "%d %d: Cost_function = %lg\n\n", iter, count, cost_function );
  return cost_function;
}

/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/
/* Arm parameters. */
float l1 = 1.0;
float l2 = 1.0;
float m1 = 1.0;
float m2 = 1.0;
float mcx1 = 0.5;
float mcy1 = 0.0;
float mcx2 = 0.5;
float mcy2 = 0.0;
/* link moments about proximal joints. */
float I1 = 0.333333;
float I2 = 0.333333;
float g = 0.0;

two_joint_forward_dynamics( p1, p2, v1, v2, t1, t2, a1, a2 )
     float p1, p2, v1, v2, t1, t2;
     float *a1, *a2;
{
  float c1, s1, c2, s2, c12, s12;
  float a, b, d, e, f, determinant;

  /* The 2x2 matrix H = [[a,b],[b,d]] is formed:
     | a b | | a1 |   | e |
     | b d | | a2 | = | f |
     Then (a1,a2) = H^(-1).(e,f)
     */

  c1 = cos( p1 );
  s1 = sin( p1 );
  c2 = cos( p2 );
  s2 = sin( p2 );
  c12 = cos( p1 + p2 );
  s12 = sin( p1 + p2 );

  a = I1 + I2 + m2*l1*l1 + 2*mcx2*l1*c2 - 2*mcy2*l1*s2;
  b = I2 + mcx2*l1*c2 - mcy2*l1*s2;
  d = I2;
  e = t1 - mcx1*g*c1 + mcy1*g*s1 - m2*g*l1*c1
    - mcx2*g*c12 + 2*mcx2*l1*v1*v2*s2 + mcx2*l1*v2*v2*s2
    + mcy2*g*s12 + 2*mcy2*l1*v1*v2*c2 + mcy2*l1*v2*v2*c2;
  f = t2 - mcx2*g*c12 - mcx2*l1*v1*v1*s2
         + mcy2*g*s12 - mcy2*l1*v1*v1*c2;

  determinant = a*d - b*b;
  *a1 = (d*e - b*f)/determinant;
  *a2 = (-b*e + a*f)/determinant;
}

/*****************************************************************************/

two_joint_inverse_dynamics( p1, p2, v1, v2, a1, a2, t1, t2 )
     float p1, p2, v1, v2, a1, a2;
     float *t1, *t2;
{
  float c1, s1, c2, s2, c12, s12;
  float a, b, d, e, f, determinant;

  /* The 2x2 matrix H = [[a,b],[b,d]] is formed:
     | a b | | a1 |   | e | + | t1 |
     | b d | | a2 | = | f | + | t2 |
     Then (t1,t2) = H.(a1,a2) - (e,f)
     */

  c1 = cos( p1 );
  s1 = sin( p1 );
  c2 = cos( p2 );
  s2 = sin( p2 );
  c12 = cos( p1 + p2 );
  s12 = sin( p1 + p2 );

  a = I1 + I2 + m2*l1*l1 + 2*mcx2*l1*c2 - 2*mcy2*l1*s2;
  b = I2 + mcx2*l1*c2 - mcy2*l1*s2;
  d = I2;
  e = - mcx1*g*c1 + mcy1*g*s1 - m2*g*l1*c1
    - mcx2*g*c12 + 2*mcx2*l1*v1*v2*s2 + mcx2*l1*v2*v2*s2
    + mcy2*g*s12 + 2*mcy2*l1*v1*v2*c2 + mcy2*l1*v2*v2*c2;
  f = - mcx2*g*c12 - mcx2*l1*v1*v1*s2
         + mcy2*g*s12 - mcy2*l1*v1*v1*c2;
  *t1 = a*a1 + b*a2 - e;
  *t2 = b*a1 + d*a2 - f;
}

/*****************************************************************************/
