/*****************************************************************************/
/*
  controller.c: control strategy.
*/
/*****************************************************************************/

#include <stdio.h>
#include <math.h>
#include <stdlib.h>

#include "../useful/trajectory/trajectory.h"
#include "main.h"

/*****************************************************************************/

#define N_BIPED_DOFS 6

#define L_HIP 0
#define L_KNEE 1
#define L_ANKLE 2
#define R_HIP 3
#define R_KNEE 4
#define R_ANKLE 5

#define UNKNOWN_STATE 0
#define WAITING 1
#define LAUNCH1 2
#define LAUNCH2 3
#define L_SWING 4
#define RL_STANCE 5
#define R_SWING 6
#define LR_STANCE 7
#define STOP 8

/* Servo modes */
#define PD_MODE 0
#define TORQUE_MODE 1

/*****************************************************************************/
/* Control variables */

int controller_print = 0;

/*****************************************************************************/
/* Storage. */

double torque_score = 0; /* sum of torque penalties */
double crashed_penalty = 0; /* amount of time walker is crashed. */
float l_foot_x = 0;
float r_foot_x = 0;
float last_l_foot_x = 0;
float last_r_foot_x = 0;

DOFS desired_trajectory[N_BIPED_DOFS];
DOFS current_desireds[N_BIPED_DOFS];

/*****************************************************************************/
/*****************************************************************************/

void reset_score( SIM *s )
{
  torque_score = 0;
  crashed_penalty = 0;
}

/*****************************************************************************/

void update_score( SIM *s )
{
  torque_score += s->hip_command[LEFT]*s->hip_command[LEFT];
  torque_score += s->hip_command[RIGHT]*s->hip_command[RIGHT];
  torque_score += s->knee_command[LEFT]*s->knee_command[LEFT];
  torque_score += s->knee_command[RIGHT]*s->knee_command[RIGHT];
  torque_score += s->ankle_command[LEFT]*s->ankle_command[LEFT];
  torque_score += s->ankle_command[RIGHT]*s->ankle_command[RIGHT];
  if ( s->status == CRASHED )
    crashed_penalty += s->time_step;
}

/*****************************************************************************/
/*
Could normalize torque and other penalties w.r.t. distance
Could penalize deviations from symmetry (if necessary).
*/

float get_score( SIM *s )
{
  float speed, speed_penalty;
  float total_score;

  speed = s->hip[XX]/s->time;
  speed_penalty = (s->desired_speed - speed)*(s->desired_speed - speed)
    *s->speed_penalty_weight;
  torque_score *= s->torque_penalty_weight;
  total_score = torque_score + speed_penalty + crashed_penalty;
  printf( "Score %g: crashed %g; speed %g; speed score: %g; torque score: %lg;\n",
	  total_score, crashed_penalty, speed, speed_penalty, torque_score );
  return total_score;
}

/*****************************************************************************/
/* call this many times to restart a simulation */

reinit_controller( SIM *s )
{

  s->controller_state = WAITING;
  s->state_start_time = s->time;
  s->state_elapsed_time = 0.0;
  s->desireds_index = s->time;
  s->last_time = s->time;

  s->hip_angle_d[LEFT] = s->hip_angle[LEFT];
  s->hip_angle_d[RIGHT] = s->hip_angle[RIGHT];
  s->knee_angle_d[LEFT] = s->knee_angle[LEFT];
  s->knee_angle_d[RIGHT] = s->knee_angle[RIGHT];
  s->ankle_angle_d[LEFT] = s->ankle_angle[LEFT];
  s->ankle_angle_d[RIGHT] = s->ankle_angle[RIGHT];
  s->hip_angled_d[LEFT] = s->hip_angled[LEFT];
  s->hip_angled_d[RIGHT] = s->hip_angled[RIGHT];
  s->knee_angled_d[LEFT] = s->knee_angled[LEFT];
  s->knee_angled_d[RIGHT] = s->knee_angled[RIGHT];
  /* sidestep angle perturbation for now
  s->ankle_angled_d[LEFT] = s->ankle_angled[LEFT];
  s->ankle_angled_d[RIGHT] = s->ankle_angled[RIGHT];
  */
  s->ankle_angled_d[LEFT] = 0;
  s->ankle_angled_d[RIGHT] = 0;

  s->hip_servo_mode[LEFT] = PD_MODE;
  s->hip_servo_mode[RIGHT] = PD_MODE;
  s->knee_servo_mode[LEFT] = PD_MODE;
  s->knee_servo_mode[RIGHT] = PD_MODE;
  s->ankle_servo_mode[LEFT] = PD_MODE;
  s->ankle_servo_mode[RIGHT] = PD_MODE;

  s->hip_command_ff[LEFT] = 0;
  s->hip_command_ff[RIGHT] = 0;
  s->knee_command_ff[LEFT] = 0;
  s->knee_command_ff[RIGHT] = 0;
  s->ankle_command_ff[LEFT] = 0;
  s->ankle_command_ff[RIGHT] = 0;

  init_trajectory( desired_trajectory, N_BIPED_DOFS );
  init_trajectory( current_desireds, N_BIPED_DOFS );

  /* add actual position as knot to current desireds */
  add_knot_point( current_desireds, L_HIP, QUINTIC_SPLINE, 0.0, 
		  s->hip_angle_d[LEFT], s->hip_angled_d[LEFT], 0.0 );
  add_knot_point( current_desireds, L_KNEE, QUINTIC_SPLINE, 0.0,
		  s->knee_angle_d[LEFT], s->knee_angled_d[LEFT], 0.0 );
  add_knot_point( current_desireds, L_ANKLE, QUINTIC_SPLINE, 0.0,
		  s->ankle_angle_d[LEFT], s->ankle_angled_d[LEFT], 0.0 );
  add_knot_point( current_desireds, R_HIP, QUINTIC_SPLINE, 0.0, 
		  s->hip_angle_d[RIGHT], s->hip_angled_d[RIGHT], 0.0 );
  add_knot_point( current_desireds, R_KNEE, QUINTIC_SPLINE, 0.0, 
		  s->knee_angle_d[RIGHT], s->knee_angled_d[RIGHT], 0.0 );
  add_knot_point( current_desireds, R_ANKLE, QUINTIC_SPLINE, 0.0, 
		  s->ankle_angle_d[RIGHT], s->ankle_angled_d[RIGHT], 0.0 );

  reset_score( s );
}

/*****************************************************************************/
/* call this many times to restart a simulation */

int reinit_sim( SIM *s )
{
  int i;
  double min;

  srand( s->rand_seed );

  s->time = 0.0;
  s->step_length = 0;

  s->hip[XX] = 0;
  s->hip[ZZ] = 0;
  s->pitch = 0;
  s->hipd[XX] = 0;
  s->hipd[ZZ] = 0;
  s->pitchd = 0;

  /* Make it fall */
  // s->pitchd = 1.0;

  s->hip_angle[LEFT] = 0;
  s->hip_angle[RIGHT] = 0;
  s->knee_angle[LEFT] = 0;
  s->knee_angle[RIGHT] = 0;
  s->hip_angled[LEFT] = 0;
  s->hip_angled[RIGHT] = 0;
  s->knee_angled[LEFT] = 0;
  s->knee_angled[RIGHT] = 0;

  compute_ankle_angles( s );

  /* Indicate foot zero positions not set */
  s->foot_zero[LEFT][ZZ] = 1.0;
  s->foot_zero[RIGHT][ZZ] = 1.0;

  /*
  init_state_one_foot_on_ground( s );
  */
  init_state_two_feet_on_ground( s );

  reinit_controller( s );

  l_foot_x = s->foot[LEFT][XX];
  r_foot_x = s->foot[RIGHT][XX];
  last_l_foot_x = s->foot[LEFT][XX];
  last_r_foot_x = s->foot[RIGHT][XX];
}

/*****************************************************************************/
/* Call this once to do one time operations like memory allocation */

int init_sim( SIM *s )
{
  init_dynamics( s );
  reinit_sim( s );
}

/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/

void
run_trajectory( SIM *s, float time )
{
  float position, velocity, acceleration;

  if ( !lookup_spline3( desired_trajectory, L_HIP, time,
			current_desireds,
			&position,
			&velocity,
			&acceleration ) )
    {
      printf( "Lookup failure.\n" );
      exit( -1 );
    }
  /* update current desired position */
  set_first_knot_to( current_desireds, L_HIP, QUINTIC_SPLINE, time,
		     position, velocity, acceleration );
  s->hip_angle_d[LEFT] = position;
  s->hip_angled_d[LEFT] = velocity;

  if ( !lookup_spline3( desired_trajectory, L_KNEE, time,
			current_desireds,
			&position,
			&velocity,
			&acceleration ) )
    {
      printf( "Lookup failure.\n" );
      exit( -1 );
    }
  /* update current desired position */
  set_first_knot_to( current_desireds, L_KNEE, QUINTIC_SPLINE, time,
		     position, velocity, acceleration );
  s->knee_angle_d[LEFT] = position;
  s->knee_angled_d[LEFT] = velocity;

  if ( !lookup_spline3( desired_trajectory, L_ANKLE, time,
			current_desireds,
			&position,
			&velocity,
			&acceleration ) )
    {
      printf( "Lookup failure.\n" );
      exit( -1 );
    }
  /* update current desired position */
  set_first_knot_to( current_desireds, L_ANKLE, QUINTIC_SPLINE, time,
		     position, velocity, acceleration );
  s->ankle_angle_d[LEFT] = position;
  s->ankle_angled_d[LEFT] = velocity;

  if ( !lookup_spline3( desired_trajectory, R_HIP, time,
			current_desireds,
			&position,
			&velocity,
			&acceleration ) )
    {
      printf( "Lookup failure.\n" );
      exit( -1 );
    }

  /* update current desired position */
  set_first_knot_to( current_desireds, R_HIP, QUINTIC_SPLINE, time,
		     position, velocity, acceleration );
  s->hip_angle_d[RIGHT] = position;
  s->hip_angled_d[RIGHT] = velocity;

  if ( !lookup_spline3( desired_trajectory, R_KNEE, time,
			current_desireds,
			&position,
			&velocity,
			&acceleration ) )
    {
      printf( "Lookup failure.\n" );
      exit( -1 );
    }
  /* update current desired position */
  set_first_knot_to( current_desireds, R_KNEE, QUINTIC_SPLINE, time,
		     position, velocity, acceleration );
  s->knee_angle_d[RIGHT] = position;
  s->knee_angled_d[RIGHT] = velocity;

  if ( !lookup_spline3( desired_trajectory, R_ANKLE, time,
			current_desireds,
			&position,
			&velocity,
			&acceleration ) )
    {
      printf( "Lookup failure.\n" );
      exit( -1 );
    }
  /* update current desired position */
  set_first_knot_to( current_desireds, R_ANKLE, QUINTIC_SPLINE, time,
		     position, velocity, acceleration );
  s->ankle_angle_d[RIGHT] = position;
  s->ankle_angled_d[RIGHT] = velocity;
}

/*****************************************************************************/

void
run_servos( SIM *s )
{

  if ( s->hip_servo_mode[LEFT] == PD_MODE )
    {
      s->hip_command[LEFT] = 
	- s->k_hip*( s->hip_angle[LEFT] - s->hip_angle_d[LEFT] )
	- s->b_hip*( s->hip_angled[LEFT] - s->hip_angled_d[LEFT] )
	+ s->hip_command_ff[LEFT];
    }

  if ( s->hip_servo_mode[RIGHT] == PD_MODE )
    {
      s->hip_command[RIGHT] = 
	- s->k_hip*( s->hip_angle[RIGHT] - s->hip_angle_d[RIGHT] )
	- s->b_hip*( s->hip_angled[RIGHT] - s->hip_angled_d[RIGHT] )
	+ s->hip_command_ff[RIGHT];
    }

  if ( s->knee_servo_mode[LEFT] == PD_MODE )
    {
      s->knee_command[LEFT] = 
	- s->k_knee*( s->knee_angle[LEFT] - s->knee_angle_d[LEFT] )
	- s->b_knee*( s->knee_angled[LEFT] - s->knee_angled_d[LEFT] )
	+ s->knee_command_ff[LEFT];
    }

  if ( s->knee_servo_mode[RIGHT] == PD_MODE )
    {
      s->knee_command[RIGHT] = 
	- s->k_knee*( s->knee_angle[RIGHT] - s->knee_angle_d[RIGHT] )
	- s->b_knee*( s->knee_angled[RIGHT] - s->knee_angled_d[RIGHT] )
	+ s->knee_command_ff[RIGHT];
    }

  if ( s->ankle_servo_mode[LEFT] == PD_MODE )
    {
      s->ankle_command[LEFT] = 
	- s->k_ankle*( s->ankle_angle[LEFT] - s->ankle_angle_d[LEFT] )
	- s->b_ankle*( s->ankle_angled[LEFT] - s->ankle_angled_d[LEFT] )
	+ s->ankle_command_ff[LEFT];
    }

  if ( s->ankle_servo_mode[RIGHT] == PD_MODE )
    {
      s->ankle_command[RIGHT] = 
	- s->k_ankle*( s->ankle_angle[RIGHT] - s->ankle_angle_d[RIGHT] )
	- s->b_ankle*( s->ankle_angled[RIGHT] - s->ankle_angled_d[RIGHT] )
	+ s->ankle_command_ff[RIGHT];
    }

  s->hip_torque[LEFT] = s->hip_command[LEFT];
  s->knee_torque[LEFT] = s->knee_command[LEFT];
  s->ankle_torque[LEFT] = s->ankle_command[LEFT];
  s->hip_torque[RIGHT] = s->hip_command[RIGHT]; 
  s->knee_torque[RIGHT] = s->knee_command[RIGHT];
  s->ankle_torque[RIGHT] = s->ankle_command[RIGHT];
  /*
  s->hip_torque[LEFT] = s->hip_torque[RIGHT] = 0.0;
  s->knee_torque[LEFT] = s->knee_torque[RIGHT] = 0.0;
  s->ankle_torque[LEFT] = s->ankle_torque[RIGHT] = 1.0;
  */

  /*
  s->hip_torque[LEFT] = s->hip_command[LEFT]
    + s->rand_scale*((2.0*rand())/RAND_MAX - 1.0);
  s->knee_torque[LEFT] = s->knee_command[LEFT]
    + s->rand_scale*((2.0*rand())/RAND_MAX - 1.0);
  s->hip_torque[RIGHT] = s->hip_command[RIGHT] 
    + s->rand_scale*((2.0*rand())/RAND_MAX - 1.0);
  s->knee_torque[RIGHT] = s->knee_command[RIGHT]
    + s->rand_scale*((2.0*rand())/RAND_MAX - 1.0);
  */
}

/*****************************************************************************/

void
reset_desireds( SIM *s )
{
  /* update current DESIRED positions */
  set_first_knot_to( current_desireds, L_HIP, QUINTIC_SPLINE, s->time,
		     s->hip_angle_d[LEFT], s->hip_angled_d[LEFT], 0.0 );
  set_first_knot_to( current_desireds, L_KNEE, QUINTIC_SPLINE, s->time,
		     s->knee_angle_d[LEFT], s->knee_angled_d[LEFT], 0.0 );
  set_first_knot_to( current_desireds, L_ANKLE, QUINTIC_SPLINE, s->time,
		     s->ankle_angle_d[LEFT], s->ankle_angled_d[LEFT], 0.0 );
  set_first_knot_to( current_desireds, R_HIP, QUINTIC_SPLINE, s->time,
		     s->hip_angle_d[RIGHT], s->hip_angled_d[RIGHT], 0.0 );
  set_first_knot_to( current_desireds, R_KNEE, QUINTIC_SPLINE, s->time,
		     s->knee_angle_d[RIGHT], s->knee_angled_d[RIGHT], 0.0 );
  set_first_knot_to( current_desireds, R_ANKLE, QUINTIC_SPLINE, s->time,
		     s->ankle_angle_d[RIGHT], s->ankle_angled_d[RIGHT], 0.0 );
}

/*****************************************************************************/
/* THIS SEEMS BOGUS. HOW ABOUT MATCHING CURRENT VELOCITY? */
void
zero_velocity_desireds( SIM *s, int side )
{
  /* When the leg has an impact, adjust DESIRED positions to reflect that. */
  if ( side == LEFT )
    {
      set_first_knot_to( current_desireds, L_HIP, QUINTIC_SPLINE, s->time,
			 s->hip_angle_d[LEFT], 0.0, 0.0 );
      set_first_knot_to( current_desireds, L_KNEE, QUINTIC_SPLINE, s->time,
			 s->knee_angle_d[LEFT], 0.0, 0.0 );
    }
  if ( side == RIGHT )
    {
      set_first_knot_to( current_desireds, R_HIP, QUINTIC_SPLINE, s->time,
			 s->hip_angle_d[RIGHT], 0.0, 0.0 );
      set_first_knot_to( current_desireds, R_KNEE, QUINTIC_SPLINE, s->time,
			 s->knee_angle_d[RIGHT], 0.0, 0.0 );
    }
}

/*****************************************************************************/

void setup_swing_knots( SIM *s, int swing_hip, int swing_knee )
{
  clear_knots( desired_trajectory, swing_hip );
  clear_knots( desired_trajectory, swing_knee );

  add_knot_point( desired_trajectory, swing_hip, QUINTIC_SPLINE, 
		  s->swing_hip_overshoot_time*s->swing_time + s->time,
		  s->swing_hip_target + s->swing_hip_overshoot - s->pitch_d,
		  0.0, 0.0 );
  add_knot_point( desired_trajectory, swing_hip, QUINTIC_SPLINE, 
		  s->swing_time + s->time,
		  /* GET RID OF MAGIC NUMBER */
		  // s->swing_hip_target - s->pitch_d, 0.285, 0.0 );
		  s->swing_hip_target - s->pitch_d, s->swing_hv1, 0.0 );
  add_knot_point( desired_trajectory, swing_knee, QUINTIC_SPLINE, 
		  s->swing_knee1_time*s->swing_time + s->time,
		  s->swing_knee1, 0.0, 0.0 );
  add_knot_point( desired_trajectory, swing_knee, QUINTIC_SPLINE, 
		  s->swing_knee2_time*s->swing_time + s->time,
		  /* GET RID OF MAGIC NUMBER */
		  // s->swing_knee2, -6.0, 0.0 );
		  s->swing_knee2, s->swing_kv1, 0.0 );
  add_knot_point( desired_trajectory, swing_knee, QUINTIC_SPLINE, 
		  s->swing_time + s->time,
		  /* GET RID OF MAGIC NUMBER */
		  // s->thrust1, -2.53, 0.0 );
		  s->thrust1, s->swing_kv2, 0.0 );
  /* and now setup swing knots in the future */
  /*
  add_knot_point( desired_trajectory, swing_hip, QUINTIC_SPLINE, 
		  s->swing_time + s->swing_time + s->time,
		  s->stance_hip_target - s->pitch_d, 0.0, 0.0 );
  add_knot_point( desired_trajectory, swing_knee, QUINTIC_SPLINE, 
// GET RID OF 0.2
		  s->swing_time + 0.2*s->swing_time + s->time,
		  s->thrust1, 0.0, 0.0 );
  add_knot_point( desired_trajectory, swing_knee, QUINTIC_SPLINE, 
		  s->swing_time + s->pushoff_time*s->swing_time + s->time,
		  s->thrust1, 0.0, 0.0 );
  add_knot_point( desired_trajectory, swing_knee, QUINTIC_SPLINE, 
		  s->swing_time + s->swing_time + s->time,
		  0.0, 0.0, 0.0 );
  */
}

/*****************************************************************************/

void setup_stance_knots( SIM *s, int stance_hip, int stance_knee )
{
  clear_knots( desired_trajectory, stance_hip );
  clear_knots( desired_trajectory, stance_knee );

  /*
  add_knot_point( desired_trajectory, stance_hip, QUINTIC_SPLINE, 
		  s->pushoff_time*s->swing_time + s->time,
		  s->stance_hp1, s->stance_hv1, 0.0 );
  */
  add_knot_point( desired_trajectory, stance_hip, QUINTIC_SPLINE, 
		  s->swing_time + s->time,
		  s->stance_hip_target - s->pitch_d, 0.0, 0.0 );
  add_knot_point( desired_trajectory, stance_knee, QUINTIC_SPLINE, 
		  /* GET RID OF 0.2 */
		  0.2*s->swing_time + s->time,
		  s->thrust1, 0.0, 0.0 );
  add_knot_point( desired_trajectory, stance_knee, QUINTIC_SPLINE, 
		  s->pushoff_time*s->swing_time + s->time,
		  s->thrust1, 0.0, 0.0 );
  add_knot_point( desired_trajectory, stance_knee, QUINTIC_SPLINE, 
		  s->swing_time + s->time,
		  0.0, 0.0, 0.0 );
  /*
  printf( "setup_stance_knots:\n" );
  print_trajectory( desired_trajectory, 4 );
  */
}

/*****************************************************************************/

void
run_state_machine( SIM *s )
{

  /* Debounce state transitions */
  if ( s->time - s->state_start_time < 0.001 )
    return;
  s->state_elapsed_time = s->time - s->state_start_time;

  /* Do actions and transitions */
  switch( s->controller_state )
    {
    case WAITING:
      // printf( "WAITING: %g %g\n", s->time, s->wait_duration );
      if ( s->time >= s->wait_duration )
	{
	  s->controller_state = LAUNCH1;
	  s->state_start_time = s->time;
	  if ( controller_print )
	    printf( "LAUNCH1 %g\n", s->time );
	  clear_all_knots( desired_trajectory, N_BIPED_DOFS );
	  /* Lift and swing left leg */
	  add_knot_point( desired_trajectory, L_HIP, QUINTIC_SPLINE, 
			  s->l1_duration + s->time,
			  s->l1_lhip_target, 0.0, 0.0 );
	  add_knot_point( desired_trajectory, L_KNEE, QUINTIC_SPLINE, 
			  s->l1_duration + s->time,
			  s->l1_lknee_target, 0.0, 0.0 );
	  s->ankle_servo_mode[LEFT] = TORQUE_MODE;
	  s->ankle_command[LEFT] = 0.0;
	  s->ankle_servo_mode[RIGHT] = TORQUE_MODE;
	  s->ankle_command[RIGHT] = s->l1_rankle_torque;

	  /* Swing left leg */
	  add_knot_point( desired_trajectory, L_HIP, QUINTIC_SPLINE, 
			  s->l1_duration + 
			  s->l2_duration + s->time,
			  s->l2_lhip_target, 0.0, 0.0 );
	  add_knot_point( desired_trajectory, L_KNEE, QUINTIC_SPLINE, 
			  s->l1_duration + 
			  s->l2_duration + s->time,
			  s->l2_lknee_target, 0.0, 0.0 );
	  add_knot_point( desired_trajectory, R_HIP, QUINTIC_SPLINE, 
			  s->l1_duration + 
			  s->l2_duration + s->time,
			  s->l2_rhip_target, 0.0, 0.0 );
	  add_knot_point( desired_trajectory, R_KNEE, QUINTIC_SPLINE, 
			  s->l1_duration + 
			  s->l2_duration + s->time,
			  s->l2_rknee_target, 0.0, 0.0 );
	}
      break;
    case LAUNCH1:
      /*
      printf( "LAUNCH1: %g %g %g\n",
	      s->state_elapsed_time, s->l1_duration, s->time );
      */
      if ( s->state_elapsed_time >= s->l1_duration )
	{
	  s->controller_state = LAUNCH2;
	  s->state_start_time = s->time;
	  if ( controller_print )
	    printf( "LAUNCH2 %g\n", s->time );
	  /* Don't need to do anything here. */
	  /* Eventually delete a launch state */
	}
      break;
    case LAUNCH2: 
      if ( s->state_elapsed_time >= s->l2_duration
	   && s->gndforce[RIGHT][ZZ] <= 0.001 )

	{
	  s->controller_state = R_SWING;
	  s->state_start_time = s->time;
	  if ( controller_print )
	    printf( "R_SWING %g\n", s->time );
	  /* Swing right leg while moving left leg back. */
	  reset_desireds( s );
	  setup_stance_knots( s, L_HIP, L_KNEE );
	  setup_swing_knots( s, R_HIP, R_KNEE );
	  last_r_foot_x = r_foot_x;
	  s->ankle_command[LEFT] = s->stance_ankle_torque;
	  s->ankle_command[RIGHT] = 0.0;
	}
      break;
    case L_SWING:
      if ( s->gndforce[LEFT][ZZ] > 0.0 && s->state_elapsed_time > s->swing_time/2 )
	{
	  s->controller_state = RL_STANCE;
	  s->state_start_time = s->time;
	  if ( controller_print )
	    printf( "RL_STANCE %g\n", s->time );
	  reset_desireds( s );
	  zero_velocity_desireds( s, LEFT );
	  setup_stance_knots( s, L_HIP, L_KNEE );
	  s->ankle_command[LEFT] = 0.0;
	  s->ankle_command[RIGHT] = 0.0;
	}
      break;
    case RL_STANCE:
      if ( s->gndforce[RIGHT][ZZ] <= 0.001 )
	{
	  s->controller_state = R_SWING;
	  s->state_start_time = s->time;
	  if ( controller_print )
	    printf( "R_SWING %g\n", s->time );
	  reset_desireds( s );
	  setup_swing_knots( s, R_HIP, R_KNEE );
	  last_r_foot_x = r_foot_x;
	  s->ankle_command[LEFT] = s->stance_ankle_torque;
	  s->ankle_command[RIGHT] = 0.0;
	}
      break;
    case R_SWING:
      if ( s->gndforce[RIGHT][ZZ] > 0.0 
	   && s->state_elapsed_time > s->swing_time/2 )
	{
	  s->controller_state = LR_STANCE;
	  s->state_start_time = s->time;
	  if ( controller_print )
	    printf( "LR_STANCE %g\n", s->time );
	  reset_desireds( s );
	  zero_velocity_desireds( s, RIGHT );
	  setup_stance_knots( s, R_HIP, R_KNEE );
	  s->ankle_command[LEFT] = 0.0;
	  s->ankle_command[RIGHT] = 0.0;
	}
      break;
    case LR_STANCE:
      if ( s->gndforce[LEFT][ZZ] <= 0.001 )
	{
	  s->controller_state = L_SWING;
	  s->state_start_time = s->time;
	  if ( controller_print )
	    printf( "L_SWING %g\n", s->time );
	  reset_desireds( s );
	  setup_swing_knots( s, L_HIP, L_KNEE );
	  last_l_foot_x = l_foot_x;
	  s->ankle_command[LEFT] = 0.0;
	  s->ankle_command[RIGHT] = s->stance_ankle_torque;
	}
      break;
    case STOP:
      break;
    default:
      fprintf( stderr, "Unknown state %d in run_state_machine.\n",
	       s->controller_state );
      exit( -1 );
    }
}

/*****************************************************************************/

controller( SIM *s )
{
  int i;

  /* Kinematics already computed by SDFAST */
  /* Forces already computed by SDFAST */

  run_state_machine( s );

  /* Keep track of foot location */
  l_foot_x = s->foot[LEFT][XX];
  r_foot_x = s->foot[RIGHT][XX];

  switch( s->controller_state )
    {
    case WAITING:
      break;
    case LAUNCH1:
      break;
    case LAUNCH2:
      break;
    case L_SWING:
      s->step_length = r_foot_x - last_r_foot_x;
      break;
    case RL_STANCE:
      s->step_length = l_foot_x - last_l_foot_x;
      break;
    case R_SWING:
      s->step_length = l_foot_x - last_l_foot_x;
      break;
    case LR_STANCE:
      s->step_length = r_foot_x - last_r_foot_x;
      break;
    case STOP:
      break;
    default:
      fprintf( stderr, "Unknown state %d in controller.\n",
	       s->controller_state );
      exit( -1 );
    }
  run_trajectory( s, (float) (s->time) );

  s->last_time = s->time;

  run_servos( s );

  update_score( s );
}

/*****************************************************************************/
