/*****************************************************************************/
/*
  parameters.c: manage simulation parameters
*/
/*****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <malloc.h>
#include <string.h>

#include "main.h"
#include "parameters.h"

/*****************************************************************************/

void
init_default_parameters( SIM *s )
{
  int i;

  /* Overall parameters */
  s->total_duration = 5.0;
  // s->total_duration = 0.01;
  s->time_step = 0.001;

  /* Objective function */
  s->desired_speed = 0.0;
  s->speed_penalty_weight = 1000.0;
  s->torque_penalty_weight = 1/20000.0;

  s->rand_scale = 0.0;
  s->rand_seed = 1;
  s->duration = s->total_duration;
  s->slowdown = 0.0;

  s->k_hip = 500.0;
  s->b_hip = 10.0;
  s->k_knee = 500.0;
  s->b_knee = 10.0;
  s->k_ankle = 500.0;
  s->b_ankle = 10.0;
  /*
  s->k_hip = 50.0;
  s->b_hip = 10.0;
  s->k_knee = 50.0;
  s->b_knee = 10.0;
  s->k_ankle = 100.0;
  s->b_ankle = 10.0;
  */

  // contact impedance parameters
  s->gnd_k_z = 300000.0;
  s->gnd_b_z = 7000.0;
  s->gnd_k_x = 1000000.0;
  s->gnd_b_x = 200000.0;

/*
Assume biped is standing.
Torso CoM is measured from hip.
Thigh CoM is measured from hip.
Calf CoM is measured from knee.

Torso (T)
Left Thigh (LT)
Left Calf (LC)
Right Thigh (RT)
Right Calf (RC)

     |
     |
     | T
     |
   __|__
LT |   | RT
   |   |
   *   *
   |   |
LC |   | RC

The feet are point feet, which can exert a limited torque while on the
ground (+/-74 Newton-meters).

Knees bend both ways.

lengths:
head to hip: 0.8m
hip to knee: 0.3918m
knee to ankle: 0.3810m

calf inertial parameters includes foot.

mass:
torso: 50kg
thigh: 5.676kg
calf: 6.8952kg

moment of inertia (about center of mass):
torso: 1.5kg-m^2
thigh: 0.09592kg-m^2
calf: 0.1535kg-m^2

center of mass
torso: 0.2868m above hip
thigh: 0.1898m below hip
calf: 0.2384m below knee

body COM location relative to hip:
(0.2868*50 - 0.1898*5.676*2 - (0.3918+ 0.2384)*6.8952*2)/(50+5.676*2+6.8952*2)
= 0.0465
a little low relative to a human, should be around 0.1
*/

  s->torso_length = 0.8;
  s->thigh_length = 0.3918;
  s->calf_length = 0.3810;
  s->torso_mass = 50;
  s->thigh_mass = 5.676;
  s->calf_mass = 6.8952;
  s->torso_I = 1.5;
  s->thigh_I = 0.09592;
  s->calf_I = 0.1535;
  s->torso_cm = 0.2868;
  s->thigh_cm = 0.1898;
  s->calf_cm = 0.2384;

  for( i = 0; i < 3; i++ )
    {
      s->head_offset[i] = 0;
      s->hip_offset[i] = 0;
      s->knee_offset[i] = 0;
      s->foot_offset[i] = 0;
    }
  s->head_offset[ZZ] = s->torso_length - s->torso_cm;
  s->hip_offset[ZZ] = -s->torso_cm;
  s->knee_offset[ZZ] = s->calf_cm;
  s->foot_offset[ZZ] = -(s->calf_length - s->calf_cm);
}

/*****************************************************************************/

void
apply_parameters( float *p, SIM *s )
{
}

/*****************************************************************************/
/******************************************************************/

PARAMETER *malloc_parameter()
{
  PARAMETER *result;

  result = malloc( sizeof( PARAMETER ) );
  if ( result == NULL )
    {
      fprintf( stderr, "No memory to allocate PARAMETER.\n" );
      exit( -1 );
    }
  result->name = NULL;
  result->value = 0;
  result->optimize = 0;
  result->regularize = 0;
  result->nominal_value = 0;
  result->regularize_weight = 0;
  result->next = NULL;
  return result;
}

/******************************************************************/

PARAMETER *read_parameter_file( char *filename )
{
  FILE *stream;
  char buffer[10000];
  PARAMETER *p, *top, *bottom;

  top = NULL;
  bottom = NULL;
  p = NULL;
  stream = fopen( filename, "r" );
  if ( stream == NULL )
    {
      fprintf( stderr, "Can't open %s\n", filename );
      exit( -1 );
    }
  for(;;)
    {
      /* get name of variable */
      if ( fscanf( stream, "%s", buffer ) < 1 )
	break;
      p = malloc_parameter();
      p->name = strdup( buffer );
      if ( fscanf( stream, "%lf", &(p->value) ) < 1 )
	{
	  fprintf( stderr, "Missing value for %s in %s\n",
		   p->name, filename );
	  exit( -1 );
	}

      /* read various commands */
      for( ; ; )
	{
	  if ( fscanf( stream, "%s", buffer ) < 1 )
	    {
	      fprintf( stderr, "Missing end for %s in %s\n",
		       p->name, filename );
	      exit( -1 );
	    }
	  if ( strcmp( buffer, "end" ) == 0 )
	    break;
	  if ( strcmp( buffer, "opt" ) == 0 )
	    p->optimize = 1;
	}

      if ( top == NULL )
	top = p;
      if ( bottom != NULL )
	bottom->next = p;
      bottom = p;
    }
  fclose( stream );
  return top;
}

/******************************************************************/

write_parameter_file( FILE *stream, PARAMETER *p )
{
  for( ; ; )
    {
      if ( p == NULL )
	break;
      fprintf( stream, "%s %lg", p->name, p->value );
      if ( p->optimize )
	fprintf( stream, " opt" );
      fprintf( stream, " end\n" );
      p = p->next;
    }
}

/******************************************************************/
/******************************************************************/

int process_parameters( PARAMETER *p, SIM *s, int verbose )
{
  int total = 0;

  for( ; ; )
    {
      if ( p == NULL )
	break;

      /*
      else if ( strcmp( p->name, "" ) == 0 )
	p->pointer = &(s->);
      */

      if ( 1 )
	;
      else 
	{
	  fprintf( stderr, "Don't recognize %s in process_parameters()\n", p->name );
	  exit( -1 );
	}
      *(p->pointer) = p->value;
      if ( p->optimize )
	{
	  if ( verbose )
	    printf( "%d: %s\n", total, p->name );
	  total++;
	}
      p = p->next;
    }

  if ( verbose )
    printf( "Optimizing %d parameters.\n", total );
  return total;
}

/************************************************************************/

void
parameters_to_vector( PARAMETER *p, float *v )
{
  int indx = 0;

  printf( "Parameters_to_vector:\n" );
  for( ; ; )
    {
      if ( p == NULL )
	break;
      if ( p->optimize )
	{
	  v[indx++] = *(p->pointer);
	  printf( "%d: %g\n", indx-1, v[indx-1] );
	}
      p = p->next;
    }
}

/************************************************************************/

void
vector_to_sim( float *v, int n_parameters, PARAMETER *p )
{
  int indx;

  for( indx = 0; indx < n_parameters; indx++ )
    {
      /* Find next parameter to optimize */
      for( ; ; )
	{
	  if ( p == NULL )
	    {
	      fprintf( stderr, "Failure in vector_to_parameters: %d\n",
		       indx );
	      exit( - 1 );
	    }
	  if ( p->optimize )
	    break;
	  p = p->next;
	}
      *(p->pointer) = v[indx];
      /*
      printf( "%d: %g\n", indx, v[indx] );
      */
      p = p->next;
    }
}

/******************************************************************/

write_param_file( char *filename, PARAMETER *p )
{
  FILE *stream;

  stream = fopen( filename, "w" );
  if ( stream == NULL )
    {
      fprintf( stderr, "Can't open %s for write.\n", filename );
      exit( -1 );
    }

  for( ; ; )
    {
      if ( p == NULL )
	break;
      fprintf( stream, "%s %18.12f", p->name, *(p->pointer) );
      if ( p->optimize )
	fprintf( stream, " opt" );
      fprintf( stream, " end\n" );
      p = p->next;
    }

  fclose( stream );
}

/*****************************************************************************/
/******************************************************************/

/*
main()
{
  PARAMETER *p;

  p = read_parameter_file( "test" );
  write_parameter_file( stdout, p ); 
}
*/

/******************************************************************/

