/************************************************************************/
/*
  dynamics.c: This is where the numerical integration and SDFAST stuff is done.

Is this done or not?
handle sdfast flag correctly (only b1gstate sets it).
*/
/************************************************************************/
/*
In this version we are going to use the following conventions (slightly
different from the SDFAST code generated from sdfast/b1g.sd):
x forward
y left
z up
pitch + pitches the torso forward (head ahead of hips)
hip +
  moving the torso forward for a fixed vertical thigh, 
  moving the thigh forward for a fixed vertical torso.
knee +
  moving the thigh forward for a fixed vertical calf, 
  moving the calf forward for a fixed vertical thigh.
ankle +
  moving the calf forward for a fixed horizontal foot,
  moving the toe upward for a fixed vertical calf.
*/
/************************************************************************/

#include <stdio.h>
#include <math.h>

#include "main.h"
#include "sdfast/b1g.h"

/************************************************************************/

SIM sim;

/************************************************************************/

/* SDFAST constants and variables */
#define NQ 7
#define NU 7
#define NSTATE (NQ+NU)
#define CTOL	1e-5	/* constraint tolerance */
#define GND -1
const double baumgarte=100.0; /* could be 10.0 - 1000.0 */

/* Here we define the state */
double xt[NSTATE];
/* Here we define the derivative of the state */
static double dxt[NSTATE];
/* Leg state machines */
const double GROUND_LEVEL = 0.0;

/************************************************************************/

int compute_ankle_angles( SIM *s )
{
  s->ankle_angle[LEFT] = s->pitch 
    - s->hip_angle[LEFT] - s->knee_angle[LEFT];
  s->ankle_angle[RIGHT] = s->pitch 
    - s->hip_angle[RIGHT] - s->knee_angle[RIGHT];
  s->ankle_angled[LEFT] = s->pitchd 
    - s->hip_angled[LEFT] - s->knee_angled[LEFT];
  s->ankle_angled[RIGHT] = s->pitchd 
    - s->hip_angled[RIGHT] - s->knee_angled[RIGHT];
  return 0;
}

/************************************************************************/

static void forward_kinematics( SIM *s )
{
  int i;

  b1gpos(BODY_TORSO, s->head_offset, s->head);    
  b1gpos(BODY_TORSO, s->hip_offset, s->hip);    
  b1gpos(BODY_L_CALF, s->knee_offset, &(s->knee[LEFT][0]));    
  b1gpos(BODY_R_CALF, s->knee_offset, &(s->knee[RIGHT][0]));    
  b1gpos(BODY_L_CALF, s->foot_offset, &(s->foot[LEFT][0]));    
  b1gpos(BODY_R_CALF, s->foot_offset, &(s->foot[RIGHT][0]));    

  /*
  printf( "head: %g\n", s->head[ZZ] );
  printf( "hip: %g\n", s->hip[ZZ] );
  printf( "kneeL: %g\n", s->knee[LEFT][ZZ] );
  printf( "footL: %g\n", s->foot[LEFT][ZZ] );
  */

  b1gvel(BODY_TORSO, s->head_offset, s->headd);    
  b1gvel(BODY_TORSO, s->hip_offset, s->hipd);    
  b1gvel(BODY_L_CALF, s->knee_offset, &(s->kneed[LEFT][0]));    
  b1gvel(BODY_R_CALF, s->knee_offset, &(s->kneed[RIGHT][0]));    
  b1gvel(BODY_L_CALF, s->foot_offset, &(s->footd[LEFT][0]));    
  b1gvel(BODY_R_CALF, s->foot_offset, &(s->footd[RIGHT][0]));    

  b1gpos(BODY_L_CALF, s->foot_offset, s->footpos[LEFT]);    
  b1gpos(BODY_R_CALF, s->foot_offset, s->footpos[RIGHT]);    

  b1gvel(BODY_L_CALF, s->foot_offset, s->footvel[LEFT]);    
  b1gvel(BODY_R_CALF, s->foot_offset, s->footvel[RIGHT]);    

  s->pitch = s->state_sdfast[Q_PITCH];
  s->hip_angle[LEFT] = -s->state_sdfast[Q_L_HIP];
  s->hip_angle[RIGHT] = -s->state_sdfast[Q_R_HIP];
  s->knee_angle[LEFT] = -s->state_sdfast[Q_L_KNEE];
  s->knee_angle[RIGHT] = -s->state_sdfast[Q_R_KNEE];

  s->pitchd = s->state_sdfast[QD_PITCH];
  s->hip_angled[LEFT] = -s->state_sdfast[QD_L_HIP];
  s->hip_angled[RIGHT] = -s->state_sdfast[QD_R_HIP];
  s->knee_angled[LEFT] = -s->state_sdfast[QD_L_KNEE];
  s->knee_angled[RIGHT] = -s->state_sdfast[QD_R_KNEE];

  compute_ankle_angles( s );

  s->torso_abs_angle = 
    -atan2( s->head[XX] - s->hip[XX], s->head[ZZ] - s->hip[ZZ] );
  for( i = 0; i < 2; i++ )
    {
      s->thigh_abs_angle[i] = 
	-atan2( s->hip[XX] - s->knee[i][XX], s->hip[ZZ] - s->knee[i][ZZ] );
      s->calf_abs_angle[i] = 
	-atan2( s->knee[i][XX] - s->foot[i][XX], s->knee[i][ZZ] - s->foot[i][ZZ] );
    }
}

/************************************************************************/
/* Initialize the state vector */

void init_dynamics_state( SIM *s, double *state )
{
  int i, j;

  for( i = 0; i < NSTATE; i++ )
    {
      xt[i] = state[i];
      dxt[i] = 0;
    }

  for( i = 0, j = NQ; i < NU; i++, j++ )
    { 
      dxt[i] = xt[j];
    }

  b1gstate( 0.0, xt, dxt );

  forward_kinematics( s );
}

/************************************************************************/

b1gdump()
{
  double value;
  double vector[3];
  double inertia[3][3];

  b1ggetmass( BODY_TORSO, &value );
  printf( "Torso mass: %g\n", value );
  b1ggetmass( BODY_L_THIGH, &value );
  printf( "L Thigh mass: %g\n", value );
  b1ggetmass( BODY_R_THIGH, &value );
  printf( "R Thigh mass: %g\n", value );
  b1ggetmass( BODY_L_CALF, &value );
  printf( "L Calf mass: %g\n", value );
  b1ggetmass( BODY_R_CALF, &value );
  printf( "R Calf mass: %g\n", value );

  b1ggetiner( BODY_TORSO, inertia );
  printf( "Torso I: %g\n", inertia[YY][YY] );
  b1ggetiner( BODY_L_THIGH, inertia );
  printf( "L Thigh I: %g\n", inertia[YY][YY] );
  b1ggetiner( BODY_R_THIGH, inertia );
  printf( "R Thigh I: %g\n", inertia[YY][YY] );
  b1ggetiner( BODY_L_CALF, inertia );
  printf( "L Calf I: %g\n", inertia[YY][YY] );
  b1ggetiner( BODY_R_CALF, inertia );
  printf( "R Calf I: %g\n", inertia[YY][YY] );

  b1ggetbtj( BODY_L_THIGH, vector );
  printf( "L_THIGH BTJ: %g %g %g\n", vector[0],
	  vector[1], vector[2] );
  b1ggetbtj( BODY_R_THIGH, vector );
  printf( "R_THIGH BTJ: %g %g %g\n", vector[0],
	  vector[1], vector[2] );
  b1ggetbtj( BODY_L_CALF, vector );
  printf( "L_Calf BTJ: %g %g %g\n", vector[0],
	  vector[1], vector[2] );
  b1ggetbtj( BODY_R_CALF, vector );
  printf( "R_Calf BTJ: %g %g %g\n", vector[0],
	  vector[1], vector[2] );

  b1ggetitj( BODY_L_THIGH, vector );
  printf( "L_THIGH ITJ: %g %g %g\n", vector[0],
	  vector[1], vector[2] );
  b1ggetitj( BODY_R_THIGH, vector );
  printf( "R_THIGH ITJ: %g %g %g\n", vector[0],
	  vector[1], vector[2] );
  b1ggetitj( BODY_L_CALF, vector );
  printf( "L_Calf ITJ: %g %g %g\n", vector[0],
	  vector[1], vector[2] );
  b1ggetitj( BODY_R_CALF, vector );
  printf( "R_Calf ITJ: %g %g %g\n", vector[0],
	  vector[1], vector[2] );
}

/************************************************************************/
/* Initialize sdfast  */

void init_dynamics( SIM *s )
{
  int i, j;
  double vector[3];
  double inertia[3][3];

  s->status = OK;

  // b1gdump();

  b1gmass( BODY_TORSO, s->torso_mass );
  b1gmass( BODY_L_THIGH, s->thigh_mass );
  b1gmass( BODY_R_THIGH, s->thigh_mass );
  b1gmass( BODY_L_CALF, s->calf_mass );
  b1gmass( BODY_R_CALF, s->calf_mass );

  for ( i = 0; i < 3; i++ )
    {
      vector[i] = 0;
      for ( j = 0; j < 3; j++ )
	{
	  if ( i == j )
	    inertia[i][i] = 1.0;
	  else
	    inertia[i][j] = 0;
	}
    }
  inertia[YY][YY] = s->torso_I;
  b1giner( BODY_TORSO, inertia );
  inertia[YY][YY] = s->thigh_I;
  b1giner( BODY_L_THIGH, inertia );
  b1giner( BODY_R_THIGH, inertia );
  inertia[YY][YY] = s->calf_I;
  b1giner( BODY_L_CALF, inertia );
  b1giner( BODY_R_CALF, inertia );

  vector[ZZ] = -s->torso_cm;
  b1gbtj( BODY_TORSO, vector );
  vector[ZZ] = s->thigh_cm;
  b1gbtj( BODY_L_THIGH, vector );
  b1gbtj( BODY_R_THIGH, vector );
  vector[ZZ] = s->calf_cm;
  b1gbtj( BODY_L_CALF, vector );
  b1gbtj( BODY_R_CALF, vector );

  vector[ZZ] = -s->torso_cm;
  b1gitj( BODY_L_THIGH, vector );
  b1gitj( BODY_R_THIGH, vector );

  vector[ZZ] = -(s->thigh_length - s->thigh_cm);
  b1gitj( BODY_L_CALF, vector );
  b1gitj( BODY_R_CALF, vector );

  // b1gdump();

  b1ginit(); /* initialize SDFAST model */

  b1gstab(2.0*baumgarte, baumgarte*baumgarte); 

  for( i = 0; i < NSTATE; i++ )
    s->state_sdfast[i] = 0;

  b1gprinterr(stderr);
}

/************************************************************************/
/* Need to handle constraints and xd, zd, and pitchd requests correctly */

void init_state_one_foot_on_ground( SIM *s )
{
  double min;

  s->status = OK;

  s->state_sdfast[Q_X] = s->hip[XX];
  s->state_sdfast[Q_Z] = s->hip[ZZ];
  s->state_sdfast[Q_PITCH] = s->pitch;
  s->state_sdfast[Q_L_HIP] = -s->hip_angle[LEFT];
  s->state_sdfast[Q_R_HIP] = -s->hip_angle[RIGHT];
  s->state_sdfast[Q_L_KNEE] = -s->knee_angle[LEFT];
  s->state_sdfast[Q_R_KNEE] = -s->knee_angle[RIGHT];

  s->state_sdfast[QD_X] = s->hipd[XX];
  s->state_sdfast[QD_Z] = s->hipd[ZZ];
  s->state_sdfast[QD_PITCH] = s->pitchd;
  s->state_sdfast[QD_L_HIP] = -s->hip_angled[LEFT];
  s->state_sdfast[QD_R_HIP] = -s->hip_angled[RIGHT];
  s->state_sdfast[QD_L_KNEE] = -s->knee_angled[LEFT];
  s->state_sdfast[QD_R_KNEE] = -s->knee_angled[RIGHT];

  init_dynamics_state( s, s->state_sdfast );

  /* Replace with bogus code
  min = s->foot[LEFT][ZZ];
  if ( min > s->foot[RIGHT][ZZ] )
    min = s->foot[RIGHT][ZZ];
  if ( min < -0.003 )
    s->state_sdfast[Q_Z] = s->state_sdfast[Q_Z] - min - 0.003;
  */
  /*
  printf( "%g %g\n", s->foot[LEFT][ZZ], s->foot[RIGHT][ZZ] );
  */
  min = s->foot[LEFT][ZZ];
  if ( min < s->foot[RIGHT][ZZ] )
    min = s->foot[RIGHT][ZZ];
  if ( min < -0.000 )
    s->state_sdfast[Q_Z] = s->state_sdfast[Q_Z] - min - 0.000;

  init_dynamics_state( s, s->state_sdfast );
}

/************************************************************************/
/* Need to handle constraints and xd, zd, and pitchd requests correctly */

void init_state_two_feet_on_ground( SIM *s )
{
  double min;
  double offset;

  s->status = OK;

  s->state_sdfast[Q_X] = s->hip[XX];
  s->state_sdfast[Q_Z] = s->hip[ZZ];
  s->state_sdfast[Q_PITCH] = s->pitch;
  s->state_sdfast[Q_L_HIP] = -s->hip_angle[LEFT];
  s->state_sdfast[Q_R_HIP] = -s->hip_angle[RIGHT];
  s->state_sdfast[Q_L_KNEE] = -s->knee_angle[LEFT];
  s->state_sdfast[Q_R_KNEE] = -s->knee_angle[RIGHT];

  s->state_sdfast[QD_X] = s->hipd[XX];
  s->state_sdfast[QD_Z] = s->hipd[ZZ];
  s->state_sdfast[QD_PITCH] = s->pitchd;
  s->state_sdfast[QD_L_HIP] = -s->hip_angled[LEFT];
  s->state_sdfast[QD_R_HIP] = -s->hip_angled[RIGHT];
  s->state_sdfast[QD_L_KNEE] = -s->knee_angled[LEFT];
  s->state_sdfast[QD_R_KNEE] = -s->knee_angled[RIGHT];

  init_dynamics_state( s, s->state_sdfast );

  /* Need to handle case where two feet together */

  if ( s->foot[RIGHT][XX] - s->foot[LEFT][XX] > 0 )
    offset = atan2( s->foot[RIGHT][ZZ] - s->foot[LEFT][ZZ],
		    s->foot[RIGHT][XX] - s->foot[LEFT][XX] );
  else
    offset = atan2( s->foot[LEFT][ZZ] - s->foot[RIGHT][ZZ],
		    s->foot[LEFT][XX] - s->foot[RIGHT][XX] );

  /*
  printf( "two feet on ground: %g %g %g %g: %g\n", 
	  s->foot[RIGHT][ZZ], s->foot[LEFT][ZZ],
	  s->foot[RIGHT][XX], s->foot[LEFT][XX], offset );
  */

  s->state_sdfast[Q_PITCH] += offset;

  init_dynamics_state( s, s->state_sdfast );

  /*
  printf( "%g %g\n", s->foot[RIGHT][ZZ], s->foot[LEFT][ZZ] );
  */

  min = s->foot[LEFT][ZZ];
  if ( min > s->foot[RIGHT][ZZ] )
    min = s->foot[RIGHT][ZZ];
  s->state_sdfast[Q_Z] = s->state_sdfast[Q_Z] - min;

  init_dynamics_state( s, s->state_sdfast );

  /*
  printf( "%g %g\n", s->foot[RIGHT][ZZ], s->foot[LEFT][ZZ] );
  */
}

/************************************************************************/
/* This is what is called on each integration step */

void integrate_one_time_step( SIM *s, double *action  )
{ 
  int i;
  int flag; /* set to 1 when things are changed or first call */
  int err; 
    /* { OK, DERIVATIVE_DISCONTINUITY, SYSTEM_LOCKED, CONSTRAINTS_ERR } */
  double errest;

  if ( s->hip[ZZ] < 0.2 )
    {
      s->status = CRASHED;
      s->time += s->time_step;
      return;
    }

  flag=1;

  // clear outstanding error flags
  b1gclearerr();

  for( i = 0; i < 20; i++ )
    {
      b1gfmotion(&(sim.time),xt,dxt,sim.time_step/20.0,CTOL,&flag,&errest,&err);
      b1gprinterr(stderr);
    }

  for( i = 0; i < NSTATE; i++ )
    s->state_sdfast[i] = xt[i];

  forward_kinematics( s );

  /*
  printf( "Press return to continue.\n" );
  getchar();
  */
}

/****************************************************************/

// Utility function to compute the ground force function.
static 
void compute_ground_force(double contactpos[3],  // contact position wrt world
			  double contact_zero[3], // original conact point.
			  double contactvel[3],  // contact velocity wrt world
			  double bodypos[3],     // contact point wrt body
			  int body,            
			  double force[3])       // resultant force vector wrt world
			  
{
  int i;

  // force only exists if foot is below ground level
  if (contactpos[2] < 0.0) {
    double fvec[3];  // ground force vector in body coordintes

    // Compute the desired ground force vector in the world
    // coordinates.  For now, the lateral force is just viscous.
    // It should really be a constraint equation.
    // CGA: Actually keep track of original contact point. */
    if ( contact_zero[ZZ] > 0.0 )
      for( i = 0; i < 3; i++ )
	contact_zero[i] = contactpos[i];

    force[0] = (contact_zero[0] - contactpos[0])*sim.gnd_k_x
						 -contactvel[0] * sim.gnd_b_x;
    force[1] = 0.0;   // not relevant
    force[2] = (-contactpos[2] * sim.gnd_k_z) - (contactvel[2] * sim.gnd_b_z);

    // ground can't pull down
    if (force[2] < 0) 
      {
	force[0] = 0.0;
	force[1] = 0.0;
	force[2] = 0.0;
	/* Slipping */
	for( i = 0; i < 3; i++ )
	  contact_zero[i] = contactpos[i];
      }

    // transform the vector into body coordinates
    b1gtrans(BODY_WORLD, force, body, fvec);

    // apply to the model
    b1gpointf(body, bodypos, fvec);

  } else {
    force[0] = force[1] = force[2] = 0.0;
    for( i = 0; i < 3; i++ )
      contact_zero[i] = contactpos[i];
  }
}
/************************************************************************/
/* SDFAST stuff */
/************************************************************************/
/* This is where the control (hip_torque) is applied. May be called many
times per integration step at any time or state. */

void b1guforce(double t, double *q, double *u)
{
  double torque[3];

  // find the position and velocity of each foot point in the world frame
  b1gpos(BODY_L_CALF, sim.foot_offset, sim.footpos[LEFT]);    
  b1gpos(BODY_R_CALF, sim.foot_offset, sim.footpos[RIGHT]);    

  b1gvel(BODY_L_CALF, sim.foot_offset, sim.footvel[LEFT]);    
  b1gvel(BODY_R_CALF, sim.foot_offset, sim.footvel[RIGHT]);    

  // Apply ground forces to each foot as point forces.

  compute_ground_force(sim.footpos[LEFT], sim.foot_zero[LEFT], sim.footvel[LEFT],
		       sim.foot_offset, BODY_L_CALF, sim.gndforce[LEFT]);
  compute_ground_force(sim.footpos[RIGHT], sim.foot_zero[RIGHT], sim.footvel[RIGHT],
		       sim.foot_offset, BODY_R_CALF, sim.gndforce[RIGHT]);

  b1ghinget( 1, 0, -sim.hip_torque[LEFT] );
  b1ghinget( 2, 0, -sim.knee_torque[LEFT] );
  b1ghinget( 3, 0, -sim.hip_torque[RIGHT] );
  b1ghinget( 4, 0, -sim.knee_torque[RIGHT] );
  /* Apply ankle torques */
  torque[XX] = torque[ZZ] = 0;
  torque[YY] = sim.ankle_torque[LEFT];
  b1gbodyt( BODY_L_CALF, torque );
  torque[YY] = sim.ankle_torque[RIGHT];
  b1gbodyt( BODY_R_CALF, torque );
}

/************************************************************************/
/************************************************************************/
/************************************************************************/
/************************************************************************/
