/*****************************************************************************/
/*
  display.c: animate a data file.
*/
/*****************************************************************************/
/* This version flips X and angles to get lighting right */
#define XFLIP -
#define XXFLIP 
/*****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>

#include "drawstuff.h" /* Graphics stuff */
#include "drawstuff-cga.h" /* CGA stuff to make things clearer */

#include "../useful/mrdplot/mrdplot.h"
#include "sdfast/b1g.h"
#include "main.h"

/*****************************************************************************/

enum channels {
  TIME,
  HEAD_X,
  HEAD_Z,
  HIP_X,
  HIP_Z,
  L_KNEE_X,
  L_KNEE_Z,
  L_FOOT_X,
  L_FOOT_Z,
  TORSO_ABS_ANGLE,
  L_THIGH_ABS_ANGLE,
  L_CALF_ABS_ANGLE,
  N_CHANNELS,
};

#define CALF_L_LENGTH (0.3810f) // length of link
#define CALF_L_WIDTH (0.05f)    // width of link

#define THIGH_L_LENGTH (0.3918f) // length of link
#define THIGH_L_WIDTH (0.05f)    // width of link

#define TORSO_LENGTH (0.8f)     // length of link
#define TORSO_WIDTH (0.05f)    // width of link

/*****************************************************************************/

SIM sim;

MRDPLOT_DATA *d;

int channels[N_CHANNELS];

/*****************************************************************************/
/*****************************************************************************/

double lookup_value( MRDPLOT_DATA *d, int point, int channel )
{
  if ( channels[channel] < 0 || channels[channel] >= d->n_channels )
    {
      fprintf( stderr, "Bad channel: %d %d\n", channel, channels[channel] );
      exit( -1 );
    }
  return d->data[point*d->n_channels + channels[channel]];
}

/********************************************************************/
// start(): set things up

static void start()
{
  // set up view point
  static float xyz[3] = {0.0f,1.4f,0.8f};
  static float hpr[3] = {-90.0000f,0.0000f,0.0000f};
  dsSetViewpoint (xyz,hpr);
}

/********************************************************************/
// called when a key pressed

static void command (int cmd)
{
  // don't handle user input yet.
  dsPrint ("received command %d (`%c')\n",cmd,cmd);
}

/*****************************************************************************/

static void display (int pause)
{
  static int index = 0;
  float f;
  float center[3];
  float R[12];
  int i;
  float calf_l_sides[3] = {CALF_L_WIDTH,CALF_L_WIDTH,CALF_L_LENGTH};
  float thigh_l_sides[3] = {THIGH_L_WIDTH,THIGH_L_WIDTH,THIGH_L_LENGTH};
  float torso_sides[3] = {TORSO_WIDTH,TORSO_WIDTH,TORSO_LENGTH};

  /* Calf */
  center[XX] = XFLIP (lookup_value( d, index, L_KNEE_X )
		+ lookup_value( d, index, L_FOOT_X ))/2;
  center[YY] = 0;
  center[ZZ] = (lookup_value( d, index, L_KNEE_Z )
		+ lookup_value( d, index, L_FOOT_Z ))/2;

  for ( i = 0; i < 12; i++ )
    R[i] = 0;
  R[0] = cos( lookup_value( d, index, L_CALF_ABS_ANGLE ) );
  R[10] = R[0];
  R[2] = XXFLIP sin( lookup_value( d, index, L_CALF_ABS_ANGLE ) );
  R[8] = - R[2];

  dsSetTexture (DS_WOOD);
  dsSetColor (1,1,0);
  dsDrawBox( center, R, calf_l_sides );

  /* Thigh */
  center[XX] = XFLIP (lookup_value( d, index, L_KNEE_X )
		+ lookup_value( d, index, HIP_X ))/2;
  center[YY] = 0;
  center[ZZ] = (lookup_value( d, index, L_KNEE_Z )
		+ lookup_value( d, index, HIP_Z ))/2;

  for ( i = 0; i < 12; i++ )
    R[i] = 0;
  R[0] = cos( lookup_value( d, index, L_THIGH_ABS_ANGLE ) );
  R[10] = R[0];
  R[2] = XXFLIP sin( lookup_value( d, index, L_THIGH_ABS_ANGLE ) );
  R[8] = - R[2];

  dsSetTexture (DS_WOOD);
  dsSetColor (0,0,1);
  dsDrawBox( center, R, thigh_l_sides );

  /* Torso */
  center[XX] = XFLIP (lookup_value( d, index, HEAD_X )
		+ lookup_value( d, index, HIP_X ))/2;
  center[YY] = 0;
  center[ZZ] = (lookup_value( d, index, HEAD_Z )
		+ lookup_value( d, index, HIP_Z ))/2;

  for ( i = 0; i < 12; i++ )
    R[i] = 0;
  R[0] = cos( lookup_value( d, index, TORSO_ABS_ANGLE ) );
  R[10] = R[0];
  R[2] = XXFLIP sin( lookup_value( d, index, TORSO_ABS_ANGLE ) );
  R[8] = - R[2];

  dsSetTexture (DS_WOOD);
  dsSetColor (1,0,0);
  dsDrawBox( center, R, torso_sides );

  /* Delay */
  for( f = 0.0; f < 1.0; f += 1.0 )
    ;

  index += 30;
  if ( index > d->n_points )
    {
      printf( ".\n" );
      index = 0;
    }
}

/*****************************************************************************/
/********************************************************************/

int main (int argc, char **argv)
{
  char *filename;
  int i;
  dsFunctions fn; 

  if ( argc > 1 )
    filename = strdup( argv[1] );
  else
    filename = last_data();
  
  printf( "Reading: %s\n", filename );

  d = read_mrdplot( filename );

  for( i = 0; i < N_CHANNELS; i++ )
    channels[i] = -1;

  channels[TIME] = find_channel( "time", d );
  channels[HEAD_X] = find_channel( "head_x", d );
  channels[HEAD_Z] = find_channel( "head_z", d );
  channels[HIP_X] = find_channel( "hip_x", d );
  channels[HIP_Z] = find_channel( "hip_z", d );
  channels[L_KNEE_X] = find_channel( "l_knee_x", d );
  channels[L_KNEE_Z] = find_channel( "l_knee_z", d );
  channels[L_FOOT_X] = find_channel( "l_foot_x", d );
  channels[L_FOOT_Z] = find_channel( "l_foot_z", d );
  channels[TORSO_ABS_ANGLE] = find_channel( "torso_abs_angle", d );
  channels[L_THIGH_ABS_ANGLE] = find_channel( "l_thigh_abs_angle", d );
  channels[L_CALF_ABS_ANGLE] = find_channel( "l_calf_abs_angle", d );

  for( i = 0; i < N_CHANNELS; i++ )
    printf( "%d %d\n", i, channels[i] );

  // setup pointers to drawstuff callback functions
  fn.version = DS_VERSION; 
  fn.start = &start; 
  fn.step = &display; 
  fn.command = &command; 
  fn.stop = 0; 
#ifdef WIN32
  fn.path_to_textures = "C:/cga/seegrid/cga-sim-windows-002/useful/drawstuff/textures";
#else
  fn.path_to_textures = "/home/cga/projects/useful/drawstuff/textures";
#endif

  // do display
  dsSimulationLoop( argc, argv, /* command line arguments */
                    352, 288, /* window size */
                    &fn ); /* callback info */

  return 0;
}

/********************************************************************/
