/*****************************************************************************/
/*
  controller.c: control strategy.
*/
/*****************************************************************************/

#include <stdio.h>
#include <math.h>
#include <stdlib.h>

#include "../useful/trajectory/trajectory.h"
#include "main.h"

/*****************************************************************************/

#define N_BIPED_DOFS 6

#define L_HIP 0
#define L_KNEE 1
#define L_ANKLE 2
#define R_HIP 3
#define R_KNEE 4
#define R_ANKLE 5

#define UNKNOWN_STATE 0
#define WAITING 1
#define LAUNCH1 2
#define LAUNCH2 3
#define L_SWING 4
#define RL_STANCE 5
#define R_SWING 6
#define LR_STANCE 7
#define STOP 8

/*****************************************************************************/
/* Control variables */
int print = 0;

/* storage. */
double torque_score = 0;
double crashed_penalty = 0;
float l_foot_x = 0;
float r_foot_x = 0;
float last_l_foot_x = 0;
float last_r_foot_x = 0;

/*****************************************************************************/
/*****************************************************************************/

DOFS desired_trajectory[N_BIPED_DOFS];
DOFS current_desireds[N_BIPED_DOFS];

/*****************************************************************************/
/*****************************************************************************/

void reset_score( SIM *s )
{
  torque_score = 0;
  crashed_penalty = 0;
}

/*****************************************************************************/

void update_score( SIM *s )
{
  torque_score += s->hip_command[LEFT]*s->hip_command[LEFT];
  torque_score += s->hip_command[RIGHT]*s->hip_command[RIGHT];
  torque_score += s->knee_command[LEFT]*s->knee_command[LEFT];
  torque_score += s->knee_command[RIGHT]*s->knee_command[RIGHT];
  torque_score += s->ankle_command[LEFT]*s->ankle_command[LEFT];
  torque_score += s->ankle_command[RIGHT]*s->ankle_command[RIGHT];
  if ( s->status == CRASHED )
    crashed_penalty += 0.001;
}

/*****************************************************************************/
/*
Could normalize torque and other penalties w.r.t. distance
Could penalize deviations from symmetry (if necessary).
*/

float get_score( SIM *s )
{
  float speed,speed_penalty;

  speed = s->hip[XX]/s->time;
  speed_penalty = (s->desired_speed - speed)*(s->desired_speed - speed)
    *s->speed_penalty_weight;
  torque_score *= s->torque_penalty_weight;
  printf( "Torque score: %lg; distance score: %g; speed %g; crashed %g\n",
	  torque_score, speed_penalty, speed, crashed_penalty );
  return (float) (torque_score + speed_penalty + crashed_penalty);
}

/*****************************************************************************/
/* call this many times to restart a simulation */

reinit_controller( SIM *s )
{

  s->controller_state = WAITING;
  s->state_start_time = s->time;
  s->state_elapsed_time = 0.0;
  s->desireds_index = s->time;
  s->last_time = s->time;

  s->hip_angle_d[LEFT] = s->hip_angle[LEFT];
  s->hip_angle_d[RIGHT] = s->hip_angle[RIGHT];
  s->knee_angle_d[LEFT] = s->knee_angle[LEFT];
  s->knee_angle_d[RIGHT] = s->knee_angle[RIGHT];
  s->ankle_angle_d[LEFT] = s->ankle_angle[LEFT];
  s->ankle_angle_d[RIGHT] = s->ankle_angle[RIGHT];

  /* sidestep angle perturbation for now
  s->hip_angled_d[LEFT] = s->hip_angled[LEFT];
  s->hip_angled_d[RIGHT] = s->hip_angled[RIGHT];
  s->knee_angled_d[LEFT] = s->knee_angled[LEFT];
  s->knee_angled_d[RIGHT] = s->knee_angled[RIGHT];
  s->ankle_angled_d[LEFT] = s->ankle_angled[LEFT];
  s->ankle_angled_d[RIGHT] = s->ankle_angled[RIGHT];
  */
  s->hip_angled_d[LEFT] = 0;
  s->hip_angled_d[RIGHT] = 0;
  s->knee_angled_d[LEFT] = 0;
  s->knee_angled_d[RIGHT] = 0;
  s->ankle_angled_d[LEFT] = 0;
  s->ankle_angled_d[RIGHT] = 0;

  init_trajectory( desired_trajectory, N_BIPED_DOFS );
  init_trajectory( current_desireds, N_BIPED_DOFS );

  /* add actual position as knot to current desireds */
  add_knot_point( current_desireds, L_HIP, QUINTIC_SPLINE, 0.0, 
		  s->hip_angle_d[LEFT], s->hip_angled_d[LEFT], 0.0 );
  add_knot_point( current_desireds, L_KNEE, QUINTIC_SPLINE, 0.0,
		  s->knee_angle_d[LEFT], s->knee_angled_d[LEFT], 0.0 );
  add_knot_point( current_desireds, L_ANKLE, QUINTIC_SPLINE, 0.0,
		  s->ankle_angle_d[LEFT], s->ankle_angled_d[LEFT], 0.0 );
  add_knot_point( current_desireds, R_HIP, QUINTIC_SPLINE, 0.0, 
		  s->hip_angle_d[RIGHT], s->hip_angled_d[RIGHT], 0.0 );
  add_knot_point( current_desireds, R_KNEE, QUINTIC_SPLINE, 0.0, 
		  s->knee_angle_d[RIGHT], s->knee_angled_d[RIGHT], 0.0 );
  add_knot_point( current_desireds, R_ANKLE, QUINTIC_SPLINE, 0.0, 
		  s->ankle_angle_d[RIGHT], s->ankle_angled_d[RIGHT], 0.0 );

  reset_score( s );
}

/*****************************************************************************/
/* call this many times to restart a simulation */

int reinit_sim( SIM *s )
{
  int i;
  double min;

  srand( s->rand_seed );

  s->time = 0.0;
  s->step_length = 0;

  s->hip[XX] = 0;
  s->hip[ZZ] = 0;
  s->pitch = 0;
  s->hipd[XX] = 0;
  s->hipd[ZZ] = 0;
  s->pitchd = 0;

  s->hip_angle[LEFT] = 0;
  s->hip_angle[RIGHT] = 0;
  s->knee_angle[LEFT] = 0;
  s->knee_angle[RIGHT] = 0;
  s->hip_angled[LEFT] = 0;
  s->hip_angled[RIGHT] = 0;
  s->knee_angled[LEFT] = 0;
  s->knee_angled[RIGHT] = 0;

  /* Make it fall */
  s->pitchd = 1.0;
  s->hip_angled[LEFT] = s->hip_angled[RIGHT] = 1.0;
  s->knee_angled[LEFT] = s->knee_angled[RIGHT] = -1.0;

  compute_ankle_angles( s );

  /* Indicate foot zero positions not set */
  s->foot_zero[LEFT][ZZ] = 1.0;
  s->foot_zero[RIGHT][ZZ] = 1.0;

  /*
  init_state_one_foot_on_ground( s );
  */
  init_state_two_feet_on_ground( s );

  reinit_controller( s );

  l_foot_x = s->foot[LEFT][XX];
  r_foot_x = s->foot[RIGHT][XX];
  last_l_foot_x = s->foot[LEFT][XX];
  last_r_foot_x = s->foot[RIGHT][XX];
}

/*****************************************************************************/
/* Call this once to do one time operations like memory allocation */

int init_sim( SIM *s )
{
  init_dynamics( s );
  reinit_sim( s );
}

/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/

void
run_trajectory( SIM *s, float time )
{
  float position, velocity, acceleration;

  if ( !lookup_spline3( desired_trajectory, L_HIP, time,
			current_desireds,
			&position,
			&velocity,
			&acceleration ) )
    {
      printf( "Lookup failure.\n" );
      exit( -1 );
    }
  /* update current desired position */
  set_first_knot_to( current_desireds, L_HIP, QUINTIC_SPLINE, time,
		     position, velocity, acceleration );
  s->hip_angle_d[LEFT] = position;
  s->hip_angled_d[LEFT] = velocity;

  if ( !lookup_spline3( desired_trajectory, L_KNEE, time,
			current_desireds,
			&position,
			&velocity,
			&acceleration ) )
    {
      printf( "Lookup failure.\n" );
      exit( -1 );
    }
  /* update current desired position */
  set_first_knot_to( current_desireds, L_KNEE, QUINTIC_SPLINE, time,
		     position, velocity, acceleration );
  s->knee_angle_d[LEFT] = position;
  s->knee_angled_d[LEFT] = velocity;

  if ( !lookup_spline3( desired_trajectory, L_ANKLE, time,
			current_desireds,
			&position,
			&velocity,
			&acceleration ) )
    {
      printf( "Lookup failure.\n" );
      exit( -1 );
    }
  /* update current desired position */
  set_first_knot_to( current_desireds, L_ANKLE, QUINTIC_SPLINE, time,
		     position, velocity, acceleration );
  s->ankle_angle_d[LEFT] = position;
  s->ankle_angled_d[LEFT] = velocity;

  if ( !lookup_spline3( desired_trajectory, R_HIP, time,
			current_desireds,
			&position,
			&velocity,
			&acceleration ) )
    {
      printf( "Lookup failure.\n" );
      exit( -1 );
    }
  /*
  if ( s->time > 1.65 )
    {
      printf( "position, velocity, acceleration: %g %g %g\n",
	      position, velocity, acceleration );
      printf( "current desireds.\n" );
      print_trajectory( current_desireds, 4 );
      printf( "desired trajectory.\n" );
      print_trajectory( desired_trajectory, 4 );
      printf( "Press return to continue.\n" );
      getchar();
    }
  */

  /* update current desired position */
  set_first_knot_to( current_desireds, R_HIP, QUINTIC_SPLINE, time,
		     position, velocity, acceleration );
  s->hip_angle_d[RIGHT] = position;
  s->hip_angled_d[RIGHT] = velocity;

  if ( !lookup_spline3( desired_trajectory, R_KNEE, time,
			current_desireds,
			&position,
			&velocity,
			&acceleration ) )
    {
      printf( "Lookup failure.\n" );
      exit( -1 );
    }
  /* update current desired position */
  set_first_knot_to( current_desireds, R_KNEE, QUINTIC_SPLINE, time,
		     position, velocity, acceleration );
  s->knee_angle_d[RIGHT] = position;
  s->knee_angled_d[RIGHT] = velocity;

  if ( !lookup_spline3( desired_trajectory, R_ANKLE, time,
			current_desireds,
			&position,
			&velocity,
			&acceleration ) )
    {
      printf( "Lookup failure.\n" );
      exit( -1 );
    }
  /* update current desired position */
  set_first_knot_to( current_desireds, R_ANKLE, QUINTIC_SPLINE, time,
		     position, velocity, acceleration );
  s->ankle_angle_d[RIGHT] = position;
  s->ankle_angled_d[RIGHT] = velocity;
}

/*****************************************************************************/

void
run_servos( SIM *s )
{
  s->hip_command[LEFT] = 0;
  s->hip_command[RIGHT] = 0;
  s->knee_command[LEFT] = 0;
  s->knee_command[RIGHT] = 0;
  s->ankle_command[LEFT] = 0;
  s->ankle_command[RIGHT] = 0;

  s->hip_command[LEFT] = 
    - s->k_hip*( s->hip_angle[LEFT] - s->hip_angle_d[LEFT] )
    - s->b_hip*( s->hip_angled[LEFT] - s->hip_angled_d[LEFT] );
  s->hip_command[RIGHT] = 
    - s->k_hip*( s->hip_angle[RIGHT] - s->hip_angle_d[RIGHT] )
    - s->b_hip*( s->hip_angled[RIGHT] - s->hip_angled_d[RIGHT] );
  s->knee_command[LEFT] = 
    - s->k_knee*( s->knee_angle[LEFT] - s->knee_angle_d[LEFT] )
    - s->b_knee*( s->knee_angled[LEFT] - s->knee_angled_d[LEFT] );
  s->knee_command[RIGHT] = 
    - s->k_knee*( s->knee_angle[RIGHT] - s->knee_angle_d[RIGHT] )
    - s->b_knee*( s->knee_angled[RIGHT] - s->knee_angled_d[RIGHT] );
  s->ankle_command[LEFT] = 
    - s->k_ankle*( s->ankle_angle[LEFT] - s->ankle_angle_d[LEFT] )
    - s->b_ankle*( s->ankle_angled[LEFT] - s->ankle_angled_d[LEFT] );
  s->ankle_command[RIGHT] = 
    - s->k_ankle*( s->ankle_angle[RIGHT] - s->ankle_angle_d[RIGHT] )
    - s->b_ankle*( s->ankle_angled[RIGHT] - s->ankle_angled_d[RIGHT] );

  s->hip_torque[LEFT] = s->hip_command[LEFT];
  s->knee_torque[LEFT] = s->knee_command[LEFT];
  s->ankle_torque[LEFT] = s->ankle_command[LEFT];
  s->hip_torque[RIGHT] = s->hip_command[RIGHT]; 
  s->knee_torque[RIGHT] = s->knee_command[RIGHT];
  s->ankle_torque[RIGHT] = s->ankle_command[RIGHT];

  /*
  s->hip_torque[LEFT] = s->hip_command[LEFT]
    + s->rand_scale*((2.0*rand())/RAND_MAX - 1.0);
  s->knee_torque[LEFT] = s->knee_command[LEFT]
    + s->rand_scale*((2.0*rand())/RAND_MAX - 1.0);
  s->hip_torque[RIGHT] = s->hip_command[RIGHT] 
    + s->rand_scale*((2.0*rand())/RAND_MAX - 1.0);
  s->knee_torque[RIGHT] = s->knee_command[RIGHT]
    + s->rand_scale*((2.0*rand())/RAND_MAX - 1.0);
  */
}

/*****************************************************************************/

void
reset_desireds( SIM *s )
{
  /* update current DESIRED positions */
  set_first_knot_to( current_desireds, L_HIP, QUINTIC_SPLINE, s->time,
		     s->hip_angle_d[LEFT], s->hip_angled_d[LEFT], 0.0 );
  set_first_knot_to( current_desireds, L_KNEE, QUINTIC_SPLINE, s->time,
		     s->knee_angle_d[LEFT], s->knee_angled_d[LEFT], 0.0 );
  set_first_knot_to( current_desireds, L_ANKLE, QUINTIC_SPLINE, s->time,
		     s->ankle_angle_d[LEFT], s->ankle_angled_d[LEFT], 0.0 );
  set_first_knot_to( current_desireds, R_HIP, QUINTIC_SPLINE, s->time,
		     s->hip_angle_d[RIGHT], s->hip_angled_d[RIGHT], 0.0 );
  set_first_knot_to( current_desireds, R_KNEE, QUINTIC_SPLINE, s->time,
		     s->knee_angle_d[RIGHT], s->knee_angled_d[RIGHT], 0.0 );
  set_first_knot_to( current_desireds, R_ANKLE, QUINTIC_SPLINE, s->time,
		     s->ankle_angle_d[RIGHT], s->ankle_angled_d[RIGHT], 0.0 );
}

/*****************************************************************************/
/* THIS SEEMS BOGUS. HOW ABOUT MATCHING CURRENT VELOCITY? */
void
zero_velocity_desireds( SIM *s, int side )
{
  /* When the leg has an impact, adjust DESIRED positions to reflect that. */
  if ( side == LEFT )
    {
      set_first_knot_to( current_desireds, L_HIP, QUINTIC_SPLINE, s->time,
			 s->hip_angle_d[LEFT], 0.0, 0.0 );
      set_first_knot_to( current_desireds, L_KNEE, QUINTIC_SPLINE, s->time,
			 s->knee_angle_d[LEFT], 0.0, 0.0 );
    }
  if ( side == RIGHT )
    {
      set_first_knot_to( current_desireds, R_HIP, QUINTIC_SPLINE, s->time,
			 s->hip_angle_d[RIGHT], 0.0, 0.0 );
      set_first_knot_to( current_desireds, R_KNEE, QUINTIC_SPLINE, s->time,
			 s->knee_angle_d[RIGHT], 0.0, 0.0 );
    }
}

/*****************************************************************************/

void setup_swing_knots( SIM *s, int swing_hip, int swing_knee )
{
}

/*****************************************************************************/

void setup_stance_knots( SIM *s, int stance_hip, int stance_knee )
{
}

/*****************************************************************************/

void
run_state_machine( SIM *s )
{

  /* Debounce state transitions */
  if ( s->time - s->state_start_time < 0.001 )
    return;
  s->state_elapsed_time = s->time - s->state_start_time;

  /* Do actions and transitions */
  switch( s->controller_state )
    {
    default:
      break;
      fprintf( stderr, "Unknown state %d in run_state_machine.\n",
	       s->controller_state );
      exit( -1 );
    }
}

/*****************************************************************************/

controller( SIM *s )
{
  int i;

  /* Kinematics already computed by SDFAST */
  /* Forces already computed by SDFAST */

  run_state_machine( s );

  /* Keep track of foot location */
  l_foot_x = s->foot[LEFT][XX];
  r_foot_x = s->foot[RIGHT][XX];

  switch( s->controller_state )
    {
    default:
      break;
      fprintf( stderr, "Unknown state %d in controller.\n",
	       s->controller_state );
      exit( -1 );
    }
  run_trajectory( s, (float) (s->time) );

  s->last_time = s->time;

  run_servos( s );

  update_score( s );
}

/*****************************************************************************/
