#define DEBUG

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <assert.h>
#include <ctime>
#include <iostream>
#include <iomanip>
#include <fstream>

#include "DP.h"
#include "fileIO.h"
#include "util.h"

using namespace std;
using namespace util;

#define INFINITE_COST 1e20
#define OUT_OF_BOUNDS -1000

const int MAX_DIMS_EVER = 5;
const int MAX_ACTION_DIMS_EVER = 3;

//macros
#define INDICES_ITERATE(dim,type) for( indices[dim] = 0; indices[dim] < RES[dimIndex(dim,type)]; (indices[dim])++ )


SweepStats::SweepStats() {
	dV2 = 0.0;
	nowGood = nowBad = polChanges = 0;
}

SweepStats SweepStats::operator+(const SweepStats &other) const {
	SweepStats ss;
	ss.dV2 = dV2 + other.dV2;
	ss.nowGood = nowGood + other.nowGood;
	ss.nowBad = nowBad + other.nowBad;
	ss.polChanges = polChanges + other.polChanges;
	return ( *this);
}

SweepStats SweepStats::operator+=(const SweepStats &other) {
	dV2 += other.dV2;
	nowGood += other.nowGood;
	nowBad += other.nowBad;
	polChanges += other.polChanges;
	return( *this );
}

void SweepStats::print() const {
	printf("Changes = %d    Now Good = %d     Now Bad = %d    dV2 = %g\n",polChanges, nowGood, nowBad, dV2);
}

void SweepStats::printV() const {
	printf("Now Good = %d     Now Bad = %d    dV2 = %g\n", nowGood, nowBad, dV2);
}


bool VERBOSE = false;

void randActUniform(const double * const state, int type, double * const action, const DP *dp) {
	for(int i = 0; i < dp->N_ACTION_DIMS[type]; i++) {
			action[i] =  ((double) rand()/ (double) RAND_MAX)*(dp->MAX_ACTION[dimIndex2(i,type,dp)] - dp->MIN_ACTION[dimIndex2(i,type,dp)]) + dp->MIN_ACTION[dimIndex2(i,type,dp)];
	}
}

inline bool IGRfalse(const double *state, int type, const DP *dp) { return false; }

//debugging - locate a specific grid point and treat specially
double specialState[2] = {0.01, 0.2};
int specialStateInd;

void DP::readConfigFile(const char *name) {
	bool VERBOSE_CONFIG = false;
    ifstream inFile;
    
    inFile.open(name);
    if (!inFile) {
        cout << "Unable to open file "<<name <<endl;
        exit(1); // terminate with error
    }
	string s;

	bool N_DIMS_specified = false;
	bool N_ACTION_DIMS_specified = false;

	while(inFile >> s) {
		//cout <<"    reading: "<< s <<endl;
		if(s == "N_DIMS") {
			int dim;
			N_DIMS = new int[N_DYN_TYPES];
			MAX_DIMS = 0;
			for(int i = 0; i < N_DYN_TYPES; i++) {
				if(!(inFile >> dim)) {
					printf("Not enough dims.\n");
					printf("Each type requires a dim.\n");
					printf("There are %d types and only %d dims specified.\n",N_DYN_TYPES,i);
					exit(-1);
				}
				N_DIMS[i] = dim;
				if(N_DIMS[i] > MAX_DIMS)	{
					MAX_DIMS = N_DIMS[i];
					if(MAX_DIMS > MAX_DIMS_EVER) {
						printf("MAX_DIMS_EVER too low (%d)\n",MAX_DIMS_EVER);
						printf("Should be at least %d\n",MAX_DIMS);
						exit(-1);
					}
				}
				if(VERBOSE_CONFIG)	printf("N_DIMS[%d] = %d\n",i,dim);
			}
			N_DIMS_specified = true;
		}
		else if(s == "N_ACTION_DIMS") {
			int dim;
			N_ACTION_DIMS = new int[N_DYN_TYPES];
			MAX_ACTION_DIMS = 0;
			for(int i = 0; i < N_DYN_TYPES; i++) {
				if(!(inFile >> dim)) {
					printf("Not enough action dims.\n");
					printf("Each type requires an action dim.\n");
					printf("There are %d types and only %d action dims specified.\n",N_DYN_TYPES,i);
					exit(-1);
				}
				N_ACTION_DIMS[i] = dim;
				if(N_ACTION_DIMS[i] > MAX_ACTION_DIMS) {
					MAX_ACTION_DIMS = N_ACTION_DIMS[i];
					if(MAX_ACTION_DIMS > MAX_ACTION_DIMS_EVER) {
						printf("MAX_ACTION_DIMS_EVER too low (%d)\n",MAX_ACTION_DIMS_EVER);
						printf("Should be at least %d\n",MAX_ACTION_DIMS);
						exit(-1);
					}
				}
				if(VERBOSE_CONFIG)	printf("N_ACTION_DIMS[%d] = %d\n",i,dim);
			}
			N_ACTION_DIMS_specified = true;
		}
		else if(s == "RES") {
			if(!N_DIMS_specified) {
				printf("N_DIMS must be specified before RES\n");
			}
			int res;
			RES = new int[N_DYN_TYPES*MAX_DIMS];
			for(int i = 0; i < N_DYN_TYPES; i++) {
				for(int j = 0; j < N_DIMS[i]; j++) {
					if(!(inFile >> res)) {
						printf("Not enough resolutions specified.\n");
						printf("Each type and dim requires a resolution.\n");
						printf("There are %d types.  The dims of the types are: ",N_DYN_TYPES);
						for(int k = 0; k < N_DYN_TYPES; k++) printf("%d ",N_DIMS[k]);
						printf("\nWe ran out of specified resolutions on type %d and dim %d\n",i,j);
						exit(-1);
					}
					RES[dimIndex(j,i)] = res;
					if(VERBOSE_CONFIG)	printf("RES[%d] = %d\n",dimIndex(j,i),res);
				}
			}
		}
		else if(s == "MIN_STATE") {
			if(!N_DIMS_specified) {
				printf("N_DIMS must be specified before MIN_STATE\n");
			}
			double min;
			MIN_STATE = new double[N_DYN_TYPES*MAX_DIMS];
			for(int i = 0; i < N_DYN_TYPES; i++) {
				for(int j = 0; j < N_DIMS[i]; j++) {
					if(!(inFile >> min)) {
						printf("Not enough min states specified.\n");
						printf("Each type and dim requires a min state.\n");
						printf("There are %d types.  The dims of the types are: ",N_DYN_TYPES);
						for(int k = 0; k < N_DYN_TYPES; k++) printf("%d ",N_DIMS[k]);
						printf("\nWe ran out of specified min states on type %d and dim %d\n",i,j);
						exit(-1);
					}
					MIN_STATE[dimIndex(j,i)] = min;
					if(VERBOSE_CONFIG)	printf("MIN_STATE[%d] = %f\n",dimIndex(j,i),min);
				}
			}
		}
		else if(s == "MAX_STATE") {
			if(!N_DIMS_specified) {
				printf("N_DIMS must be specified before MAX_STATE\n");
			}
			double max;
			MAX_STATE = new double[N_DYN_TYPES*MAX_DIMS];
			for(int i = 0; i < N_DYN_TYPES; i++) {
				for(int j = 0; j < N_DIMS[i]; j++) {
					if(!(inFile >> max)) {
						printf("Not enough max states specified.\n");
						printf("Each type and dim requires a max state.\n");
						printf("There are %d types.  The dims of the types are: ",N_DYN_TYPES);
						for(int k = 0; k < N_DYN_TYPES; k++) printf("%d ",N_DIMS[k]);
						printf("\nWe ran out of specified max states on type %d and dim %d\n",i,j);
						exit(-1);
					}
					MAX_STATE[dimIndex(j,i)] = max;
					if(VERBOSE_CONFIG)	printf("MAX_STATE[%d] = %f\n",dimIndex(j,i),max);
				}
			}
		}
		else if(s == "MIN_ACTION") {
			if(!N_DIMS_specified) {
				printf("N_ACTION_DIMS must be specified before MIN_ACTION\n");
			}
			double min;
			MIN_ACTION = new double[N_DYN_TYPES*MAX_ACTION_DIMS];
			for(int i = 0; i < N_DYN_TYPES; i++) {
				for(int j = 0; j < N_ACTION_DIMS[i]; j++) {
					if(!(inFile >> min)) {
						printf("Not enough min actions specified.\n");
						printf("Each type and dim requires a min action.\n");
						printf("There are %d types.  The action dims of the types are: ",N_DYN_TYPES);
						for(int k = 0; k < N_DYN_TYPES; k++) printf("%d ",N_ACTION_DIMS[k]);
						printf("\nWe ran out of specified min actions on type %d and action dim %d\n",i,j);
						exit(-1);
					}
					MIN_ACTION[dimIndex(j,i)] = min;
					hasMA = true;
					if(VERBOSE_CONFIG)	printf("MIN_ACTION[%d] = %f\n",dimIndex(j,i),min);
				}
			}
		}
		else if(s == "MAX_ACTION") {
			if(!N_ACTION_DIMS_specified) {
				printf("N_ACTION_DIMS must be specified before MAX_ACTION\n");
			}
			double max;
			MAX_ACTION = new double[N_DYN_TYPES*MAX_ACTION_DIMS];
			for(int i = 0; i < N_DYN_TYPES; i++) {
				for(int j = 0; j < N_ACTION_DIMS[i]; j++) {
					if(!(inFile >> max)) {
						printf("Not enough max actions specified.\n");
						printf("Each type and action dim requires a max action.\n");
						printf("There are %d types.  The action dims of the types are: ",N_DYN_TYPES);
						for(int k = 0; k < N_DYN_TYPES; k++) printf("%d ",N_ACTION_DIMS[k]);
						printf("\nWe ran out of specified max actions on type %d and action dim %d\n",i,j);
						exit(-1);
					}
					MAX_ACTION[dimIndex(j,i)] = max;
					hasMA = true;
					if(VERBOSE_CONFIG)	printf("MAX_ACTION[%d] = %f\n",dimIndex(j,i),max);
				}
			}
		}
		else if(s == "N_DYN_TYPES") {
			if(N_DIMS_specified) {
				printf("Must specify N_DYN_TYPES before N_DIMS.\n");
				exit(-1);
			}
			if(N_ACTION_DIMS_specified) {
				printf("Must specify N_DYN_TYPES before N_ACTION_DIMS.\n");
				exit(-1);
			}
			int num;
			if(!(inFile >> num)) {
				printf("Could not load the number of types.\n");
				exit(-1);
			}
			N_DYN_TYPES = num;
			if(VERBOSE_CONFIG)	printf("N_DYN_TYPES = %d\n",N_DYN_TYPES);
		}
		else if(s == "N_AUX") {
			int num;
			if(!(inFile >> num)) {
				printf("Could not load the number of aux.\n");
				exit(-1);
			}
			N_AUX = num;
			if(VERBOSE_CONFIG)	printf("N_AUX = %d\n",N_AUX);
		}
		else if(s == "DISCOUNT") {
			double num;
			if(!(inFile >> num)) {
				printf("Could not load the discount.\n");
				exit(-1);
			}
			DISCOUNT = num;
			if(VERBOSE_CONFIG)	printf("DISCOUNT = %d\n",DISCOUNT);
		}
		else {
			cout <<"Unknown command <" << s << "> in config file " << name << endl;
			exit(-1);
		}
	}

	inFile.close();

}

//constructor, does nothing
DP::DP() {};

void noMap(const double *in, double *out, int type) {
	fprintf(stderr,"No Map Specified\n");
	exit(-1);
}

//constructor - initializes stuff
#ifdef MULTI_MODEL
DP::DP(Dynamics *dyn, const char *configName) {
	dynList = new Dynamics *[1];
	*dynList = dyn;
	N_MODEL = 1;
	activeModel = 0;
	construct(dyn,configName);
}

DP::DP(Dynamics **dyn, int N_MOD, const char *configName) {
	N_MODEL = N_MOD;
	dynList = dyn;
	activeModel = 0;
	construct(dyn[0], configName);
}
#else

DP::DP(Dynamics *dyn, const char *configName) {
	construct(dyn,configName);
}
#endif


void DP::construct(Dynamics *dyn, const char *configName) {
	if(dyn) {	
#ifdef MULTI_MODEL
	for(int i = 0; i < N_MODEL; i++)		dynList[i]->initDynamics();
		dynamics = dynList[0];
#else
		dynamics = dyn;
		dyn->initDynamics();
#endif
	}
	
	//defaults
	hasMA = false;
	randActPtr = &randActUniform;
	N_DYN_TYPES = 1;
	DISCOUNT = 0.999;
	N_AUX = 0;

	readConfigFile(configName);

	//handle counting
	int nTotStates = 0;
	N_TYPE_STATES = new int[N_DYN_TYPES];
	statesBeforeType = new int[N_DYN_TYPES];
	dx = new double[N_DYN_TYPES*MAX_DIMS];
	for(int i = 0; i < N_DYN_TYPES; i++) {
		int nStates = 1;
		statesBeforeType[i] = nTotStates;
		for(int j = 0; j < N_DIMS[i]; j++) {
			dx[dimIndex(j,i)] = (MAX_STATE[dimIndex(j,i)]-MIN_STATE[dimIndex(j,i)])/(RES[dimIndex(j,i)]-1);
			nStates *= RES[dimIndex(j,i)];
		}
		N_TYPE_STATES[i] = nStates;
		nTotStates += N_TYPE_STATES[i];
	}
	N_TOTAL_STATES = nTotStates;

	//use to speed up getI1d()
	FREQ1D = new int[N_DYN_TYPES*MAX_DIMS];
	for(int i = 0; i < N_DYN_TYPES; i++) {
		int prev = 1;
		FREQ1D[dimIndex(N_DIMS[i]-1,i)] = 1;
		for(int j = N_DIMS[i]-2; j >=0; j--) {
			FREQ1D[dimIndex(j,i)] = prev*RES[dimIndex(j+1,i)];
			prev = FREQ1D[dimIndex(j,i)];
		}
	}

	//initialize big arrays
#ifdef MULTI_MODEL
	Vlist = new double *[N_MODEL];
	for(int i = 0; i < N_MODEL; i++)	Vlist[i] = new double[N_TOTAL_STATES]();
	V = Vlist[0];
	VtmpList = new double *[N_MODEL];
	for(int i = 0; i < N_MODEL; i++)	VtmpList[i] = NULL;
#else
	V = new double[N_TOTAL_STATES]();
#endif

	Vtmp = NULL;
	U = new double *[N_TOTAL_STATES];

	assert(U);

	for(int i = 0; i < N_DYN_TYPES; i++) {
		for(int j = 0; j < N_TYPE_STATES[i]; j++) {
			U[statesBeforeType[i]+j] = new double[N_ACTION_DIMS[i]]();
		}
	}

	V = new double[N_TOTAL_STATES]();
	assert(V);

	if(N_AUX > 0) {
		aux = new double *[N_TOTAL_STATES];
		for(int i = 0; i < N_DYN_TYPES; i++) {
			for(int j = 0; j < N_TYPE_STATES[i]; j++) {
				aux[statesBeforeType[i]+j] = new double[N_AUX]();
				//if(aux[statesBeforeType[i]+j] == NULL)	printf("bad allocation\n");
			}
		}
	}


	//figure out the corners for cell interpolation
	cornerOffsets = new int *[N_DYN_TYPES];
	corners = new int[N_DYN_TYPES];
	for(int i = 0; i < N_DYN_TYPES; i++) {
		corners[i] = intPow(2,N_DIMS[i]);
		cornerOffsets[i] = new int[corners[i]];	
		calcCornerOffsets(cornerOffsets[i],i);
	}


	//for debugging
	sweep = 0;
	specialStateInd = nearestInds(specialState);

	//no goal region
	inGoalRegion = &IGRfalse;

	//no restart
	useRestart = false;

	//use actionValueML() as opposed to actionValue()
	avML = true;

#ifdef ENABLE_MAPPING
	//make sure maps get set
	grid2natFunk = noMap;
	nat2gridFunk = noMap;
#endif
}

inline bool DP::sameTile(const double * const s1, const double * const s2, int type1, int type2) const {
	if(type1 != type2) return false;

	int dim = dimIndex(0,type1);
	for(int i = 0; i < N_DIMS[type1]; i++) {
#ifdef DEBUG
		if(VERBOSE)	printf("ind %d:  dist = %g;     dx = %g;    diff = %g\n",i,abs(s1[i]-s2[i]),dx[dim],abs(s1[i]-s2[i])-dx[dim]);
#endif
		if(abs(s1[i]-s2[i]) > dx[dim++])	return false;
	}
	return true;
}


void DP::setAVML(bool mode) {
	avML = mode;
}

void DP::calcCornerOffsets(int *offsets, int type) {
	int dims = N_DIMS[type];
	switch(dims) {
	case 1:
		offsets[0] = 0;
		offsets[1] = 1;
		break;
	case 2:
		offsets[0] = 0;
		offsets[1] = 1;
		offsets[2] = FREQ1D[dimIndex(0,type)];
		offsets[3] = FREQ1D[dimIndex(0,type)]+1;
		break;
	case 3:
		offsets[0] = 0;
		offsets[1] = 1;
		offsets[2] = FREQ1D[dimIndex(1,type)];
		offsets[3] = FREQ1D[dimIndex(1,type)]+1;
		offsets[4] = FREQ1D[dimIndex(0,type)];
		offsets[5] = FREQ1D[dimIndex(0,type)]+1;
		offsets[6] = FREQ1D[dimIndex(0,type)]+FREQ1D[dimIndex(1,type)];
		offsets[7] = FREQ1D[dimIndex(0,type)]+FREQ1D[dimIndex(1,type)]+1;
		break;
	case 4:
		offsets[0] = 0;
		offsets[1] = 1;
		offsets[2] = FREQ1D[dimIndex(2,type)];
		offsets[3] = FREQ1D[dimIndex(2,type)] + 1;
		offsets[4] = FREQ1D[dimIndex(1,type)];
		offsets[5] = FREQ1D[dimIndex(1,type)] + 1;
		offsets[6] = FREQ1D[dimIndex(1,type)] + FREQ1D[dimIndex(2,type)];
		offsets[7] = FREQ1D[dimIndex(1,type)] + FREQ1D[dimIndex(2,type)] + 1;
		offsets[8] = FREQ1D[dimIndex(0,type)];
		offsets[9] = FREQ1D[dimIndex(0,type)] + 1;
		offsets[10] = FREQ1D[dimIndex(0,type)] + FREQ1D[dimIndex(2,type)];
		offsets[11] = FREQ1D[dimIndex(0,type)] + FREQ1D[dimIndex(2,type)] + 1;
		offsets[12] = FREQ1D[dimIndex(0,type)] + FREQ1D[dimIndex(1,type)];
		offsets[13] = FREQ1D[dimIndex(0,type)] + FREQ1D[dimIndex(1,type)] + 1;
		offsets[14] = FREQ1D[dimIndex(0,type)] + FREQ1D[dimIndex(1,type)] + FREQ1D[dimIndex(2,type)];
		offsets[15] = FREQ1D[dimIndex(0,type)] + FREQ1D[dimIndex(1,type)] + FREQ1D[dimIndex(2,type)] + 1;
		break;
	case 5:
		offsets[0] = 0;
		offsets[1] = 1;
		offsets[2] = FREQ1D[dimIndex(3,type)] + 0;
		offsets[3] = FREQ1D[dimIndex(3,type)] + 1;
		offsets[4] = FREQ1D[dimIndex(2,type)] + 0;
		offsets[5] = FREQ1D[dimIndex(2,type)] + 1;
		offsets[6] = FREQ1D[dimIndex(2,type)] + FREQ1D[dimIndex(3,type)] + 0;
		offsets[7] = FREQ1D[dimIndex(2,type)] + FREQ1D[dimIndex(3,type)] + 1;
		offsets[8] = FREQ1D[dimIndex(1,type)] + 0;
		offsets[9] = FREQ1D[dimIndex(1,type)] + 1;
		offsets[10] = FREQ1D[dimIndex(1,type)] + FREQ1D[dimIndex(3,type)] + 0;
		offsets[11] = FREQ1D[dimIndex(1,type)] + FREQ1D[dimIndex(3,type)] + 1;
		offsets[12] = FREQ1D[dimIndex(1,type)] + FREQ1D[dimIndex(2,type)] + 0;
		offsets[13] = FREQ1D[dimIndex(1,type)] + FREQ1D[dimIndex(2,type)] + 1;
		offsets[14] = FREQ1D[dimIndex(1,type)] + FREQ1D[dimIndex(2,type)] + FREQ1D[dimIndex(3,type)] + 0;
		offsets[15] = FREQ1D[dimIndex(1,type)] + FREQ1D[dimIndex(2,type)] + FREQ1D[dimIndex(3,type)] + 1;
		offsets[16] = FREQ1D[dimIndex(0,type)] + 0;
		offsets[17] = FREQ1D[dimIndex(0,type)] + 1;
		offsets[18] = FREQ1D[dimIndex(0,type)] + FREQ1D[dimIndex(3,type)] + 0;
		offsets[19] = FREQ1D[dimIndex(0,type)] + FREQ1D[dimIndex(3,type)] + 1;
		offsets[20] = FREQ1D[dimIndex(0,type)] + FREQ1D[dimIndex(2,type)] + 0;
		offsets[21] = FREQ1D[dimIndex(0,type)] + FREQ1D[dimIndex(2,type)] + 1;
		offsets[22] = FREQ1D[dimIndex(0,type)] + FREQ1D[dimIndex(2,type)] + FREQ1D[dimIndex(3,type)] + 0;
		offsets[23] = FREQ1D[dimIndex(0,type)] + FREQ1D[dimIndex(2,type)] + FREQ1D[dimIndex(3,type)] + 1;
		offsets[24] = FREQ1D[dimIndex(0,type)] + FREQ1D[dimIndex(1,type)] + 0;
		offsets[25] = FREQ1D[dimIndex(0,type)] + FREQ1D[dimIndex(1,type)] + 1;
		offsets[26] = FREQ1D[dimIndex(0,type)] + FREQ1D[dimIndex(1,type)] + FREQ1D[dimIndex(3,type)] + 0;
		offsets[27] = FREQ1D[dimIndex(0,type)] + FREQ1D[dimIndex(1,type)] + FREQ1D[dimIndex(3,type)] + 1;
		offsets[28] = FREQ1D[dimIndex(0,type)] + FREQ1D[dimIndex(1,type)] + FREQ1D[dimIndex(2,type)] + 0;
		offsets[29] = FREQ1D[dimIndex(0,type)] + FREQ1D[dimIndex(1,type)] + FREQ1D[dimIndex(2,type)] + 1;
		offsets[30] = FREQ1D[dimIndex(0,type)] + FREQ1D[dimIndex(1,type)] + FREQ1D[dimIndex(2,type)] + FREQ1D[dimIndex(3,type)] + 0;
		offsets[31] = FREQ1D[dimIndex(0,type)] + FREQ1D[dimIndex(1,type)] + FREQ1D[dimIndex(2,type)] + FREQ1D[dimIndex(3,type)] + 1;
		break;
	default:
		printf("Unsupported dim %d in calcCornerOffsets()\n",dims);
		exit(-1);
	}
}

void DP::printValueSliceGrid(double *point, int dim1, int dim2, int type, const char *name) const {
	char tmpString[100];
	sprintf_s(tmpString,"%s.slcv",name);

	int inds[MAX_DIMS_EVER];
	nearestInds(point, type, inds);
	//util::printArray(inds,4);
	getState(inds,type,point);
	printf("Nearest Grid Point: ");printArray(point,N_DIMS[type]);

	int dim1I = dimIndex(dim1,type), dim2I = dimIndex(dim2,type);
	double *vals = new double[RES[dim1I]*RES[dim2I]];
	for( int ind1 = 0; ind1 < RES[dim1I]; ind1++ ) {
		inds[dim1] = ind1;
		for( int ind2 = 0; ind2 < RES[dim2I]; ind2++ ) {
			inds[dim2] = ind2;
			int i1d = getI1d(inds,type);
			vals[ind2*RES[dim1I]+ind1] = V[i1d];
		}
	}

	writeSSV(tmpString,vals,RES[dim1I],RES[dim2I]);

	delete[] vals;


	sprintf_s(tmpString,"%s.datv",name);
	ofstream metaData(tmpString);

	//plot of what
	sprintf_s(tmpString, "%d ", type);
	metaData << tmpString;
	//data about 1st axis
	sprintf_s(tmpString, "%d %f %f ", dim1, MIN_STATE[dim1I], MAX_STATE[dim1I]);
	metaData << tmpString;
	//data about 2nd axis
	sprintf_s(tmpString, "%d %f %f ", dim2, MIN_STATE[dim2I], MAX_STATE[dim2I]);
	metaData << tmpString;
	//data about stationary point
	for(int i = 0; i < N_DIMS[type]; i++) metaData << point[i] << " ";
	metaData << endl;


	metaData.close();
}

void DP::printValueSlice(double *pt, int dim1, int dim2, int type, const char *name) const {
	char tmpString[100];
	sprintf_s(tmpString,"%s.slcv",name);
	
	double *point = new double[N_DIMS[type]];
	for(int i = 0; i < N_DIMS[type]; i++)	point[i] = pt[i];

	int dim1I = dimIndex(dim1,type), dim2I = dimIndex(dim2,type);
	double *vals = new double[RES[dim1I]*RES[dim2I]];
	for( int ind1 = 0; ind1 < RES[dim1I]; ind1++ ) {
		point[dim1] = MIN_STATE[dim1I] + dx[dim1I]*ind1;
		for( int ind2 = 0; ind2 < RES[dim2I]; ind2++ ) {
			point[dim2] = MIN_STATE[dim2I] + dx[dim2I]*ind2;
#ifdef MULTI_MODEL
			vals[ind2*RES[dim1I]+ind1] = lookupValue(point, activeModel, type);
#else
			vals[ind2*RES[dim1I]+ind1] = lookupValue(point,type);
#endif
		}
	}
	writeSSV(tmpString,vals,RES[dim1I],RES[dim2I]);
	delete[] vals;
	

	sprintf_s(tmpString,"%s.datv",name);
	ofstream metaData(tmpString);

	//plot of what
	sprintf_s(tmpString, "%d ", type);
	metaData << tmpString;
	//data about 1st axis
	sprintf_s(tmpString, "%d %f %f ", dim1, MIN_STATE[dim1I], MAX_STATE[dim1I]);
	metaData << tmpString;
	//data about 2nd axis
	sprintf_s(tmpString, "%d %f %f ", dim2, MIN_STATE[dim2I], MAX_STATE[dim2I]);
	metaData << tmpString;
	//data about stationary point
	for(int i = 0; i < N_DIMS[type]; i++) metaData << point[i] << " ";
	metaData << endl;
	delete[] point;

	metaData.close();
}

void DP::printPolSliceGrid(double *point, int dim1, int dim2, int type, const char *name, int polIndex) const {
	char tmpString[100];
	sprintf_s(tmpString,"%s.slcp",name);


	int *inds = new int[N_DIMS[type]];
	nearestInds(point, type, inds);
	getState(inds,type,point);
	printf("Nearest Grid Point: ");printArray(point,N_DIMS[type]);

	int dim1I = dimIndex(dim1,type), dim2I = dimIndex(dim2,type);
	double *vals = new double[RES[dim1I]*RES[dim2I]];
	for( int ind1 = 0; ind1 < RES[dim1I]; ind1++ ) {
		inds[dim1] = ind1;
		for( int ind2 = 0; ind2 < RES[dim2I]; ind2++ ) {
			inds[dim2] = ind2;
			int i1d = getI1d(inds,type);
			vals[ind2*RES[dim1I]+ind1] = U[i1d][polIndex];
		}
	}
	writeSSV(tmpString,vals,RES[dim1I],RES[dim2I]);
	delete[] vals;
	delete[] inds;

	sprintf_s(tmpString,"%s.datp",name);
	ofstream metaData(tmpString);

	//plot of what
	sprintf_s(tmpString, "%d %d ", type, polIndex);
	metaData << tmpString;
	//data about 1st axis
	sprintf_s(tmpString, "%d %f %f ", dim1, MIN_STATE[dim1I], MAX_STATE[dim1I]);
	metaData << tmpString;
	//data about 2nd axis
	sprintf_s(tmpString, "%d %f %f ", dim2, MIN_STATE[dim2I], MAX_STATE[dim2I]);
	metaData << tmpString;
	//data about stationary point
	for(int i = 0; i < N_DIMS[type]; i++) metaData << point[i] << " ";
	metaData << endl;


	metaData.close();
}

void DP::printPolSlice(double *pt, int dim1, int dim2, int type, const char *name, int polIndex) const {
	char tmpString[100];
	sprintf_s(tmpString,"%s.slcp",name);

	double *point = new double[N_DIMS[type]];
	for(int i = 0; i < N_DIMS[type]; i++)	point[i] = pt[i];
	
	const int dim1I = dimIndex(dim1,type), dim2I = dimIndex(dim2,type);
	double *polTmp = new double[MAX_ACTION_DIMS];
	double *vals = new double[RES[dim1I]*RES[dim2I]];
	for( int ind1 = 0; ind1 < RES[dim1I]; ind1++ ) {
		point[dim1] = MIN_STATE[dim1I] + dx[dim1I]*ind1;
		for( int ind2 = 0; ind2 < RES[dim2I]; ind2++ ) {
			point[dim2] = MIN_STATE[dim2I] + dx[dim2I]*ind2;
			lookupPolicy(polTmp,point,type);
			vals[ind2*RES[dim1I]+ind1] = polTmp[polIndex];
		}
	}
	writeSSV(tmpString,vals,RES[dim1I],RES[dim2I]);
	delete[] vals;
	delete[] polTmp;

	sprintf_s(tmpString,"%s.datp",name);
	ofstream metaData(tmpString);

	//plot of what
	sprintf_s(tmpString, "%d %d ", type, polIndex);
	metaData << tmpString;
	//data about 1st axis
	sprintf_s(tmpString, "%d %f %f ", dim1, MIN_STATE[dim1I], MAX_STATE[dim1I]);
	metaData << tmpString;
	//data about 2nd axis
	sprintf_s(tmpString, "%d %f %f ", dim2, MIN_STATE[dim2I], MAX_STATE[dim2I]);
	metaData << tmpString;
	//data about stationary point
	for(int i = 0; i < N_DIMS[type]; i++) metaData << point[i] << " ";
	metaData << endl;


	metaData.close();
}


void DP::findValue(void (*polFunk)(const double *, int, double * const)) {
	//assign the policy
	setPolicy(polFunk);

	for(int i = 1; i < 1000; i++) {
		printf("sweep %d\n", i);
		doVsweep();
		if(i%100 ==0) {
			writeUV("findLQRval2",i);
		}
	}
	writeUV("findLQRval2");
}

//updates Vtmp for the current policy without changing the policy
#ifdef MULTI_MODEL
void DP::updateV(int ind1d, int mod, SweepStats &ss) {
#else
void DP::updateV(int ind1d, SweepStats &ss) {
#endif

#ifdef DEBUG
	bool verbose = false;
#endif


	double state[MAX_DIMS_EVER];
	int inds[MAX_DIMS_EVER];
	int type;
	
	//lookup state for this grid point
	getState(ind1d,state,type);
	getImd(ind1d,type,inds);

#ifdef DEBUG
	if(verbose)	{
		printf("state = ");
		for(int i = 0; i < N_DIMS[type]; i++)	printf("%g ",state[i]);
		printf("\n");
	}
#endif


	if(inGoalRegion(state,type,this)) {
#ifdef DEBUG
		if(verbose)	printf("IGR\n");
#endif
		Vtmp[ind1d] = goalRegionValue(state,type);
		return;
	}


	double *curAction = U[ind1d];
	if(curAction == NULL)	{
		printf("ind1d = %d of %d\n",ind1d, N_TOTAL_STATES);
	}
#ifdef DEBUG
#ifdef MULTI_MODEL
	double curVal = actionValueML(state, curAction, mod, type, inds, verbose);
#else
	double curVal = actionValueML(state, curAction, type, inds, verbose);
#endif
	if(verbose) {
		printf("action = ");
		for(int i = 0; i < N_ACTION_DIMS[type]; i++)	printf("%g ",curAction[i]);
		printf("\n");
		printf("new Val = %g\n",curVal);
		printf("old Val = %g\n",V[ind1d]);
	}
#else
	double curVal = actionValueML(state, curAction, type, indices, false);
#endif
	

	Vtmp[ind1d] = curVal;

	if(V[ind1d] == INFINITE_COST) {
		if(Vtmp[ind1d] != INFINITE_COST)	ss.nowGood += 1;
	}
	else {
		if(Vtmp[ind1d] == INFINITE_COST)	ss.nowBad += 1;
		else {
			double dV = Vtmp[ind1d]-V[ind1d];
			ss.dV2 += dV*dV;
		}
	}

}

void DP::adjustAll() {
	double state[MAX_DIMS_EVER];
	int inds[MAX_DIMS_EVER];
	int type;
	for(int i = 0; i < N_TOTAL_STATES; i++) {
		getImd(i,type,inds);
		getState(i,state,type);
		/****************************************/
		/*            BEGIN STUFF               */

		if(V[i] >0)	printf("%d %g\n", i, V[i]);
		//if(type == 0 || type ==1) { U[i][0] /= 95*9.81;}

		/*             End  STUFF               */
		/****************************************/
	}
}

void DP::plotQvsU(const double *state, double min, double max, char *fileName, int type, int polInd, const double *defAct, bool verbose) const {
	int inds[MAX_DIMS_EVER];
	int ind = nearestInds(state,type,inds);
	double gridState[MAX_DIMS_EVER];
	int tmpType;
	getState(ind,gridState,tmpType);
	assert(tmpType == type);

	ofstream out(fileName);

	double action[MAX_ACTION_DIMS_EVER];
	if(defAct != NULL) {
		for(int i = 0; i < N_ACTION_DIMS[type]; i++) action[i] = defAct[i];
	}
	double bestVal = 1e20;
	double bestAct;
	for(double act = min; act < max; act += (max-min)/1000) {
		action[polInd] = act;
#ifdef MULTI_MODEL
		double val = actionValue(state, action, activeModel, type, inds);
#else
		double val = actionValue(state, action, type, inds);
#endif
		out << act << " " << val << std::endl;
		if(val < bestVal) {
			bestVal = val;
			bestAct = act;
		}
	}
	if(verbose) printf("Best val is %g at action[%d] = %g\n",bestVal, polInd, bestAct);
	out.close();
}

#ifdef MULTI_MODEL
void DP::setActiveModel(int mod) {
	activeModel = mod;
	V = Vlist[mod];
	Vtmp = VtmpList[mod];
	dynamics = dynList[mod];
}
#endif

double DP::getValueBySim(double *state, int type, int ts) {
	double cost = 0;
	double discount = 1.0;
	double *action = new double[MAX_ACTION_DIMS];
	
	for(int i = 0; i < ts; i++) {
		lookupPolicy(action, state, type);
		cost += discount*dynamics->integrateCost(state,action,state,type,&type);
		discount *= DISCOUNT;
		if(!inBounds(state,type)) {
			delete[] action;
			return INFINITE_COST;
		}
	}

	delete[] action;
	return cost;
}

//update U and Vtmp
void DP::updateState(int ind1d, SweepStats &ss) {
	//printf("US %d\n",ind1d);
#ifdef DEBUG
	allocateVtmp();
#endif
	

//	if(specialStateInd == ind1d) printf("%d\n", ind1d);
	bool verbose = false;

	//if(specialStateInd == ind1d)	verbose = true;


	int type;
	int inds[MAX_DIMS_EVER];
	double state[MAX_DIMS_EVER];

	//lookup state for this grid point
	getImd(ind1d,type,inds);
	getState(inds,type, state);

	if(inGoalRegion(state,type,this)) {
		goalRegAction(state,type,U[ind1d],this);
		Vtmp[ind1d] = goalRegionValue(state,type);
		return;
	}

#ifdef DEBUG
	if(verbose)	{printf("BEGIN UPDATE\nstate:  ");printArray(state,N_DIMS[type]);}
#endif
	//look up current policy
	double *curAction = U[ind1d];
#ifdef DEBUG
	if(verbose)	{printf("current action: ");printArray(curAction,N_ACTION_DIMS[type]);}
#endif
	//check what the value is under the current policy
#ifdef MULTI_MODEL
	double curVal=0;
	int curNumInf = 0;
	double *curValList = new double[N_MODEL];
	for(int i = 0; i < N_MODEL; i++) {
		if(avML)	curValList[i] = actionValueML(state, curAction, i, type, inds, verbose);
		else		curValList[i] = actionValue(  state, curAction, i, type, inds, verbose);
		if(curValList[i] == INFINITE_COST)		curNumInf++;
		else									curVal += curValList[i];
		//if(specialStateInd == ind1d) printf("%g %g\n", curValList[i], curVal);
	}
#else
	double curVal;
	if(avML)	curVal = actionValueML(state, curAction, type, inds, verbose);
	else		curVal = actionValue(  state, curAction, type, inds, verbose);
#endif

	//get a random action
	double guessAction[MAX_ACTION_DIMS_EVER];
	randActPtr(state,type,guessAction,this);

	//evaluate it
#ifdef MULTI_MODEL
	double guessVal=0;
	int guessNumInf = 0;
	double *guessValList = new double[N_MODEL];
	for(int i = 0; i < N_MODEL; i++) {
		if(avML)	guessValList[i] = actionValueML(state, guessAction, i, type, inds, verbose);
		else		guessValList[i] = actionValue(  state, guessAction, i, type, inds, verbose);
		if(guessValList[i] == INFINITE_COST)		guessNumInf++;
		else										guessVal += guessValList[i];
	}
#else
	double guessVal;
	if(avML)	guessVal = actionValueML(state, guessAction, type, inds, verbose);
	else		guessVal = actionValue(  state, guessAction, type, inds, verbose);
#endif


#ifdef MULTI_MODEL
	bool improve = false;
	if(guessNumInf < curNumInf)										improve = true;
	else if(guessNumInf == curNumInf   &&   guessVal < curVal)		improve = true;
	//if no improvement
	if(!improve) {
		for(int i = 0; i < N_MODEL; i++) {
			VtmpList[i][ind1d] = curValList[i];
#ifdef DEBUG
			if(verbose)	printf("Keeping old action: new value is %g for dyn %d\n",VtmpList[i][ind1d],i);
#endif
		}
	}
	else {//if an improvement
		ss.polChanges++;
		copyArray(guessAction, U[ind1d], N_ACTION_DIMS[type]);
		for(int i = 0; i < N_MODEL; i++) {
			VtmpList[i][ind1d] = guessValList[i];
#ifdef DEBUG
			if(verbose)	printf("Updating action: new value is %f\n",Vtmp[ind1d]);
#endif
		}
	}

#ifdef DEBUG
	if(verbose)	printf("END UPDATE\n\n\n");
#endif
	for(int i = 0; i < N_MODEL; i++) {
		if(Vlist[i][ind1d] == INFINITE_COST) {
			if(VtmpList[i][ind1d] != INFINITE_COST)		ss.nowGood += 1;
		}
		else {
			if(VtmpList[i][ind1d] == INFINITE_COST) {
				ss.nowBad += 1;
			}
			else {
				double dV = VtmpList[i][ind1d]-Vlist[i][ind1d];
				ss.dV2 += dV*dV;
			}
		}
	}

	delete[] curValList;
	delete[] guessValList;
#else	//not MULTI_MODEL

	//if no improvement
	if(curVal <= guessVal) {
		Vtmp[ind1d] = curVal;
#ifdef DEBUG
		if(verbose)	printf("Keeping old action: new value is %f\n",Vtmp[ind1d]);
#endif
	}
	else {//if an improvement
		ss.polChanges++;
		Vtmp[ind1d] = guessVal;
		copyArray(guessAction, U[ind1d], N_ACTION_DIMS[type]);
#ifdef DEBUG
		if(verbose)	printf("Updating action: new value is %f\n",Vtmp[ind1d]);
#endif
	}

#ifdef DEBUG
	if(verbose)	printf("END UPDATE\n\n\n");
#endif

	if(V[ind1d] == INFINITE_COST) {
		if(Vtmp[ind1d] != INFINITE_COST)	ss.nowGood += 1;
	}
	else {
		if(Vtmp[ind1d] == INFINITE_COST) {
			ss.nowBad += 1;
		}
		else {
			double dV = Vtmp[ind1d]-V[ind1d];
			ss.dV2 += dV*dV;
		}
	}
#endif //end MULTI_MODEL
	
}



void DP::setRandomActionFunction(void (*funk)(const double * const, int, double * const, const DP *)) {
	randActPtr = funk;
}

void DP::setPolicy(void (*polFunk)(const double *, int, double * const)) {
	double *state = new double[MAX_DIMS];
	double *action = new double[MAX_ACTION_DIMS];
	//printf("%d %d\n",MAX_DIMS, MAX_ACTION_DIMS);
	int type;
	for(int i = 0; i < N_TOTAL_STATES; i++) {
		//printf("AA %d\n",i);
		getState(i,state,type);
		//printf("BB %f %f %d\n",state[0], state[1], type);
		polFunk(state,type,action);
		//printf("CC %f\n",action[0]);
		copyArray(action,U[i],N_ACTION_DIMS[type]);
		//printf("DD %d\n",N_ACTION_DIMS[i]);
	}
	delete[] state;
	delete[] action;
}


int goalType;
double *goalState;
double *goalSize;
double *LQR_K;
double *LQR_Qdiag;



bool IGRcellDist(const double *state, int type, const DP *dp) {

	if(type != goalType)	return false;

	for(int i = 0; i < dp->N_DIMS[type]; i++) {
		if(state[i] < goalState[i]-goalSize[i])	return false;
		if(state[i] > goalState[i]+goalSize[i])	return false;
	}
	return true;
}

//u = -Kx
//K reads like text (LR->TB)
void Kcon(const double *state, int type, double * const action, const DP *dp) {
	int k = 0;
	for(int i = 0; i < dp->N_ACTION_DIMS[type]; i++) {
		action[i] = 0;
		for(int j = 0; j < dp->N_DIMS[type]; j++) {
			action[i] += -(state[j]-goalState[j])*LQR_K[k];
			k++;
		}
	}
}

double QdiagVal(const double *state, int type, const DP *dp) {
	double val = 0;
	for(int i = 0; i < dp->N_DIMS[type]; i++) {
		double delx = state[i]-goalState[i];
		val += delx*delx*LQR_Qdiag[i];
	}
	return val;
}

void DP::setGoalRegion(	bool (*IGRfunk)(const double *, int, const DP *), double (*GRVfunk)(const double *, int, const DP *),	void (*GRAfunk)(const double *, int, double * const, const DP *)) {
	inGoalRegion = IGRfunk;
	goalRegVal = GRVfunk;
	goalRegAction = GRAfunk;
}

void DP::setRestartPoint(double ttRestart, double *point, int type) {
	useRestart = true;
	restartType = type;
	restartPt = point;
	timeToRestart = ttRestart;
}

//wrapper for the goalRegVal function pointer that handles adding restarts if necessary
double DP::goalRegionValue(const double *state, int type) const {
#ifdef MULTI_MODEL
	if(useRestart) {
		printf("goalRegionValue not written for this case\n");
		exit(-1);
	}
#endif
	if(useRestart)	return goalRegVal(state, type, this) + lookupValue(restartPt, restartType)*pow(DISCOUNT, timeToRestart/dynamics->getTimeStep());
	return goalRegVal(state, type, this);
}

void DP::setSimpleLQRgoal( double *state, int type, double *K, double *Qdiag) {
	//record info for checking if in the goal region
	goalType = type;
	goalState = new double[N_DIMS[type]];
	goalSize = new double[N_DIMS[type]];
	for(int i = 0; i < N_DIMS[type]; i++) {
		goalState[i] = state[i];
		goalSize[i] = dx[dimIndex(i,type)];
	}

	//save K - for Kcon
	LQR_K = new double[N_DIMS[type]*N_ACTION_DIMS[type]];
	for(int i = 0; i < N_DIMS[type]*N_ACTION_DIMS[type]; i++) {
		LQR_K[i] = K[i];
	}

	//save Qdiag
	LQR_Qdiag = new double[N_DIMS[type]];
	for(int i = 0; i < N_DIMS[type]; i++) {
		LQR_Qdiag[i] = Qdiag[i];
	}

	setGoalRegion(IGRcellDist,QdiagVal,Kcon);
}


#ifdef MULTI_MODEL
double DP::lookupValue(const double *state, int mod, int type) const {
#else
double DP::lookupValue(const double * const state, int type) const {
#endif

#ifdef DEBUG
	if(VERBOSE) printf("Begin LookupVal\n");
#endif

	if(inGoalRegion(state,type,this)) {
#ifdef DEBUG
		if(VERBOSE) printf("In Goal Region\n");
#endif
		return goalRegionValue(state,type);
	}

	const int dims = N_DIMS[type];

	//figure out table location
	double *fracs = new double[dims];

	int baseInd1d = setupInterp(state, type, dims, fracs);
#ifdef DEBUG
	if(VERBOSE)	printf("Base Index is %d\n",baseInd1d);
#endif
	if(baseInd1d == OUT_OF_BOUNDS)	{
#ifdef DEBUG
		if(VERBOSE)	printf("Is out of bounds - Returning INFINITE_COST\n");
#endif
		delete[] fracs;
		return INFINITE_COST;
	}

	bool hasBadValue = false;
	bool *mask = new bool[corners[type]];
	//lookup corner values
	double *coVal = new double[corners[type]];
	for(int i = 0; i < corners[type]; i++) {
#ifdef MULTI_MODEL
		coVal[i] = Vlist[mod][baseInd1d+cornerOffsets[type][i]];
#else
		coVal[i] = V[baseInd1d+cornerOffsets[type][i]];
#endif
		if(coVal[i] >= INFINITE_COST) {
			hasBadValue = true;
			mask[i] = false;
		}
		else {
			mask[i] = true;
		}
	}

#ifdef DEBUG
	if(VERBOSE) {
		double corState[100];
		int typeTmp;
		printf("Interp Corners:  (state)  val  index\n");
		for(int i = 0; i < corners[type]; i++) {
			getState(baseInd1d+cornerOffsets[type][i],corState,typeTmp);
			printf("( ");
			for(int j = 0; j < dims; j++) printf("%g ", corState[j]);
			printf(")     %g     %d\n",coVal[i],baseInd1d+cornerOffsets[type][i]);
		}
	}
#endif
			

	double value;
	//do interpolation
	if(hasBadValue)		value = doInterpDW(coVal, fracs, dims, mask);
	else				value = doInterpML(coVal, fracs, dims);

	delete[] mask;
	delete[] fracs;
	delete[] coVal;

	return value;
}

int DP::lookupPolicy(double *action, const double * const state, int type, bool verbose) const {

	if(inGoalRegion(state,type,this)) {
#ifdef DEBUG
		if(verbose)		printf("In Goal Region - getting goal region action\n");
#endif
		goalRegAction(state,type,action,this);
		return 1;
	}
	const int dims = N_DIMS[type];
	//figure out table location
	double *fracs = new double[dims];
	int baseInd1d = setupInterp(state, type, dims, fracs);
#ifdef DEBUG
	if(verbose)		printf("baseInd = %d\n",baseInd1d);
#endif

	//if state is off the grid
	if(baseInd1d == OUT_OF_BOUNDS) {
#ifdef DEBUG
		if(verbose)		printf("Out of Bounds - returning 0 action\n");
#endif
		delete[] fracs;
		//set all actions to 0
		for(int j = 0; j < N_ACTION_DIMS[type]; j++) {
			action[j] = 0.0;
		}
		return 0;
	};


	/*
	bool hasBadValue = false;
	bool hasGoodValue = false;
	bool *mask = new bool[corners[type]];
	//lookup corner values
	for(int i = 0; i < corners[type]; i++) {
		if(V[baseInd1d+cornerOffsets[type][i]] >= INFINITE_COST) {
			hasBadValue = true;
			mask[i] = false;
		}
		else {
			hasGoodValue = true;
			mask[i] = true;
		}
	}
	/**/

	/*
	//if this is a totally bad cell
	if(!hasGoodValue) {
#ifdef DEBUG
		if(verbose)		printf("In a bad cell (all bad values) - returning 0 action\n");
#endif
		for(int j = 0; j < N_ACTION_DIMS[type]; j++) {
			action[j] = 0.0;
		}
		delete[] mask;
		delete[] fracs;
		return 0;
	}
	*/

	
	//lookup corner values
	double *coVal = new double[corners[type]];
	//for each dimension in U
	for(int j = 0; j < N_ACTION_DIMS[type]; j++) {

		for(int i = 0; i < corners[type]; i++) {
			coVal[i] = U[baseInd1d+cornerOffsets[type][i]][j];
			//printf("%d   %g\n", cornerOffsets[type][i], coVal[i]);
		}

#ifdef DEBUG
		if(verbose) {
			printf("Covals are:\n");
			for(int i = 0; i < corners[type]; i++)	{
				printf("%d %g   ", baseInd1d+cornerOffsets[type][i], coVal[i]);
				double coState[MAX_DIMS_EVER];
				int tp;
				getState(baseInd1d+cornerOffsets[type][i],coState,tp);
				util::printArray(coState,N_DIMS[tp]);
			}
		}
#endif

		//do interpolation
	//	if(false && hasBadValue)		action[j] = doInterpDW(coVal, fracs, dims, mask);
//		else	
		action[j] = doInterpML(coVal, fracs, dims);
		//printf("found action %d to be %f\n",j,action[j]);

		if(action[j] == INFINITE_COST)	action[j] = 0;
	}

	//keeps things neater
	for(int j = N_ACTION_DIMS[type]; j < MAX_ACTION_DIMS; j++) {
		action[j] = 0;
	}

//	delete[] mask;
	delete[] fracs;
	delete[] coVal;
	return 1;
}



int DP::lookupAux(double *auxArr, const double *state, int type) const {

	const int dims = N_DIMS[type];
	//figure out table location
	double *fracs = new double[dims];
	int baseInd1d = setupInterp(state, type, dims, fracs);

	if(baseInd1d == OUT_OF_BOUNDS) {
		delete[] fracs;
		return 0;
		
		printf("Out of bounds in lookupAux():\n (");
		for(int i = 0; i < dims; i++) {
			printf("%f ",state[i]);
		}
		printf(") type: %d\n",type);
		exit(-1);
		
	};

	//lookup corner values
	double *coVal = new double[corners[type]];
	//for each dimension in AUX
	for(int j = 0; j < N_AUX; j++) {

		for(int i = 0; i < corners[type]; i++) {
			coVal[i] = aux[baseInd1d+cornerOffsets[type][i]][j];
		}

		//do interpolation
		auxArr[j] = doInterpML(coVal, fracs, dims);
		//printf("found action %d to be %f\n",j,action[j]);

	}

	delete[] fracs;
	delete[] coVal;
	return 1;
}


int DP::nearestInds(const double *state, int type, int *inds, bool verbose) const {
	int dims = N_DIMS[type];
	
	bool del = false;
	if(inds == NULL) {
		del = true;
		inds = new int[dims];
	}
#ifdef DEBUG
	if(verbose)	for(int i = 0; i < 15; i++)	printf("%d\n",RES[i]);
#endif
	int dim = type*MAX_DIMS;
#ifdef DEBUG
	if(verbose)	printf("t MD d: %d %d %d\n",type, MAX_DIMS, dim);
#endif
	for(int i = 0; i < dims; i++) {
		inds[i] = (int)floor((state[i]-MIN_STATE[dim])/dx[dim]+.5);
#ifdef DEBUG
		if(verbose)	printf("inds[%d] = %d\n",i,inds[i]);
		if(verbose) printf("s MS dx: %g %g %g\n",state[i], MIN_STATE[dim], dx[dim]);
#endif
		
		//check bounds - bounds are [start,end)
		if(inds[i] < 0 || inds[i] >= RES[dim])	{
			if(del)	delete[] inds;
#ifdef DEBUG
			if(verbose)	printf("dim %d out of bounds: not in [0, %d] - dim = %d\n",i,RES[dim]-1,dim);
#endif
			return OUT_OF_BOUNDS;
		}
		dim++;
	}
	int ans =  getI1d(inds, type);
	if(del)	delete[] inds;
	return ans;
}

void DP::printResolutions() const {
	for(int i = 0; i < N_DYN_TYPES; i++) {
		printf("Type %d:\n",i);
		for(int j =0; j < N_DIMS[i]; j++) {
			printf("%f\n",dx[dimIndex(j,i)]);
		}
	}
}

void DP::explainValue(int ind) const {
	double state[MAX_DIMS_EVER];
	int type;
	getState(ind,state,type);
	explainValue(state,type);
}

void DP::explainValue(double *state, int type) const {
#ifdef DEBUG
	printf("Input State: ");printArray(state,N_DIMS[type]);
	int ind = nearestInds(state,type,NULL,false);
	printf("Single Index = %d\n",ind);
	double action[MAX_ACTION_DIMS_EVER];
	getState(ind,state,type);
	int tmpType;
	int indices[MAX_DIMS_EVER];
	getImd(ind, tmpType, indices);
	assert(tmpType == type);
	printf("Nearest Grid Point: ");printArray(state,N_DIMS[type]);
	printf("Indices are:  ");printArray(indices,N_DIMS[type]);
	for(int i = 0; i < N_ACTION_DIMS[type]; i++)	action[i] = U[ind][i];
	printf("Existing action is:");
	for(int i = 0; i < N_ACTION_DIMS[type]; i++)	printf("  %f", action[i]);
	printf("\n");
	VERBOSE = true;
#ifdef MULTI_MODEL
	printf("Existing Value is %f\n",lookupValue(state, activeModel, type));
	actionValueML(state, action, activeModel, type, indices, true);
#else
	printf("Existing Value is %f\n",lookupValue(state,type));
	actionValueML(state, action, type, indices, true);
#endif
	VERBOSE = false;

#else
	printf("Must have DEBUG turned on for explain value to work\n");
	printf("Defined at the top of DP.cpp\n");
#endif

	
}


//find fracs, check bounds, get base index
int DP::setupInterp(const double *state, int type, int dims, double *fracs) const {
	int baseInd[MAX_DIMS_EVER];
#ifdef ENABLE_MAPPING
	double gridState[MAX_DIMS_EVER];
	nat2gridFunk(state,gridState,type);
	//util::printArray(state,4);
	//util::printArray(gridState,4);
#endif
	//printf("val state: ");printArray(state,2);
	int dim = type*MAX_DIMS;
	for(int i = 0; i < dims; i++) {
#ifdef ENABLE_MAPPING
		fracs[i] = (gridState[i]-MIN_STATE[dim])/dx[dim];
#else
		fracs[i] = (state[i]-MIN_STATE[dim])/dx[dim];
#endif
		baseInd[i] = (int)floor(fracs[i]);

		fracs[i] -= baseInd[i];
		
		//check bounds - bounds are [start,end]
		if(baseInd[i] < 0 || baseInd[i] >= RES[dim]-1)	{
			//makes the upper bound inclusive
			if(baseInd[i] == RES[dim]-1 && fracs[i] < 1e-5) {
				fracs[i] = 1.0;
				baseInd[i] -= 1;
			}
			else {
#ifdef DEBUG
				if(VERBOSE)	printf("ind %d (%g) is out of bounds\n",i,state[i]);
#endif
				return OUT_OF_BOUNDS;
			}
		}
		
#ifdef DEBUG
		assert(fracs[i] >= 0);
		assert(fracs[i] <= 1);
#endif

		dim++;
	}
	
	return getI1d(baseInd, type);
}
//multilinear interpolation
double DP::doInterpML(const double *coVal, double *fracs, int dims) const {
	double val;
	double *aFrac = new double[dims];	//anti-fractions (1-fracs)

#ifdef DEBUG
	if(VERBOSE) printf("Begin Interp ML\n");
#endif
	for(int i = 0; i < dims; i++)	aFrac[i] = 1.0-fracs[i];

	switch(dims) {
	case 1:
		val = coVal[0]*aFrac[0]+coVal[1]*fracs[0];
		break;
	case 2:
		val =  
			aFrac[0]*aFrac[1]*coVal[0]+
			aFrac[0]*fracs[1]*coVal[1]+
			fracs[0]*aFrac[1]*coVal[2]+
			fracs[0]*fracs[1]*coVal[3];	
		break;
	case 3:
		val = 
			aFrac[0]*aFrac[1]*aFrac[2]*coVal[0]+
			aFrac[0]*aFrac[1]*fracs[2]*coVal[1]+
			aFrac[0]*fracs[1]*aFrac[2]*coVal[2]+
			aFrac[0]*fracs[1]*fracs[2]*coVal[3]+
			fracs[0]*aFrac[1]*aFrac[2]*coVal[4]+
			fracs[0]*aFrac[1]*fracs[2]*coVal[5]+
			fracs[0]*fracs[1]*aFrac[2]*coVal[6]+
			fracs[0]*fracs[1]*fracs[2]*coVal[7];
		break;
	case 4:
		val = 
			aFrac[0]*aFrac[1]*aFrac[2]*aFrac[3]*coVal[0]+
			aFrac[0]*aFrac[1]*aFrac[2]*fracs[3]*coVal[1]+
			aFrac[0]*aFrac[1]*fracs[2]*aFrac[3]*coVal[2]+
			aFrac[0]*aFrac[1]*fracs[2]*fracs[3]*coVal[3]+
			aFrac[0]*fracs[1]*aFrac[2]*aFrac[3]*coVal[4]+
			aFrac[0]*fracs[1]*aFrac[2]*fracs[3]*coVal[5]+
			aFrac[0]*fracs[1]*fracs[2]*aFrac[3]*coVal[6]+
			aFrac[0]*fracs[1]*fracs[2]*fracs[3]*coVal[7]+
			fracs[0]*aFrac[1]*aFrac[2]*aFrac[3]*coVal[8]+
			fracs[0]*aFrac[1]*aFrac[2]*fracs[3]*coVal[9]+
			fracs[0]*aFrac[1]*fracs[2]*aFrac[3]*coVal[10]+
			fracs[0]*aFrac[1]*fracs[2]*fracs[3]*coVal[11]+
			fracs[0]*fracs[1]*aFrac[2]*aFrac[3]*coVal[12]+
			fracs[0]*fracs[1]*aFrac[2]*fracs[3]*coVal[13]+
			fracs[0]*fracs[1]*fracs[2]*aFrac[3]*coVal[14]+
			fracs[0]*fracs[1]*fracs[2]*fracs[3]*coVal[15];
		break;
	case 5:
		val = 
			aFrac[0]*aFrac[1]*aFrac[2]*aFrac[3]*aFrac[4]*coVal[0]+
			aFrac[0]*aFrac[1]*aFrac[2]*aFrac[3]*fracs[4]*coVal[1]+
			aFrac[0]*aFrac[1]*aFrac[2]*fracs[3]*aFrac[4]*coVal[2]+
			aFrac[0]*aFrac[1]*aFrac[2]*fracs[3]*fracs[4]*coVal[3]+
			aFrac[0]*aFrac[1]*fracs[2]*aFrac[3]*aFrac[4]*coVal[4]+
			aFrac[0]*aFrac[1]*fracs[2]*aFrac[3]*fracs[4]*coVal[5]+
			aFrac[0]*aFrac[1]*fracs[2]*fracs[3]*aFrac[4]*coVal[6]+
			aFrac[0]*aFrac[1]*fracs[2]*fracs[3]*fracs[4]*coVal[7]+
			aFrac[0]*fracs[1]*aFrac[2]*aFrac[3]*aFrac[4]*coVal[8]+
			aFrac[0]*fracs[1]*aFrac[2]*aFrac[3]*fracs[4]*coVal[9]+
			aFrac[0]*fracs[1]*aFrac[2]*fracs[3]*aFrac[4]*coVal[10]+
			aFrac[0]*fracs[1]*aFrac[2]*fracs[3]*fracs[4]*coVal[11]+
			aFrac[0]*fracs[1]*fracs[2]*aFrac[3]*aFrac[4]*coVal[12]+
			aFrac[0]*fracs[1]*fracs[2]*aFrac[3]*fracs[4]*coVal[13]+
			aFrac[0]*fracs[1]*fracs[2]*fracs[3]*aFrac[4]*coVal[14]+
			aFrac[0]*fracs[1]*fracs[2]*fracs[3]*fracs[4]*coVal[15]+
			fracs[0]*aFrac[1]*aFrac[2]*aFrac[3]*aFrac[4]*coVal[16]+
			fracs[0]*aFrac[1]*aFrac[2]*aFrac[3]*fracs[4]*coVal[17]+
			fracs[0]*aFrac[1]*aFrac[2]*fracs[3]*aFrac[4]*coVal[18]+
			fracs[0]*aFrac[1]*aFrac[2]*fracs[3]*fracs[4]*coVal[19]+
			fracs[0]*aFrac[1]*fracs[2]*aFrac[3]*aFrac[4]*coVal[20]+
			fracs[0]*aFrac[1]*fracs[2]*aFrac[3]*fracs[4]*coVal[21]+
			fracs[0]*aFrac[1]*fracs[2]*fracs[3]*aFrac[4]*coVal[22]+
			fracs[0]*aFrac[1]*fracs[2]*fracs[3]*fracs[4]*coVal[23]+
			fracs[0]*fracs[1]*aFrac[2]*aFrac[3]*aFrac[4]*coVal[24]+
			fracs[0]*fracs[1]*aFrac[2]*aFrac[3]*fracs[4]*coVal[25]+
			fracs[0]*fracs[1]*aFrac[2]*fracs[3]*aFrac[4]*coVal[26]+
			fracs[0]*fracs[1]*aFrac[2]*fracs[3]*fracs[4]*coVal[27]+
			fracs[0]*fracs[1]*fracs[2]*aFrac[3]*aFrac[4]*coVal[28]+
			fracs[0]*fracs[1]*fracs[2]*aFrac[3]*fracs[4]*coVal[29]+
			fracs[0]*fracs[1]*fracs[2]*fracs[3]*aFrac[4]*coVal[30]+
			fracs[0]*fracs[1]*fracs[2]*fracs[3]*fracs[4]*coVal[31];
		break;
	default:
		printf("Unsupported dim %d in lookupValueML()\n",dims);
	}

	delete[] aFrac;

	return val;
}

inline double interpWeight(double dist2) {		return exp(-7.0*dist2);		}



int collapseMask[5][5][2][16] = {
	{{{1},  {0}}},

	{{{2,3},  {0,1}},
	 {{1,3},  {0,2}}},

	{{{4,5,6,7},  {0,1,2,3}},
	 {{2,3,6,7},  {0,1,4,5}},
	 {{1,3,5,7},  {0,2,4,6}}},

	{{{8,9,10,11,12,13,14,15},   {0,1,2,3,4,5,6,7}},
	 {{4,5,6,7,12,13,14,15},     {0,1,2,3,8,9,10,11}},
	 {{2,3,6,7,10,11,14,15},     {0,1,4,5,8,9,12,13}},
	 {{1,3,5,7,9,11,13,15},      {0,2,4,6,8,10,12,14}}},

	{{{16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31},   {0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15}},
	 {{8,9,10,11,12,13,14,15,24,25,26,27,28,29,30,31},     {0,1,2,3,4,5,6,7,16,17,18,19,20,21,22,23}},
	 {{4,5,6,7,12,13,14,15,20,21,22,23,28,29,30,31},       {0,1,2,3,8,9,10,11,16,17,18,19,24,25,26,27}},
	 {{2,3,6,7,10,11,14,15,18,19,22,23,26,27,30,31},       {0,1,4,5,8,9,12,13,16,17,20,21,24,25,28,29}},
	 {{1,3,5,7,9,11,13,15,17,19,21,23,25,27,29,31},        {0,2,4,6,8,10,12,14,16,18,20,22,24,26,28,30}}}
};

const double COLLAPSE_MARGIN = 0.05;

//interpolation by weighting based on distance - ignores bad values
double DP::doInterpDW(const double *coVal, double *fracs, int dims, bool *mask) const {
	double val = 0;
	double weightTot = 0;
	double *aFrac = new double[dims];	//anti-fractions (1-fracs)
#ifdef DEBUG
	if(VERBOSE) printf("Begin Do InterpDW\n");
#endif


	//collapse irrelevant dimensions by deciding which points to ignore
	for(int i = 0; i < dims; i++) {
		if(fracs[i] < COLLAPSE_MARGIN) {
			for(int j = 0; j < util::intPow(2,dims-1); j++) {
				mask[collapseMask[dims-1][i][0][j]] = false;
				
			}
		}
		else if(fracs[i] > 1.0-COLLAPSE_MARGIN) {
			for(int j = 0; j < util::intPow(2,dims-1); j++) {
				mask[collapseMask[dims-1][i][1][j]] = false;
			}
		}
	}

	for(int i = 0; i < dims; i++) {	
		aFrac[i] = 1.0-fracs[i];
		aFrac[i] = aFrac[i]*aFrac[i];
		fracs[i] = fracs[i]*fracs[i];
	}

	double *dists2 = new double[util::intPow(2,dims)];

	switch(dims) {
	case 1:
		dists2[0] = fracs[0];
		dists2[1] = aFrac[0];
		break;
	case 2:
		dists2[0] = fracs[0]+fracs[1];
		dists2[1] = fracs[0]+aFrac[1];
		dists2[2] = aFrac[0]+fracs[1];
		dists2[3] = aFrac[0]+aFrac[1];
		break;
	case 3:
		dists2[0] = fracs[0]+fracs[1]+fracs[2];
		dists2[1] = fracs[0]+fracs[1]+aFrac[2];
		dists2[2] = fracs[0]+aFrac[1]+fracs[2];
		dists2[3] = fracs[0]+aFrac[1]+aFrac[2];
		dists2[4] = aFrac[0]+fracs[1]+fracs[2];
		dists2[5] = aFrac[0]+fracs[1]+aFrac[2];
		dists2[6] = aFrac[0]+aFrac[1]+fracs[2];
		dists2[7] = aFrac[0]+aFrac[1]+aFrac[2];
		break;
	case 4:
		dists2[0] = fracs[0]+fracs[1]+fracs[2]+fracs[3];
		dists2[1] = fracs[0]+fracs[1]+fracs[2]+aFrac[3];
		dists2[2] = fracs[0]+fracs[1]+aFrac[2]+fracs[3];
		dists2[3] = fracs[0]+fracs[1]+aFrac[2]+aFrac[3];
		dists2[4] = fracs[0]+aFrac[1]+fracs[2]+fracs[3];
		dists2[5] = fracs[0]+aFrac[1]+fracs[2]+aFrac[3];
		dists2[6] = fracs[0]+aFrac[1]+aFrac[2]+fracs[3];
		dists2[7] = fracs[0]+aFrac[1]+aFrac[2]+aFrac[3];
		dists2[8] = aFrac[0]+fracs[1]+fracs[2]+fracs[3];
		dists2[9] = aFrac[0]+fracs[1]+fracs[2]+aFrac[3];
		dists2[10] = aFrac[0]+fracs[1]+aFrac[2]+fracs[3];
		dists2[11] = aFrac[0]+fracs[1]+aFrac[2]+aFrac[3];
		dists2[12] = aFrac[0]+aFrac[1]+fracs[2]+fracs[3];
		dists2[13] = aFrac[0]+aFrac[1]+fracs[2]+aFrac[3];
		dists2[14] = aFrac[0]+aFrac[1]+aFrac[2]+fracs[3];
		dists2[15] = aFrac[0]+aFrac[1]+aFrac[2]+aFrac[3];
		break;
	case 5:
		dists2[0] = fracs[0]+fracs[1]+fracs[2]+fracs[3]+fracs[4];
		dists2[1] = fracs[0]+fracs[1]+fracs[2]+fracs[3]+aFrac[4];
		dists2[2] = fracs[0]+fracs[1]+fracs[2]+aFrac[3]+fracs[4];
		dists2[3] = fracs[0]+fracs[1]+fracs[2]+aFrac[3]+aFrac[4];
		dists2[4] = fracs[0]+fracs[1]+aFrac[2]+fracs[3]+fracs[4];
		dists2[5] = fracs[0]+fracs[1]+aFrac[2]+fracs[3]+aFrac[4];
		dists2[6] = fracs[0]+fracs[1]+aFrac[2]+aFrac[3]+fracs[4];
		dists2[7] = fracs[0]+fracs[1]+aFrac[2]+aFrac[3]+aFrac[4];
		dists2[8] = fracs[0]+aFrac[1]+fracs[2]+fracs[3]+fracs[4];
		dists2[9] = fracs[0]+aFrac[1]+fracs[2]+fracs[3]+aFrac[4];
		dists2[10] = fracs[0]+aFrac[1]+fracs[2]+aFrac[3]+fracs[4];
		dists2[11] = fracs[0]+aFrac[1]+fracs[2]+aFrac[3]+aFrac[4];
		dists2[12] = fracs[0]+aFrac[1]+aFrac[2]+fracs[3]+fracs[4];
		dists2[13] = fracs[0]+aFrac[1]+aFrac[2]+fracs[3]+aFrac[4];
		dists2[14] = fracs[0]+aFrac[1]+aFrac[2]+aFrac[3]+fracs[4];
		dists2[15] = fracs[0]+aFrac[1]+aFrac[2]+aFrac[3]+aFrac[4];
		dists2[16] = aFrac[0]+fracs[1]+fracs[2]+fracs[3]+fracs[4];
		dists2[17] = aFrac[0]+fracs[1]+fracs[2]+fracs[3]+aFrac[4];
		dists2[18] = aFrac[0]+fracs[1]+fracs[2]+aFrac[3]+fracs[4];
		dists2[19] = aFrac[0]+fracs[1]+fracs[2]+aFrac[3]+aFrac[4];
		dists2[20] = aFrac[0]+fracs[1]+aFrac[2]+fracs[3]+fracs[4];
		dists2[21] = aFrac[0]+fracs[1]+aFrac[2]+fracs[3]+aFrac[4];
		dists2[22] = aFrac[0]+fracs[1]+aFrac[2]+aFrac[3]+fracs[4];
		dists2[23] = aFrac[0]+fracs[1]+aFrac[2]+aFrac[3]+aFrac[4];
		dists2[24] = aFrac[0]+aFrac[1]+fracs[2]+fracs[3]+fracs[4];
		dists2[25] = aFrac[0]+aFrac[1]+fracs[2]+fracs[3]+aFrac[4];
		dists2[26] = aFrac[0]+aFrac[1]+fracs[2]+aFrac[3]+fracs[4];
		dists2[27] = aFrac[0]+aFrac[1]+fracs[2]+aFrac[3]+aFrac[4];
		dists2[28] = aFrac[0]+aFrac[1]+aFrac[2]+fracs[3]+fracs[4];
		dists2[29] = aFrac[0]+aFrac[1]+aFrac[2]+fracs[3]+aFrac[4];
		dists2[30] = aFrac[0]+aFrac[1]+aFrac[2]+aFrac[3]+fracs[4];
		dists2[31] = aFrac[0]+aFrac[1]+aFrac[2]+aFrac[3]+aFrac[4];		
		break;
	default:
		printf("Unsupported dim %d in lookupValueDW()\n",dims);
	}


	double done = intPow(2,dims);
	for(int i = 0; i < done; i++) {
		if(mask[i]) {
			double weight = interpWeight(dists2[i]);
			val += weight*coVal[i];
			weightTot += weight;
		}
	}
#ifdef DEBUG
	if(VERBOSE) {
		printf("\n\nFRACS:\n");
		for(int i = 0; i < dims; i++) printf("%g\n",sqrt(fracs[i]));
		printf("\n\nDists   Weights  Mask:\n");
		for(int i = 0; i < intPow(2,dims); i++) printf("%g     %g         %d\n",sqrt(dists2[i]),interpWeight(dists2[i]),mask[i]);
	}
#endif

	delete[] aFrac;
	delete[] dists2;

	if(weightTot == 0)	return INFINITE_COST;
	return val/weightTot;
}

void DP::readV(const char *name) {
	char fullName[100];
	if(N_DYN_TYPES>1) {
		for(int i = 0; i < N_DYN_TYPES; i++) {
			if(N_DIMS[i] == 2) {
				sprintf_s(fullName,"%s_%d",name,i);
				int dim = dimIndex(0,i);
				readSSV(fullName,&V[statesBeforeType[i]],RES[dim+1]*RES[dim]);
			}
			else {
				sprintf_s(fullName,"%s_%d_dims%d",name,i,N_DIMS[i]);	
				readSSV(fullName,&V[statesBeforeType[i]],N_TYPE_STATES[i]);
			}
		}
	}
	else {
		if(N_DIMS[0] == 2)	readSSV(name,V,RES[1]*RES[0]);
		else {
			sprintf_s(fullName,"%s_dims%d",name,N_DIMS[0]);	
			readSSV(fullName,&V[0],N_TOTAL_STATES);
		}
	}
}

void DP::writeV(const char *name) const {
	char fullName[100];
	if(N_DYN_TYPES>1) {
		for(int i = 0; i < N_DYN_TYPES; i++) {
			if(N_DIMS[i] == 2) {
				sprintf_s(fullName,"%s_%d",name,i);
				int dim = dimIndex(0,i);
				writeSSV(fullName,&V[statesBeforeType[i]],RES[dim+1],RES[dim]);
			}
			else {
				sprintf_s(fullName,"%s_%d_dims%d",name,i,N_DIMS[i]);	
				writeSSV(fullName,&V[statesBeforeType[i]],1,N_TYPE_STATES[i]);
			}
		}
	}
	else {
		if(N_DIMS[0] == 2)	writeSSV(name,V,RES[1],RES[0]);
		else {
			sprintf_s(fullName,"%s_dims%d",name,N_DIMS[0]);	
			writeSSV(fullName,&V[0],1,N_TOTAL_STATES);
		}
	}
}

void DP::readU(const char *name) {
	char fullName[100];
	if(N_DYN_TYPES>1) {
		for(int i = 0; i < N_DYN_TYPES; i++) {
			if(N_DIMS[i] == 2) {
				for(int j = 0; j < N_ACTION_DIMS[i]; j++) {
					sprintf_s(fullName,"%s%d_%d",name,i,j);
					int dim = dimIndex(0,i);
					readSSVlayer(fullName,&U[statesBeforeType[i]],RES[dim+1]*RES[dim],j);
				}
			}
			else {
				for(int j = 0; j < N_ACTION_DIMS[i]; j++) {
					sprintf_s(fullName,"%s%d_%d_dims%d",name,i,j,N_DIMS[i]);	
					readSSVlayer(fullName,&U[statesBeforeType[i]],N_TYPE_STATES[i],j);
				}
			}
		}
	}
	else {
		if(N_DIMS[0] == 2) {
			for(int i = 0; i < N_ACTION_DIMS[0]; i++) {
				sprintf_s(fullName,"%s%d",name,i);
				readSSVlayer(fullName,U,RES[1]*RES[0],i);
			}
		}
		else {
			for(int j = 0; j < N_ACTION_DIMS[0]; j++) {
				sprintf_s(fullName,"%s%d_dims%d",name,j,N_DIMS[0]);	
				readSSVlayer(fullName,&U[0],N_TOTAL_STATES,j);
			}
		}
	}
}


void DP::readAux(const char *name) {
	char fullName[100];
	if(N_DYN_TYPES>1) {
		for(int i = 0; i < N_DYN_TYPES; i++) {
			if(N_DIMS[i] == 2) {
				for(int j = 0; j < N_AUX; j++) {
					sprintf_s(fullName,"%s%d_%d",name,i,j);
					int dim = dimIndex(0,i);
					readSSVlayer(fullName,&aux[statesBeforeType[i]],RES[dim+1]*RES[dim],j);
				}
			}
			else {
				for(int j = 0; j < N_AUX; j++) {
					sprintf_s(fullName,"%s%d_%d",name,i,j);	
					readSSVlayer(fullName,&aux[statesBeforeType[i]],N_TYPE_STATES[i],j);
				}
			}
		}
	}
	else {
		if(N_DIMS[0] == 2) {
			for(int i = 0; i < N_AUX; i++) {
				sprintf_s(fullName,"%s%d",name,i);
				readSSVlayer(fullName,aux,RES[1]*RES[0],i);
			}
		}
		else {
			for(int j = 0; j < N_AUX; j++) {
				sprintf_s(fullName,"%s%d",name,j);	
				readSSVlayer(fullName,&aux[0],N_TOTAL_STATES,j);
			}
		}
	}
}



void DP::writeU(const char *name) const {

	char fullName[100];
	if(N_DYN_TYPES>1) {
		for(int i = 0; i < N_DYN_TYPES; i++) {
			if(N_DIMS[i] == 2) {
				for(int j = 0; j < N_ACTION_DIMS[i]; j++) {
					sprintf_s(fullName,"%s%d_%d",name,i,j);
					int dim = dimIndex(0,i);
					writeSSVlayer(fullName,&U[statesBeforeType[i]],RES[dim+1],RES[dim],j);
				}
			}
			else {
				for(int j = 0; j < N_ACTION_DIMS[i]; j++) {
					sprintf_s(fullName,"%s%d_%d_dims%d",name,i,j,N_DIMS[i]);	
					writeSSVlayer(fullName,&U[statesBeforeType[i]],1,N_TYPE_STATES[i],j);
				}
			}
		}
	}
	else {
		if(N_DIMS[0] == 2) {
			for(int i = 0; i < N_ACTION_DIMS[0]; i++) {
				sprintf_s(fullName,"%s%d",name,i);
				writeSSVlayer(fullName,U,RES[1],RES[0],i);
			}
		}
		else {
			for(int j = 0; j < N_ACTION_DIMS[0]; j++) {
				sprintf_s(fullName,"%s%d_dims%d",name,j,N_DIMS[0]);	
				writeSSVlayer(fullName,&U[0],1,N_TOTAL_STATES,j);
			}
		}
	}
}

void DP::writeAux(const char *name) const {
	char fullName[100];
	if(N_DYN_TYPES>1) {
		for(int i = 0; i < N_DYN_TYPES; i++) {
			if(N_DIMS[i] == 2) {
				for(int j = 0; j < N_AUX; j++) {
					sprintf_s(fullName,"%s%d_%d",name,i,j);
					int dim = dimIndex(0,i);
					writeSSVlayer(fullName,&aux[statesBeforeType[i]],RES[dim+1],RES[dim],j);
				}
			}
			else {
				for(int j = 0; j < N_AUX; j++) {
					sprintf_s(fullName,"%s%d_%d",name,i,j);	
					writeSSVlayer(fullName,&aux[statesBeforeType[i]],1,N_TYPE_STATES[i],j);
				}
			}
		}
	}
	else {
		if(N_DIMS[0] == 2) {
			for(int i = 0; i < N_AUX; i++) {
				sprintf_s(fullName,"%s%d",name,i);
				writeSSVlayer(fullName,aux,RES[1],RES[0],i);
			}
		}
		else {
			for(int j = 0; j < N_AUX; j++) {
				sprintf_s(fullName,"%s%d",name,j);	
				writeSSVlayer(fullName,&aux[0],1,N_TOTAL_STATES,j);
			}
		}
	}
}


bool DP::isCorner(const int *base, const int *indices, int dims) const {
	if(VERBOSE)	printf("checking corner\n");
	for(int i = 0; i < dims; i++) {
		if(VERBOSE)	printf("%d: %d %d\n",i, base[i], indices[i]);
		if(indices[i] != base[i] && indices[i] != base[i]-1)	return false;
	}
	if(VERBOSE)	printf("same corner\n");
	return true;
}

#ifdef MULTI_MODEL
double DP::actionValue(const double *state, double *action, int mod,  int type, const int *indices, bool verbose) const {
#else
double DP::actionValue(const double *state, double *action, int type, const int *indices, bool verbose) const {
#endif
	double discount = DISCOUNT;
	double goState[MAX_DIMS_EVER];
	int goType = type;	//initialized so 1-type dynamics don't have to


#ifdef DEBUG
	if(verbose) {printf("BEGIN ACTION VALUE\nstate0 ");printArray(state,N_DIMS[type]);printf("type = %d\n",type);}
	if(verbose) {printf("action ");printArray(action,N_ACTION_DIMS[type]);}
#endif

#ifdef MULTI_MODEL
	double cost = dynList[mod]->integrateCost(state, action, goState, type, &goType);
#else
	double cost = dynamics->integrateCost(state, action, goState, type, &goType);
#endif

#ifdef DEBUG
	if(verbose) {printf("cost1 = %f\n",cost);}
	if(verbose)	{printf("state1 ");printArray(goState,N_DIMS[type]);printf("type = %d\n",goType);}
#endif

	int goIndices[MAX_DIMS_EVER];
	//while(sameTile(state, goState,type,goType)&&!inGoalRegion(goState,goType,this)) {
	while(true) {
		if(inGoalRegion(goState,type,this)) {
#ifdef DEBUG
			if(verbose) {printf("In Goal Region - value = %g\n",goalRegionValue(state, goType));}
			if(verbose)	{printf("Total cost is %g\n\n",cost + discount*goalRegionValue(state, goType));}
#endif
			return cost + discount*goalRegionValue(state, goType);
		}
		if(!getBaseInds(goState,goType,goIndices))			return INFINITE_COST;
		if(!isCorner(indices, goIndices, N_DIMS[type]))		break;
		if(goType != type)									break;

#ifdef MULTI_MODEL
		cost += discount*dynList[mod]->integrateCost(goState, action, goState, type, &goType);
#else
		cost += discount*dynamics->integrateCost(goState, action, goState, type, &goType);
#endif

		discount *= DISCOUNT;
#ifdef DEBUG
		if(verbose) {printf("new state ");printArray(goState,N_DIMS[type]);printf("type = %d\n",goType);}
		if(verbose) {printf("cost increased to %f\n",cost);}
#endif
	}

#ifdef MULTI_MODEL
	double endVal = lookupValue(goState, mod, goType);
#else
	double endVal = lookupValue(goState, goType);
#endif

#ifdef DEBUG
	if(verbose) {printf("Value Function at final state = %f\n",endVal);}
#endif

	if(endVal == INFINITE_COST)		cost = INFINITE_COST;
	else							cost += endVal*discount;


#ifdef DEBUG
	if(verbose) {printf("final cost = %f\nEND ACTION VALUE\n\n",cost);}
#endif
	return cost;
}

/*
double DP::interpActionValue(const double *state, double *newAction, int type, const int *indices, bool verbose) const {
	double discount = DISCOUNT;
	double goState[MAX_DIMS_EVER];
	int goType = type;	//initialized so 1-type dynamics don't have to

	int ind = getI1d(indices,type);
	double holdV = V[ind];
	double *holdU = U[ind];

	V[ind] = 1;
	U[ind] = newAction;

#ifdef DEBUG
	if(verbose) {printf("BEGIN ACTION VALUE\nstate0 ");printArray(state,N_DIMS[type]);printf("type = %d\n",type);}
	if(verbose) {printf("action ");printArray(newAction,N_ACTION_DIMS[type]);}
#endif
	double action[MAX_ACTION_DIMS_EVER];
	lookupPolicy(action,state,type);

	double cost = dynamics->integrateCost(state, action, goState, type, &goType);
#ifdef DEBUG
	if(verbose) {printf("cost1 = %f\n",cost);}
	if(verbose)	{printf("state1 ");printArray(goState,N_DIMS[type]);printf("type = %d\n",goType);}
#endif

	int goIndices[MAX_DIMS_EVER];

	//while(sameTile(state, goState,type,goType)&&!inGoalRegion(goState,goType,this)) {
	int count = 0;
	while(true) {
		count++;
		if(count>200)	{
			V[ind] = holdV;
			U[ind] = holdU;
			return INFINITE_COST;
		}
		if(inGoalRegion(goState,type,this)) {
#ifdef DEBUG
			if(verbose) {printf("In Goal Region - value = %g\n",goalRegionValue(state, goType));}
			if(verbose)	{printf("Total cost is %g\n\n",cost + discount*goalRegionValue(state, goType));}
#endif
			return cost + discount*goalRegionValue(state, goType);
		}
		if(!getBaseInds(goState,goType,goIndices))	{
			V[ind] = holdV;
			U[ind] = holdU;
			return INFINITE_COST;
		}
		if(!isCorner(indices, goIndices, N_DIMS[type]))		break;
		if(goType != type)									break;

		lookupPolicy(action,goState,goType);
		cost += discount*dynamics->integrateCost(goState, action, goState, type, &goType);
		discount *= DISCOUNT;
#ifdef DEBUG
		if(verbose) {printf("new state ");printArray(goState,N_DIMS[type]);printf("type = %d\n",goType);}
		if(verbose) {printf("cost increased to %f\n",cost);}
#endif
	}

	double endVal = lookupValue(goState, goType);
#ifdef DEBUG
	if(verbose) {printf("Value Function at final state = %f\n",endVal);}
#endif
	if(endVal == INFINITE_COST)		cost = INFINITE_COST;
	else							cost += endVal*discount;


#ifdef DEBUG
	if(verbose) {printf("final cost = %f\nEND ACTION VALUE\n\n",cost);}
#endif

	V[ind] = holdV;
	U[ind] = holdU;
	return cost;
}
*/


//like actionValue, but continues simulating until you hit a cell that can do ML interpolation
//preferred version
#ifdef MULTI_MODEL
double DP::actionValueML(const double *state, double *action, int mod, int type, const int *indices, bool verbose) const {
#else
double DP::actionValueML(const double *state, double *action, int type, const int *indices, bool verbose) const {
#endif
	//int ind1 = getI1d(indices,type);
	//if(ind1 == 79814942)	verbose = true;
	double discount = DISCOUNT;
	double goState[MAX_DIMS_EVER];

	double simAction[MAX_ACTION_DIMS_EVER];

	for(int i = 0; i < N_ACTION_DIMS[type]; i++)	simAction[i] = action[i];

	int goType = type;	//initialized so 1-type dynamics don't have to
#ifdef DEBUG
	if(verbose) {printf("BEGIN ACTION VALUE\nstate0 ");printArray(state,N_DIMS[type]);printf("type = %d\n",type);}
	if(verbose) {printf("action ");printArray(action,N_ACTION_DIMS[type]);}
#endif
#ifdef MULTI_MODEL
	double cost = dynList[mod]->integrateCost(state, simAction, goState, type, &goType);
#else
	double cost = dynamics->integrateCost(state, simAction, goState, type, &goType);
#endif

#ifdef DEBUG
	if(verbose) {printf("cost1 = %f\n",cost);}
	if(verbose)	{printf("state1 ");printArray(goState,N_DIMS[goType]);printf("type = %d\n",goType);}
#endif

	int prevCell = -1;
	int prevType = type;

	int numCells = 0;
	int loops = 0;
	//loop until in goal region, out of bounds, or in a fully defined cell
	while(true) {
		loops++;
		if(loops > 10000) {
			return INFINITE_COST;
		}

		if(inGoalRegion(goState,type,this)) {
#ifdef DEBUG
			if(verbose) {printf("In Goal Region - value = %g\n",goalRegionValue(state, goType));}
			if(verbose)	{printf("Total cost is %g\n\n",cost + discount*goalRegionValue(state, goType));}
#endif
			return cost + discount*goalRegionValue(state, goType);
		}

		int goIndices[MAX_DIMS_EVER];
		int cell = getI1dFromState(goState, goType, goIndices);
#ifdef DEBUG
		if(verbose) {printf("In cell %d\n",cell);}
#endif

		//if out of bounds - bad
		if(cell == -1) {
#ifdef DEBUG
			if(verbose) {printf("Out of bounds - returning INFINITE_COST\n\n");}
#endif
			return INFINITE_COST;
		}


		bool tooClose = true;	//
		//only check for completion when in new cells
		if(cell != prevCell) {
			numCells++;
			if(numCells > 7) {
#ifdef MULTI_MODEL
				double val = lookupValue(goState, mod, goType);
#else
				double val = lookupValue(goState, goType);
#endif
				if(val == INFINITE_COST)	return val;
				return cost + discount*val;
			}
#ifdef DEBUG
			if(verbose) {printf("Entering Cell %d from cell %d\n",cell, prevCell);}
#endif


			tooClose = isCorner(indices, goIndices, N_DIMS[type])    &&    (type == goType);
#ifdef DEBUG
			if(verbose) {
				if(tooClose)	printf("This tile is too close.\n");
				else			printf("This tile is not too close.\n");
			}
#endif
			if(!tooClose) {
				//lookup corner values
				bool lookForGood = false;
				bool hasBadValue = false;
				int i = 0;
				for(; i < corners[goType]; i++) {
#ifdef MULTI_MODEL
					if(Vlist[mod][cell+cornerOffsets[goType][i]] >= INFINITE_COST) {
#else
					if(V[cell+cornerOffsets[goType][i]] >= INFINITE_COST) {
#endif
#ifdef DEBUG
						if(verbose) printf("Corner number %d is bad\n",i);
#endif
						hasBadValue = true;
						//first value is bad - now need to see if it's mixed or all bad
						if(i == 0)	lookForGood = true;
						break;
					}
				}
			
				//if this is a good cell to interpolate value in
				//normal good return path
				if(!hasBadValue) {
#ifdef DEBUG
#ifdef MULTI_MODEL
					double val = lookupValue(goState, mod, goType);
#else	//not MULTI_MODEL
					double val = lookupValue(goState, goType);
#endif	//end MULTI_MODEL
					if(verbose) {printf("Value Function at final state = %f\n",val);}
					if(verbose) {printf("final cost = %f\nEND ACTION VALUE\n\n",cost + discount*val);}
					return cost + discount*val;
#else		//not DEBUG
#ifdef MULTI_MODEL
					return cost + discount*lookupValue(goState, mod, goType); 
#else	//not MULTI_MODEL
					return cost + discount*lookupValue(goState, goType); 
#endif	//end MULTI_MODEL
#endif	//end debug
				}
				//if must check for good values
				if(lookForGood) {
					bool hasGoodValue = false;
					for(; i < corners[goType]; i++) {
						if(V[cell+cornerOffsets[goType][i]] < INFINITE_COST) {
#ifdef DEBUG
							if(verbose) printf("Corner number %d is good\n",i);
#endif	
							hasGoodValue = true;
							break;
						}
					}
					if(!hasGoodValue) {
#ifdef DEBUG
						if(verbose) {printf("Entered a bad cell - returning INFINITE_COST\n");}
#endif
						return INFINITE_COST;
					}
				}
				if(goType != prevType) {
					lookupPolicy(simAction,goState,goType);
#ifdef DEBUG
					if(verbose) { printf("New type - so selecting a new action:\n");	util::printArray(simAction,N_ACTION_DIMS[goType]);}
#endif
				}
			}
			prevCell = cell;
			prevType = goType;
		}

		//sim forward
		if(true) {
#ifdef MULTI_MODEL
			cost += discount*dynList[mod]->integrateCost(goState, simAction, goState, goType, &goType);
#else
			cost += discount*dynamics->integrateCost(goState, simAction, goState, goType, &goType);
#endif
		}
		else {	//lookup action if outside of the region local to the tested grid point
			double localAction[MAX_ACTION_DIMS_EVER];
			lookupPolicy(localAction,goState,goType);
#ifdef MULTI_MODEL
			cost += discount*dynList[mod]->integrateCost(goState, localAction, goState, goType, &goType);
#else
			cost += discount*dynamics->integrateCost(goState, localAction, goState, goType, &goType);
#endif
		}

		discount *= DISCOUNT;
#ifdef DEBUG
		if(verbose) {printf("new state ");printArray(goState,N_DIMS[goType]);printf("type = %d\n",goType);}
		if(verbose) {printf("cost increased to %f\n",cost);}
#endif
	}


}

#ifdef ENABLE_MAPPING
void DP::setMaps(void (*grid2nat)(const double *gridState, double *natState, int type), void (*nat2grid)(const double *natState, double *gridState, int type)) {
	grid2natFunk = grid2nat;
	nat2gridFunk = nat2grid;

	//test that the maps are inverses
	double state1[MAX_DIMS_EVER], state2[MAX_DIMS_EVER], state11[MAX_DIMS_EVER];
	for(int i = 0; i < N_DYN_TYPES; i++) {
		//get a random vector
		for(int j = 0; j < N_DIMS[i]; j++)	state1[i] = util::randMM(-1,1);
		grid2natFunk(state1, state2, i);
		nat2gridFunk(state2, state11, i);
		for(int j = 0; j < N_DIMS[i]; j++) {
			if(!util::almostEqual(state1[i], state11[i])) {
				printf("Bad Mapping in type %d\n", i);
				printf("Started with:    ");	util::printArray(state1,  N_DIMS[i]);
				printf("Finished with:   ");	util::printArray(state11, N_DIMS[i]);
				exit(-1);
			}
		}
	}
}
#endif

//fully general
//can speed it up in reasonable cases - get rid of loop and stateBeforeType
int DP::getI1d(const int *indices, int type) const {
	int ind = statesBeforeType[type];
	int dimShort = N_DIMS[type]-1;
	for(int i = 0; i < dimShort; i++) {
		ind += indices[i]*FREQ1D[dimIndex(i,type)];
	}
	return ind + indices[dimShort];
}

//completely general
//should be sped up
//also not tested - may not work
void DP::getImd(int ind1d, int &type, int *indices) const{
	type = 0;
	for(int i = 1; i < N_DYN_TYPES; i++) {
		if(ind1d >= statesBeforeType[i])	type = i;
	}
	ind1d -= statesBeforeType[type];
	for(int i = 0; i < N_DIMS[type]; i++) {
		indices[i] = ind1d/FREQ1D[dimIndex(i,type)];
		ind1d = ind1d%FREQ1D[dimIndex(i,type)];
	}
}

bool DP::getBaseInds(const double *state, int type, int *indices) const {

	int dims = N_DIMS[type];

	int dim = type*MAX_DIMS;
#ifdef ENABLE_MAPPING
	double stateGrid[MAX_DIMS_EVER];
	nat2gridFunk(state, stateGrid, type);

	for(int i = 0; i < dims; i++) {
		indices[i] = (int)floor((stateGrid[i]-MIN_STATE[dim])/dx[dim]);
		if(indices[i] >= RES[dim]-1) {
			if(stateGrid[i] == MAX_STATE[dim])		indices[i] -= 1;
#else
	for(int i = 0; i < dims; i++) {
		indices[i] = (int)floor((state[i]-MIN_STATE[dim])/dx[dim]);
		if(indices[i] >= RES[dim]-1) {
			if(state[i] == MAX_STATE[dim])		indices[i] -= 1;
#endif 
			else								return false;
		}
		if(indices[i] < 0)						return false;
		dim++;
	}
	

	return true;

}

//returns -1 if out of bounds
int DP::getI1dFromState(const double *state, int type) const {
	int dims = N_DIMS[type];
	int baseInd[MAX_DIMS_EVER];

	if(!getBaseInds(state,type,baseInd))	return -1;

#ifdef DEBUG
	if(VERBOSE) {
		printf("inds: ");	util::printArray(baseInd,N_DIMS[type]);
	}
#endif

	return getI1d(baseInd, type);
}

//returns -1 if out of bounds
int DP::getI1dFromState(const double *state, int type, int *baseInd) const {
	int dims = N_DIMS[type];

	if(!getBaseInds(state,type,baseInd))	return -1;

#ifdef DEBUG
	if(VERBOSE) {
		printf("inds: ");	util::printArray(baseInd,N_DIMS[type]);
	}
#endif

	return getI1d(baseInd, type);
}

double *DP::getState(const int *indices, int type, double *state) const {
#ifdef ENABLE_MAPPING
	double gridState[MAX_DIMS_EVER];
#endif
	int dims = N_DIMS[type];

	int dimI = dimIndex(0,type);
	for(int i = 0; i < dims; i++) {
#ifdef ENABLE_MAPPING
		gridState[i] = MIN_STATE[dimI] + dx[dimI]*indices[i];
#else
		state[i] = MIN_STATE[dimI] + dx[dimI]*indices[i];
#endif
		dimI++;
	}
#ifdef ENABLE_MAPPING
	grid2natFunk(gridState, state, type);
#endif
	return state;
}


#ifdef THREADING

typedef struct fragWrapBundle {
	DP *dp;
	long long start;
	long long end;
	unsigned int seed;
	int threadNum;
	SweepStats *ss;
} FragWrapBundle;

unsigned _stdcall sweepFragWrapper(void *params) {
	FragWrapBundle *FWB = (FragWrapBundle*)params;
	srand(FWB->seed);		//very important
	//if(FWB->threadNum==0)	printf("%f\n",util::rand1());
	//printf("Starting frag from %d to %d\n",FWB->start,FWB->end);
	FWB->dp->doSweepFrag(FWB->start, FWB->end, FWB->threadNum, *(FWB->ss));
	//printf("Finishing frag from %d to %d\n",FWB->start,FWB->end);
	return 0;
}


unsigned _stdcall VsweepFragWrapper(void *params) {
	FragWrapBundle *FWB = (FragWrapBundle*)params;
	srand(FWB->seed);		//very important
	//if(FWB->threadNum==0)	printf("%f\n",util::rand1());
	//printf("Starting frag from %d to %d\n",FWB->start,FWB->end);
	FWB->dp->doVsweepFrag(FWB->start, FWB->end, FWB->threadNum, *(FWB->ss));
	//printf("Finishing frag from %d to %d\n",FWB->start,FWB->end);
	return 0;
}

void DP::doSweepFrag(int start, int end, int threadNum, SweepStats &ss) {
	ss.dV2 = 0.0;
	ss.polChanges = 0;
	
	int *indices = new int[MAX_DIMS];	//length the right number of dimensions for this type
	//end-=(end-start)/2.0;

	//printf("sweep from %d to %d\n",start,end);
	for(int i = start; i < end; i++) {
		updateState(i, ss);		
	}
	delete[] indices;

}

void DP::doVsweepFrag(int start, int end, int threadNum, SweepStats &ss) {
	ss.dV2 = 0.0;
	ss.polChanges = 0;
	
	int *indices = new int[MAX_DIMS];	//length the right number of dimensions for this type
	//end-=(end-start)/2.0;
	int type;
	//printf("sweep from %d to %d\n",start,end);
	for(int i = start; i < end; i++) {
#ifdef MULTI_MODEL
		for(int j = 0; j < N_MODEL; j++)		updateV(i, j, ss);		
#else
		updateV(i,ss);	
#endif
	}
	delete[] indices;

}


void DP::doSweepMT(int threads) {

	allocateVtmp();
	SweepStats *ssList = new SweepStats[threads]();

	HANDLE *threadHandles = new HANDLE[threads];
	FragWrapBundle *FWB = new FragWrapBundle[threads];
	//spawn threads
	for(long long i = 0; i < threads; i++) {
		FWB[i].dp = this;
		FWB[i].start = (i*N_TOTAL_STATES)/threads;
		FWB[i].end = ((i+1)*N_TOTAL_STATES)/threads;
		FWB[i].seed = rand();
		FWB[i].threadNum = i;
		FWB[i].ss = &(ssList[i]);
		//printf("Creating thread from %d to %d\n",FWB[i].start,FWB[i].end);
		threadHandles[i] = (HANDLE)_beginthreadex(NULL,0,sweepFragWrapper,&(FWB[i]),0,NULL);
	}	
	//printf("about to wait\n");
	WaitForMultipleObjects(threads, threadHandles, TRUE, INFINITE);
	//printf("done waiting\n");

	swapVVtmp();


	for(int i = 1; i < threads; i++) {
		ssList[0] += ssList[i];
	}

	ssList[0].print();

	//double state[5] = {0.0, 0.0, 0.1, 0.0, 0.5};
	//printf("%g\n",getValueBySim(state,1,5000));



	delete[] threadHandles;
	delete[] FWB;
	delete[] ssList;

}


void DP::doVsweepMT(int threads) {
	allocateVtmp();
	SweepStats *ssList = new SweepStats[threads]();
	
	//DP *dps = new DP[threads];

	HANDLE *threadHandles = new HANDLE[threads];
	FragWrapBundle *FWB = new FragWrapBundle[threads];
	//spawn threads
	for(long long i = 0; i < threads; i++) {
		//dps[i] = sameDataCopy();
		FWB[i].dp = this;
		FWB[i].start = (i*N_TOTAL_STATES)/threads;
		FWB[i].end = ((i+1)*N_TOTAL_STATES)/threads;
		FWB[i].seed = rand();
		FWB[i].threadNum = i;
		FWB[i].ss = &(ssList[i]);
		//printf("Creating thread from %d to %d\n",FWB[i].start,FWB[i].end);
		threadHandles[i] = (HANDLE)_beginthreadex(NULL,0,VsweepFragWrapper,&(FWB[i]),0,NULL);
	}	
	//printf("about to wait\n");
	WaitForMultipleObjects(threads, threadHandles, TRUE, INFINITE);
	//printf("done waiting\n");

	swapVVtmp();


	for(int i = 1; i < threads; i++) {
		ssList[0] += ssList[i];
	}

	ssList[0].printV();


	delete[] threadHandles;
	delete[] FWB;
	delete[] ssList;
}


#endif

void DP::applyToAll(void(*func)(int ind, DP *dp)) {
	for(int i = 0; i < N_TOTAL_STATES; i++) {
			func(i, this);
	}
}

void DP::allocateVtmp() {
	
#ifdef MULTI_MODEL
	if(VtmpList[0] != NULL)		return;
	for(int i = 0; i < N_MODEL; i++) {
		VtmpList[i] = new double[N_TOTAL_STATES]();
		assert(VtmpList[i]);
	}

	Vtmp = VtmpList[0];

#else
	if(Vtmp != NULL)	return;
	Vtmp = new double[N_TOTAL_STATES]();
	assert(Vtmp);
#endif
}

void DP::doSweep(int numThreads) {
	sweep++;
	if(numThreads > 1)	{

		doSweepMT(numThreads);
		return;
	}
	if(Vtmp == NULL) {
		Vtmp = new double[N_TOTAL_STATES]();
		assert(Vtmp);
	}
	SweepStats ss;

	for(int i = 0; i < N_TOTAL_STATES; i++) {
		updateState(i, ss);		
	}

	
	swapVVtmp();

	ss.print();
}

//swap V arrays - double buffering
void DP::swapVVtmp() {
#ifdef MULTI_MODEL
	for(int i = 0; i < N_MODEL; i++) {
		double *tmp = Vlist[i];
		Vlist[i] = VtmpList[i];
		VtmpList[i] = tmp;
	}
#else
	double *tmp = V;
	V = Vtmp;
	Vtmp = tmp;
#endif
}

void DP::doVsweep(int numThreads) {
	sweep++;
	if(numThreads > 1)	{
		doVsweepMT(numThreads);
		return;
	}

	if(Vtmp == NULL) {
		Vtmp = new double[N_TOTAL_STATES]();
		assert(Vtmp);
	}
	SweepStats ss;

	int *indices = new int[MAX_DIMS];	//length the right number of dimensions for this type


	for(int i = 0; i < N_TOTAL_STATES; i++) {
#ifdef MULTI_MODEL
		for(int j = 0; j < N_MODEL; j++)	updateV(i, j, ss);		
#else
		updateV(i,ss);	
#endif
	}
	delete[] indices;

	printf("dV^2 = %g\n",ss.dV2);

	swapVVtmp();
}

//return 1 if successful
int DP::saveUV(const char *name) const {
	char fullname[100];
	sprintf_s(fullname,"%s.pol",name);
	ofstream out(fullname, ios::out | ios::binary);
	if(!out) {
		cout << "Cannot open file for write in saveUV():  " << fullname << endl;
		return 0;
	}
	
	//save V
#ifdef MULTI_MODEL
	for(int j = 0; j < N_MODEL; j++) {
		out.write((char *) Vlist[j], sizeof(double)*N_TOTAL_STATES);
	}
#else
	out.write((char *) V, sizeof(double)*N_TOTAL_STATES);
#endif

	//save U
	for(int i = 0; i < N_DYN_TYPES; i++) {
		for(int j = 0; j < N_TYPE_STATES[i]; j++) {
			out.write((char *) U[statesBeforeType[i]+j], sizeof(double)*N_ACTION_DIMS[i]);
		}
	}

	out.close();

	return 1;
}

//return 1 if successful
int DP::saveUVX(const char *name) const {
	char fullname[100];
	sprintf_s(fullname,"%s.polx",name);
	ofstream out(fullname, ios::out | ios::binary);
	if(!out) {
		cout << "Cannot open file for write in saveUV():  " << fullname << endl;
		return 0;
	}
	
	//save V
	out.write((char *) V, sizeof(double)*N_TOTAL_STATES);

	//save U
	for(int i = 0; i < N_DYN_TYPES; i++) {
		for(int j = 0; j < N_TYPE_STATES[i]; j++) {
			out.write((char *) U[statesBeforeType[i]+j], sizeof(double)*N_ACTION_DIMS[i]);
		}
	}

	//save aux
	for(int i = 0; i < N_DYN_TYPES; i++) {
		for(int j = 0; j < N_TYPE_STATES[i]; j++) {
			out.write((char *) aux[statesBeforeType[i]+j], sizeof(double)*N_AUX);
		}
	}

	out.close();

	return 1;
}

//return 1 if successful
int DP::loadUV(const char *name) {
	char fullname[100];
	sprintf_s(fullname,"%s.pol",name);
	ifstream in(fullname, ios::in | ios::binary);

	//load V
#ifdef MULTI_MODEL
	for(int j = 0; j < N_MODEL; j++) {
		in.read((char *) Vlist[j], sizeof(double)*N_TOTAL_STATES);
	}
#else
	in.read((char *) V, sizeof(double)*N_TOTAL_STATES);
#endif

	if(!in) {
		cout << "Cannot open file for read in loadUV():  " << fullname << endl;
		return 0;
	}

	//load U
	for(int i = 0; i < N_DYN_TYPES; i++) {
		for(int j = 0; j < N_TYPE_STATES[i]; j++) {
			in.read((char *) U[statesBeforeType[i]+j], sizeof(double)*N_ACTION_DIMS[i]);
		}
	}

	in.close();

	return 1;
}

//return 1 if successful
int DP::loadUVX(const char *name) {
	char fullname[100];
	sprintf_s(fullname,"%s.polx",name);
	ifstream in(fullname, ios::in | ios::binary);
	
	//load V
	in.read((char *) V, sizeof(double)*N_TOTAL_STATES);
	if(!in) {
		cout << "Cannot open file for read in loadUVX():  " << fullname << endl;
		return 0;
	}

	//load U
	for(int i = 0; i < N_DYN_TYPES; i++) {
		for(int j = 0; j < N_TYPE_STATES[i]; j++) {
			in.read((char *) U[statesBeforeType[i]+j], sizeof(double)*N_ACTION_DIMS[i]);
		}
	}

	//load aux
	for(int i = 0; i < N_DYN_TYPES; i++) {
		for(int j = 0; j < N_TYPE_STATES[i]; j++) {
			in.read((char *) aux[statesBeforeType[i]+j], sizeof(double)*N_AUX);
		}
	}

	in.close();

	return 1;
}

void DP::writeUV(const char *prefix, int suffix) const {
	char name[100];
	sprintf_s(name,"%sV%d",prefix,suffix);
	writeV(name);
	sprintf_s(name,"%sU%d_",prefix,suffix);
	writeU(name);
}

void DP::writeUV(const char *prefix) const {
	char name[100];
	sprintf_s(name,"%sV",prefix);
	writeV(name);
	sprintf_s(name,"%sU",prefix);
	writeU(name);
}

void DP::readUV(const char *prefix, int suffix) {
	char name[100];
	sprintf_s(name,"%sV%d",prefix,suffix);
	readV(name);
	sprintf_s(name,"%sU%d_",prefix,suffix);
	readU(name);
}

void DP::readUV(const char *prefix) {
	char name[100];
	sprintf_s(name,"%sV",prefix);
	readV(name);
	sprintf_s(name,"%sU",prefix);
	readU(name);
}

void DP::doManySweeps(int N, int recFreq, const char *fileName, int numThreads, bool justV) {
	int start = clock();
	//ofstream trajFile("trajCost.out");
	for(int i = 0; i < N; i++) {
		printf("sweep %d\n",i);
		int sweep_start = clock();
		if(justV)		doVsweep(numThreads);
		else			doSweep(numThreads);
		printf("sweep time = %0.3f\n",((double)(clock()-sweep_start))/1000);
		if(recFreq && i%recFreq == 0 && fileName && i!=0) {
			printf("Saving ...");
			saveUV(fileName);
			printf("... Saved\n");
		}
		////test the policy
		//if(i%10 == 0 || i== N-1) {
		//	double x[2] = {util::PI+0.001, 0.0};
		//	double a[1];
		//	double cumCost = 0.0;
		//	int typeTmp;
		//	for(int i = 0; i < 50000; i++) {
		//		lookupPolicy(a,x);
		//		cumCost += dynamics->integrateCost(x,a,x,typeTmp, &typeTmp);
		//	}
		//	trajFile << i << " " << x[0] << " " << x[1] << " " << cumCost << std::endl;
		//	printf("TrajCost = %g   {%g, %g}\n", cumCost,x[0],x[1]);
		//}
		//if(i%500 == 0 || i== N-1) {
		//	char name[100];
		//	sprintf(name, "traj%d.txt",i);
		//	ofstream thisTraj(name);
		//	double x[2] = {util::PI+0.001, 0.0};
		//	double a[1];
		//	double cumCost = 0.0;
		//	int typeTmp;
		//	for(int i = 0; i < 50000; i++) {
		//		lookupPolicy(a,x);
		//		cumCost += dynamics->integrateCost(x,a,x,typeTmp, &typeTmp);
		//		thisTraj << x[0] << " " << x[1] << " " << a[0] << " " << cumCost << std::endl;
		//	}
		//	thisTraj.close();
		//}
	}
	printf("time = %0.3f\n",((double)(clock()-start))/1000);
	if(fileName)	saveUV(fileName);
}


int DP::compareValues(DP dp1, DP DP, double thresh) {
	int numBetter = 0;
	for(int i = 0; i < dp1.N_TOTAL_STATES; i++) {
		if(DP.V[i] < dp1.V[i]*thresh)	numBetter++;
	}

	return numBetter;
}

double DP::getCellVolume(int type) const {
	double vol=1;
	for(int i = 0; i < N_DIMS[type]; i++) {
		vol*=dx[dimIndex(i,type)];
	}
	return vol;
}

void DP::getState(int ind1d, double *state, int &type) const {
	int *indicies = new int[MAX_DIMS];
	getImd(ind1d, type, indicies);
	getState(indicies,type,state);
	delete[] indicies;
}

bool DP::inBounds(const double *state, int type) const {
#ifdef ENABLE_MAPPING
	double gridState[MAX_DIMS_EVER];
#endif
	for(int i = 0; i < N_DIMS[type]; i++) {
#ifdef ENABLE_MAPPING
		if(!inBounds(gridState[i], i, type))	return false;
#else
		if(!inBounds(state[i], i, type))	return false;
#endif
	}
	return true;
}

bool DP::inBounds(double val, int dim, int type) const {
	if(val < MIN_STATE[dimIndex(dim,type)])		return false;
	if(val > MAX_STATE[dimIndex(dim,type)])		return false;
	return true;
}

bool DP::putInBounds(double &val, int dim, int type) const {
	if(val < MIN_STATE[dimIndex(dim,type)]) {
		val = MIN_STATE[dimIndex(dim,type)];
		return false;
	}
	else if(val > MAX_STATE[dimIndex(dim,type)]) {
		val = MAX_STATE[dimIndex(dim,type)];
		return false;
	}
	else if(_isnan(val)) {
		val = MIN_STATE[dimIndex(dim,type)];
		return false;
	}
	return true;
}


bool DP::isGoodValue(const double *state, int type) const {
#ifdef MULTI_MODEL
	return lookupValue(state, activeModel, type) != INFINITE_COST;
#else
	return lookupValue(state, type) != INFINITE_COST;
#endif
}


#ifdef MULTI_MODEL
void DP::addDyn(Dynamics *dyn, int refV) {

	allocateVtmp();
	N_MODEL++;

	//update list of models
	Dynamics **newDynList = new Dynamics*[N_MODEL];

	assert(newDynList);
	for(int i = 0; i < N_MODEL-1; i++)		newDynList[i] = dynList[i];

	newDynList[N_MODEL-1] = dyn;

	delete[] dynList;
	dynList = newDynList;



	//add a new value function
	double **newVlist = new double*[N_MODEL];

	assert(newVlist);

	for(int i = 0; i < N_MODEL-1; i++)		newVlist[i] = Vlist[i];

	newVlist[N_MODEL-1] = new double[N_TOTAL_STATES];
	assert(newVlist[N_MODEL-1]);



	delete[] Vlist;
	


	Vlist = newVlist;

	double **newTmpVlist = new double*[N_MODEL];

	for(int i = 0; i < N_MODEL-1; i++)		newTmpVlist[i] = VtmpList[i];

	newTmpVlist[N_MODEL-1] = new double[N_TOTAL_STATES];

	delete[] VtmpList;

	VtmpList = newTmpVlist;

	//copy values
	for(int i = 0; i < N_TOTAL_STATES; i++)		Vlist[N_MODEL-1][i] = Vlist[refV][i];
	for(int i = 0; i < N_TOTAL_STATES; i++)		VtmpList[N_MODEL-1][i] = VtmpList[refV][i];

	//printf("%d %g\n",N_MODEL-1, dynList[N_MODEL-1]->getTimeStep());
}
#endif

void DP::getAllValuesBySim(int ts) {
	int type;
	double *state = new double[MAX_DIMS];

	for(int i = 0; i < N_TOTAL_STATES; i++) {
		getState(i,state,type);
		V[i] = getValueBySim(state,type,ts);
	}
	delete[] state;
}

void DP::freeV() {
	delete[] V;
	delete[] Vtmp;
}

void DP::freeVtmp() {
	delete[] Vtmp;
}

void DP::clearV() {
	for(int i = 0; i < N_TOTAL_STATES; i++) {
#ifdef MULTI_MODEL
		for(int j = 0; j < N_MODEL; j++) {
			Vlist[j][i] = 0;
			if(Vtmp!=NULL)	VtmpList[j][i] = 0;
		}
#else
		V[i] = 0;
		if(Vtmp!=NULL)	Vtmp[i] = 0;
#endif
	}
}

void DP::allVbad() {
	for(int i = 0; i < N_TOTAL_STATES; i++) {
#ifdef MULTI_MODEL
		for(int j = 0; j < N_MODEL; j++) {
			Vlist[j][i] = INFINITE_COST;
			if(Vtmp!=NULL)	VtmpList[j][i] = INFINITE_COST;
		}
#else
		V[i] = INFINITE_COST;
		if(Vtmp!=NULL)	Vtmp[i] = INFINITE_COST;
#endif
	}
}

void DP::multV(double a) {
	for(int i = 0; i < N_TOTAL_STATES; i++) {
#ifdef MULTI_MODEL
		for(int j = 0; j < N_MODEL; j++) {
			if(Vlist[j][i] < INFINITE_COST)									Vlist[j][i] *= a;
			if(Vtmp!=NULL)		if(VtmpList[j][i] < INFINITE_COST)			VtmpList[j][i] *= a;
		}
#else
		if(V[i] < INFINITE_COST)	V[i] *= a;
		if(Vtmp!=NULL)	if(Vtmp[i] < INFINITE_COST)		Vtmp[i] *= a;
#endif
	}
}

void DP::cleanBadPol() {
	for(int i = 0; i < N_DYN_TYPES; i++) {
		for(int j = 0; j < N_TYPE_STATES[i]; j++) {
			int ind1d = statesBeforeType[i]+j;
			if(V[ind1d] == INFINITE_COST) {
				for(int k = 0; k <N_ACTION_DIMS[i]; k++) {
					U[ind1d][k] = 0;
				}
			}
		}
	}
}

void DP::clearU() {
	for(int i = 0; i < N_DYN_TYPES; i++) {
		for(int j = 0; j < N_TYPE_STATES[i]; j++) {
			for(int k = 0; k <N_ACTION_DIMS[i]; k++) {
				U[statesBeforeType[i]+j][k] = 0;
			}
		}
	}
}

void DP::seedLegalPolicy() {
	double *state = new double[MAX_DIMS];
	double *randAct = new double[MAX_ACTION_DIMS];
	printf("%d\n",N_TOTAL_STATES);
	int type;
	for(int i = 0; i < N_TOTAL_STATES; i++) {
		getState(i,state,type);	
		randActPtr(state,type,randAct,this);
		copyArray(randAct, U[i], N_ACTION_DIMS[type]);
	}

	delete[] state;
	delete[] randAct;
}

//for changing bounds without starting from scratch
//interpolates off of the old configuration
void DP::loadUVfromConfig(const char *config, const char *pol) {

	DP dpOld(NULL,config);

	dpOld.loadUV(pol);

	int type;
	double *state = new double[MAX_DIMS];
	double *action = new double[MAX_ACTION_DIMS];
	for(int i = 0; i < N_TOTAL_STATES; i++) {
		getState(i,state,type);
		V[i] = dpOld.lookupValue(state,type);
		dpOld.lookupPolicy(action,state,type);
		copyArray(action, U[i], N_ACTION_DIMS[type]);		//not sure if copying is necessary
	}
	delete[] state;
	delete[] action;
}

void DP::setAux(void (*auxFunk)(DP *dp, double * aux, double *state, int type)) {
	double *state = new double[MAX_DIMS];
	int type;
	for(int i = 0; i < N_TOTAL_STATES; i++) {
		getState(i,state,type);
		auxFunk(this, aux[i],state,type);		
	}

	delete[] state;
}
