/************************************************************************/
/************************************************************************/
/*
Optimize a trajectory.
representation
knot[0][J0][POS]
knot[0][J0][VEL]
knot[0][J0][ACC]
knot[0][J1][POS]
knot[0][J1][VEL]
knot[0][J1][ACC]
knot[0][J2][POS]
knot[0][J2][VEL]
knot[0][J2][ACC]
knot[1][J0][POS]
knot[1][J0][VEL]
knot[1][J0][ACC]
...
 */
/************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "f2c.h"
#include "snopt.h"
#include "snfilewrapper.h"

/************************************************************************/
/************************************************************************/

// control resolution of representation
#define N_KNOTS 21
// 11 2.29316
// 16 2.27978
// 21 2.25913
// 26 2.85898
// 31 2.84977
// control resolution of evaluation
#define N_EVALS 10 // don't evaluate starting knot point

#define N_JOINTS 3
#define K_POS 0
#define K_VEL 1
#define K_ACC 2

#define GOAL M_PI
#define DURATION 2.5
#define STATE_PENALTY 0.1

/* Parameters */
#define LENGTH (1.0/3.0)	// length of link
#define WIDTH (0.1)    // width of link
#define MASS (1.0/3.0)     // mass of link
#define GRAVITY 9.81

// N_parameters
#define N_X (N_KNOTS*N_JOINTS*3) 

// N_constraints + objective
#define F_OBJECTIVE 0
#define N_F (1+6+6+6)
// objective
// initial and final position constraints
// initial and final velocity constraints
// initial and final acceleration constraints

/************************************************************************/
// basis functions

double c_p0[N_EVALS];
double c_v0[N_EVALS];
double c_a0[N_EVALS];
double c_p1[N_EVALS];
double c_v1[N_EVALS];
double c_a1[N_EVALS];

double cd_p0[N_EVALS];
double cd_v0[N_EVALS];
double cd_a0[N_EVALS];
double cd_p1[N_EVALS];
double cd_v1[N_EVALS];
double cd_a1[N_EVALS];

double cdd_p0[N_EVALS];
double cdd_v0[N_EVALS];
double cdd_a0[N_EVALS];
double cdd_p1[N_EVALS];
double cdd_v1[N_EVALS];
double cdd_a1[N_EVALS];

/************************************************************************/

void load_basis_functions()
{
  int i;
  double t1, t2, t3, t4, t5;
  double dt = 1.0/N_EVALS;
  double segment_T = DURATION/(N_KNOTS-1);
  double segment_T2;

  segment_T2 = segment_T*segment_T;

  t1 = 0;

  for ( i = 0; i < N_EVALS; i++ )
    {
      t1 += dt;
      t2 = t1*t1;
      t3 = t1*t2;
      t4 = t1*t3;
      t5 = t1*t4;

      // from http://www.rose-hulman.edu/~finn/CCLI/Notes/day09.pdf

      c_p1[i] = 10*t3 - 15*t4 + 6*t5;
      c_p0[i] = 1 - c_p1[i];
      c_v0[i] = (t1 - 6*t3 + 8*t4 - 3*t5)*segment_T;
      c_a0[i] = (t2/2 - 3*t3/2 +3*t4/2 - t5/2)*segment_T2;
      c_v1[i] = (-4*t3 + 7*t4 - 3*t5)*segment_T;
      c_a1[i] = (t3/2 - t4 + t5/2)*segment_T2;

      cd_p1[i] = (30*t2 - 60*t3 + 30*t4)/segment_T;
      cd_p0[i] = -cd_p1[i];
      cd_v0[i] = (1 - 18*t2 + 32*t3 - 15*t4);
      cd_a0[i] = (t1 - 9*t2/2 + 6*t3 - 5*t4/2)*segment_T;
      cd_v1[i] = (-12*t2 + 28*t3 - 15*t4);
      cd_a1[i] = (3*t2/2 - 4*t3 + 5*t4/2)*segment_T;

      cdd_p1[i] = (60*t1 - 180*t2 + 120*t3)/segment_T2;
      cdd_p0[i] = -cdd_p1[i];
      cdd_v0[i] = (-36*t1 + 96*t2 - 60*t3)/segment_T;
      cdd_a0[i] = (1 - 9*t1 + 18*t2 - 10*t3);
      cdd_v1[i] = (-24*t1 + 84*t2 - 60*t3)/segment_T;
      cdd_a1[i] = (3*t1 - 12*t2 + 10*t3);

      /*
      printf( "%d %g %g %g %g %g %g %g\n", i, t1,
	      c_p0[i],
	      c_v0[i],
	      c_a0[i],
	      c_p1[i],
	      c_v1[i],
	      c_a1[i] );
      */
    }
}

/************************************************************************/
/************************************************************************/

void do_snopt(); // do snopt stuff

/************************************************************************/
/************************************************************************/
  /*     ================================================================== */
  /* objective and constraints for SNOPT.                                   */
  /*     On exit:                                                           */
  /*     inform      is 0 if there is enough storage, 1 otherwise.          */
  /*     neF         is the number of problem functions                     */
  /*                 (objective and constraints, linear and nonlinear).     */
  /*     n           is the number of variables.                            */
  /*     xlow        holds the lower bounds on x.                           */
  /*     xupp        holds the upper bounds on x.                           */
  /*     Flow        holds the lower bounds on F.                           */
  /*     Fupp        holds the upper bounds on F.                           */
  /*     xstate(1:n) is a set of initial states for each x  (0,1,2,3,4,5).  */
  /*     x (1:n)     is a set of initial values for x.                      */
  /*     Fmul(1:neF) is a set of initial values for the dual variables.     */
  /*                                                                        */
  /*     ================================================================== */

void usrfun_init
( integer *inform, char *Prob, integer *neF, integer *n, doublereal *ObjAdd,
  integer *ObjRow, doublereal *xlow, doublereal *xupp,
  doublereal *Flow, doublereal *Fupp, doublereal *x,
  integer *xstate, doublereal *Fmul )
{
  int i, j, index;
  int pose;

  load_basis_functions();

  // I think there is enough storage????
  *inform = 0;

  /* Give the problem a name.  */
  sprintf( Prob, "%s", "usrfun" );

  /* Assign the dimensions of the constraint Jacobian */
  *n      = N_X;
  *neF    = N_F;

  *ObjRow = 1; /* NOTE: Me must add one to mesh with fortran */
  *ObjAdd = 0;

  printf( "n_x: %d; n_f: %d; n_x: %d; n_F: %d\n",
	  N_X, N_F, (int) (*n), (int) (*neF) );

  for ( i = 0; i < N_X; i++ )
    {
      xstate[i] = 0;
      x[i] = 0;
    }

  /* Set upper and lower bounds for parameters. */
  for ( i = 0; i < N_KNOTS; i++ )
    {
      for ( j = 0; j < N_JOINTS; j++ )
	{
	  index = i*N_JOINTS*3 + j*3;
	  xlow[index + K_POS] = -10.0;
	  xupp[index + K_POS] = 10.0;
	  xlow[index + K_VEL] = -20.0;
	  xupp[index + K_VEL] = 20.0;
	  xlow[index + K_ACC] = -100.0;
	  xupp[index + K_ACC] = 100.0;
	}
    }

  // initial guess on positions
  for ( i = 0; i < N_KNOTS; i++ )
    {
      index = i*N_JOINTS*3; 
      x[index + K_POS] = GOAL*((double) i)/(N_KNOTS-1);
      x[index + K_VEL] = GOAL/DURATION;
    }
  x[0 + K_VEL] = 0;
  index = (N_KNOTS-1)*N_JOINTS*3; 
  x[index + K_VEL] = 0;

  /* Set upper and lower bounds for constraints */
  // default
  for ( i = 0; i < N_F; i++ )
    {
      Flow[i] = 0;
      Fupp[i] = 0;
    }

  // F[0] OBJECTIVE
  Flow[ F_OBJECTIVE ] = 0.0;
  Fupp[ F_OBJECTIVE ] = 1e10;
  // joint1 final position
  Flow[4] = GOAL; 
  Fupp[4] = GOAL;

  for ( i = 0; i < N_X; i++ )
    {
      printf( "init X: %d: %g <= %g <= %g; %d\n", i,
	      xlow[i], x[i], xupp[i], (int) (xstate[i]) );
    }

  for ( i = 0; i < N_F; i++ )
    {
      printf( "init F: %d: %g %g\n", i, Flow[i], Fupp[i] );
    }

  printf( "knots:\n" );
  for ( i = 0; i < N_KNOTS; i ++ )
    {
      index = i*N_JOINTS*3;
      printf( "%d %d: %g %g %g\n",
	      i, 0, x[index], x[index+1], x[index+2] );
      printf( "%d %d: %g %g %g\n",
	      i, 1, x[index+3], x[index+4], x[index+5] );
      printf( "%d %d: %g %g %g\n",
	      i, 2, x[index+6], x[index+7], x[index+8] );
    } 
}

/************************************************************************/
/************************************************************************/

void inverse_dynamics( double *pos, double *vel, double *acc,
		       double *trq )
{
  /* Slightly faster to have these as variables than defines. Go figure */
#ifdef COMMENT
  // values for robot
  double m1 = 11.0;
  double m2 = 18.5;
  double m3 = 111.1;
  double l1cm = 0.235;
  double l2cm = 0.1640;
  double l3cm = 0.3929;
  double l1 = 0.422;
  double l2 = 0.374;
  double I1 = 0.152; /* Icom */
  double I2 = 0.18; /* Icom */
  double I3 = 7.67; /* Icom */
#endif

  double m1 = MASS;
  double m2 = MASS;
  double m3 = MASS;
  double l1cm = (LENGTH/2);
  double l2cm = (LENGTH/2);
  double l3cm = (LENGTH/2);
  double l1 = LENGTH;
  double l2 = LENGTH;
  double I1 = (MASS*(LENGTH*LENGTH + WIDTH*WIDTH)/12); /* Icom */
  double I2 = (MASS*(LENGTH*LENGTH + WIDTH*WIDTH)/12); /* Icom */
  double I3 = (MASS*(LENGTH*LENGTH + WIDTH*WIDTH)/12); /* Icom */
  double s1, c1, s2, c2, s3, c3;
  double a, b, c, d, e, f, g, h, ii, j, k, l;
  double determinant;
  double l3cm_m3;
  double s23, c23;
  double a1d_a1d;
  double a1d_a2d;
  double a1d_a3d;
  double a2d_a2d;
  double a2d_a3d;
  double a3d_a3d;
  double expr1, expr2, expr3, expr4, expr5;
  double l1_l2cm_m2;
  double l1_l2_m3;
  double l2_l3cm_m3;
  double l2_l3cm_m3_s3;
  double l2_l3cm_m3_c3;
  double l1_l3cm_m3;
  double l1_l3cm_m3_s23;
  double l1_l3cm_m3_c23;
  double l1cm_m1;
  double l2cm_m2;
  double l2_m3;
  double a1d, a2d, a3d;

  s1 = sin( pos[0] );
  c1 = cos( pos[0] );
  s2 = sin( pos[1] );
  c2 = cos( pos[1] );
  s3 = sin( pos[2] );
  c3 = cos( pos[2] );
  s23 = s2*c3 + c2*s3;
  c23 = c2*c3 - s2*s3;

  a1d = vel[0];
  a2d = vel[1];
  a3d = vel[2];

  l1cm_m1 = l1cm*m1;
  l2cm_m2 = l2cm*m2;
  l3cm_m3 = l3cm*m3;
  l2_m3 = l2*m3;
  l1_l2_m3 = l1*l2_m3;
  l1_l2cm_m2 = l1*l2cm_m2;
  l1_l3cm_m3 = l1*l3cm_m3;
  l1_l3cm_m3_s23 = l1_l3cm_m3*s23;
  l1_l3cm_m3_c23 = l1_l3cm_m3*c23;
  l2_l3cm_m3 = l2*l3cm_m3;
  l2_l3cm_m3_s3 = l2_l3cm_m3*s3;
  l2_l3cm_m3_c3 = l2_l3cm_m3*c3;
  a1d_a1d = a1d*a1d;
  a1d_a2d = a1d*a2d;
  a1d_a3d = a1d*a3d;
  a2d_a2d = a2d*a2d;
  a2d_a3d = a2d*a3d;
  a3d_a3d = a3d*a3d;
  expr1 = (l1_l2cm_m2 + l1_l2_m3)*s2 + l1_l3cm_m3_s23;
  expr2 = l2_l3cm_m3_s3 + l1_l3cm_m3_s23;
  expr3 = GRAVITY*l3cm_m3*(c1*s23 + s1*c23); 
  expr4 = GRAVITY*(c2*s1 + c1*s2)*(l2cm_m2 + l2_m3);
  expr5 = (l1*m2 + l1*m3);

  /* Third row */
  ii = I3 + l3cm*l3cm_m3;
  h = ii + l2_l3cm_m3_c3;
  g = h + l1_l3cm_m3_c23;

  l = expr3 + a1d_a1d*l1_l3cm_m3_s23 
    + (a1d_a1d + 2*a1d_a2d + a2d_a2d)*l2_l3cm_m3_s3;

  /* Second row */
  f =  ii + l2_l3cm_m3_c3;
  e = f + h - ii + I2 + l2cm*l2cm_m2 + l2*l2_m3;
  d = e + g - h + (l1_l2cm_m2 + l1_l2_m3)*c2;

  k = expr4 + expr3 + a1d_a1d*expr1
    - (2*a1d_a3d + 2*a2d_a3d + a3d_a3d)*l2_l3cm_m3_s3;

  /* First row */
  c = f + l1_l3cm_m3_c23;
  b = c + e - f + (l1_l2cm_m2 + l1_l2_m3)*c2;
  a = b + d - e + I1 + l1cm*l1cm_m1 + l1*expr5;
    
  j = GRAVITY*s1*(l1cm_m1 + expr5) + expr4 + expr3
       - (2*a1d_a2d + a2d_a2d)*expr1
       - (2*a1d_a3d + 2*a2d_a3d + a3d_a3d)*expr2;

  trq[0] = a*acc[0] + b*acc[1] + c*acc[2] + j;
  trq[1] = d*acc[0] + e*acc[1] + f*acc[2] + k;
  trq[2] = g*acc[0] + h*acc[1] + ii*acc[2] + l;
}

/************************************************************************/

void stash_traj( char *file_name, double *x )
{
  int i, j, index0, index1;
  int segment;
  double score = 0;
  double pos[3];
  double vel[3];
  double acc[3];
  double trq[3];
  double dt = DURATION/((N_KNOTS-1)*N_EVALS);
  double ttime = 0;
  FILE *stream;

  stream = fopen( file_name, "w" );
  if ( stream == NULL )
    {
      fprintf( stderr, "Can't open %s for write.\n", file_name );
      exit( -1 );
    }

  // get first knot
  for ( j = 0; j < N_JOINTS; j++ )
    {
      pos[j] = x[j*3 + K_POS];
      vel[j] = x[j*3 + K_VEL];
      acc[j] = x[j*3 + K_ACC];
    }
  inverse_dynamics( pos, vel, acc, trq );

  fprintf( stream, "%g %g %g %g ", ttime, pos[0], pos[1], pos[2] );
  fprintf( stream, "%g %g %g ", vel[0], vel[1], vel[2] );
  fprintf( stream, "%g %g %g ", acc[0], acc[1], acc[2] );
  fprintf( stream, "%g %g %g\n", trq[0], trq[1], trq[2] );
  ttime += dt;

  // get rest of trajectory
  for ( segment = 1; segment < N_KNOTS; segment++ )
    {
      index0 = (segment-1)*N_JOINTS*3;
      index1 = segment*N_JOINTS*3;

      for ( i = 0; i < N_EVALS; i++ )
	{
	  for ( j = 0; j < N_JOINTS; j++ )
	    {
	      pos[j] = c_p0[i]*x[index0 + j*3 + K_POS]
		+ c_v0[i]*x[index0 + j*3 + K_VEL]
		+ c_a0[i]*x[index0 + j*3 + K_ACC]
		+ c_p1[i]*x[index1 + j*3 + K_POS]
		+ c_v1[i]*x[index1 + j*3 + K_VEL]
		+ c_a1[i]*x[index1 + j*3 + K_ACC];
	      vel[j] = cd_p0[i]*x[index0 + j*3 + K_POS]
		+ cd_v0[i]*x[index0 + j*3 + K_VEL]
		+ cd_a0[i]*x[index0 + j*3 + K_ACC]
		+ cd_p1[i]*x[index1 + j*3 + K_POS]
		+ cd_v1[i]*x[index1 + j*3 + K_VEL]
		+ cd_a1[i]*x[index1 + j*3 + K_ACC];
	      acc[j] = cdd_p0[i]*x[index0 + j*3 + K_POS]
		+ cdd_v0[i]*x[index0 + j*3 + K_VEL]
		+ cdd_a0[i]*x[index0 + j*3 + K_ACC]
		+ cdd_p1[i]*x[index1 + j*3 + K_POS]
		+ cdd_v1[i]*x[index1 + j*3 + K_VEL]
		+ cdd_a1[i]*x[index1 + j*3 + K_ACC];
	    }

	  inverse_dynamics( pos, vel, acc, trq );

	  fprintf( stream, "%g %g %g %g ", ttime, pos[0], pos[1], pos[2] );
	  fprintf( stream, "%g %g %g ", vel[0], vel[1], vel[2] );
	  fprintf( stream, "%g %g %g ", acc[0], acc[1], acc[2] );
	  fprintf( stream, "%g %g %g\n", trq[0], trq[1], trq[2] );
	  ttime += dt;
	}
    }

  fclose( stream );
}

/************************************************************************/

double evaluate_segment( int segment, double *x )
{
  int i, j, index0, index1;
  double score = 0;
  double pos[3];
  double vel[3];
  double acc[3];
  double trq[3];

  index0 = (segment-1)*N_JOINTS*3;
  index1 = segment*N_JOINTS*3;

  for ( i = 0; i < N_EVALS; i++ )
    {
      for ( j = 0; j < N_JOINTS; j++ )
	{
	  pos[j] = c_p0[i]*x[index0 + j*3 + K_POS]
	    + c_v0[i]*x[index0 + j*3 + K_VEL]
	    + c_a0[i]*x[index0 + j*3 + K_ACC]
	    + c_p1[i]*x[index1 + j*3 + K_POS]
	    + c_v1[i]*x[index1 + j*3 + K_VEL]
	    + c_a1[i]*x[index1 + j*3 + K_ACC];
	  vel[j] = cd_p0[i]*x[index0 + j*3 + K_POS]
	    + cd_v0[i]*x[index0 + j*3 + K_VEL]
	    + cd_a0[i]*x[index0 + j*3 + K_ACC]
	    + cd_p1[i]*x[index1 + j*3 + K_POS]
	    + cd_v1[i]*x[index1 + j*3 + K_VEL]
	    + cd_a1[i]*x[index1 + j*3 + K_ACC];
	  acc[j] = cdd_p0[i]*x[index0 + j*3 + K_POS]
	    + cdd_v0[i]*x[index0 + j*3 + K_VEL]
	    + cdd_a0[i]*x[index0 + j*3 + K_ACC]
	    + cdd_p1[i]*x[index1 + j*3 + K_POS]
	    + cdd_v1[i]*x[index1 + j*3 + K_VEL]
	    + cdd_a1[i]*x[index1 + j*3 + K_ACC];
	}

      inverse_dynamics( pos, vel, acc, trq );

      j = 0; // joint 1
      score += STATE_PENALTY*(pos[j] - GOAL)*(pos[j] - GOAL);
      j = 1; // joint 2
      score += STATE_PENALTY*(pos[j])*(pos[j]);
      j = 2; // joint 3
      score += STATE_PENALTY*(pos[j])*(pos[j]);

      for ( j = 0; j < 3; j++ )
	{
	  score += trq[j]*trq[j];
	}
    }

  return score;
}

/************************************************************************/

/*     ================================================================== */
/*     Computes the nonlinear objective and constraint terms. */
/*     ================================================================== */
int usrfunf_
( integer    *Status, integer *n,    doublereal x[],
  integer    *needF,  integer *neF,  doublereal F[],
  integer    *needG,  integer *neG,  doublereal G[],
  char       *cu,     integer *lencu,
  integer    iu[],    integer *leniu,
  doublereal ru[],    integer *lenru )
{
  int i, j, index;
  double dt = DURATION/((N_KNOTS-1)*N_EVALS);
  double score = 0;

  F[0] = 0; // objective
  F[1] = x[0 + K_POS]; // initial joint1
  F[2] = x[3 + K_POS]; // initial joint2
  F[3] = x[6 + K_POS]; // initial joint3

  index = (N_KNOTS-1)*N_JOINTS*3; 
  F[4] = x[index + K_POS]; // final joints
  F[5] = x[index + 3 + K_POS]; 
  F[6] = x[index + 6 + K_POS]; 

  F[7] = x[0 + K_VEL]; // initial joint velocities
  F[8] = x[3 + K_VEL];
  F[9] = x[6 + K_VEL];

  index = (N_KNOTS-1)*N_JOINTS*3; 
  F[10] = x[index + K_VEL]; // final joint velocities
  F[11] = x[index + 3 + K_VEL];
  F[12] = x[index + 6 + K_VEL];

  F[13] = x[0 + K_ACC]; // initial joint accelerations
  F[14] = x[3 + K_ACC];
  F[15] = x[6 + K_ACC];

  index = (N_KNOTS-1)*N_JOINTS*3; 
  F[16] = x[index + K_ACC]; // final joint accelerations
  F[17] = x[index + 3 + K_ACC];
  F[18] = x[index + 6 + K_ACC];

  for ( i = 1; i < N_KNOTS; i++ )
    score += evaluate_segment( i, x );

  F[0] += score*dt;

  if ( *Status == 1 )
    printf( "First call.\n" );
  else if ( *Status > 1 )
    {
      if ( *Status == 2 )
	printf( "Successful. Optimal X:\n" );
      if ( *Status == 3 )
	printf( "Infeasible. X:\n" );
      if ( *Status == 4 )
	printf( "Unbounded. X:\n" );
      if ( *Status == 5 )
	printf( "Iteration limit reached. X:\n" );
      if ( *Status > 5 )
	printf( "Unknown status %d. X:\n", (int) (*Status) );
      printf( "usrfunf x:\n" );
      for ( i = 0; i < N_KNOTS; i ++ )
	{
	  index = i*N_JOINTS*3;
	  printf( "%d %d: %g %g %g\n",
		  i, 0, x[index], x[index+1], x[index+2] );
	  printf( "%d %d: %g %g %g\n",
		  i, 1, x[index+3], x[index+4], x[index+5] );
	  printf( "%d %d: %g %g %g\n",
		  i, 2, x[index+6], x[index+7], x[index+8] );
	} 
      printf( "usrfunf F:\n" );
      for ( i = 0; i < N_F; i++ )
	printf( "%d %g\n", i, F[i] );
      stash_traj( "traj", x );
    }

  /*
  printf( "usrfunf x:\n" );
  for ( i = 0; i < n_xx; i++ )
    printf( "%d %g\n", i, x[i] );
  */

  /*
  printf( "x[N]: %g\n", x[N] );
  printf( "usrfunf F:\n" );
  for ( i = 0; i < N_F; i++ )
    printf( "%d %g\n", i, F[i] );
  */

  return 0;
}

/************************************************************************/
/************************************************************************/
/************************************************************************/

int main()
{
  int i;

  do_snopt();

  return 0;
}

/************************************************************************/
