#include <cmath>
#include <iostream>
#include "ekffilt2.h"

#ifdef MATLAB
#include "mex.h"
#endif

typedef LibmEKFilter2::Vector Vector;
typedef LibmEKFilter2::Matrix Matrix;

using namespace std;

#define N_X	18
#define N_Y	11
#define N_U	2


LibmEKFilter2::LibmEKFilter2() 
{
    // nx, nu, nw, nm, nv
    setDim(N_X, N_U, N_X, N_Y, N_Y);
    Period = 0.0025;		
    Mass = 95;
    Gravity = 9.81;
    Height = 0.9;
}

void LibmEKFilter2::makeProcess()
{
    Vector x_(N_X);
    
    x_(1) = x(1) + x(2)*Period;
    x_(2) = x(2) + Gravity/Height*Period*((x(1)-x(3)-x(4))*x(13) + (x(1)-x(5)-x(6))*(1-x(13))) + 1/Mass*Period*x(14);
    x_(3) = x(3);
    x_(4) = x(4);
    x_(5) = x(5);
    x_(6) = x(6);
    
    x_(7) = x(7) + x(8)*Period;
    x_(8) = x(8) + Gravity/Height*Period*((x(7)-x(9)-x(10))*x(13) + (x(7)-x(11)-x(12))*(1-x(13))) + 1/Mass*Period*x(15);
    x_(9) = x(9);
    x_(10) = x(10);
    x_(11) = x(11);
    x_(12) = x(12);
    
    x_(13) = x(13);
    
    x_(14) = x(14);    
    x_(15) = x(15);
    
    x_(16) = x(16) - 2*Period*(x(16)-Height);
    x_(17) = x(17) - 2*Period*x(17);
    x_(18) = x(18) - 2*Period*x(18);
    x.swap(x_);
}


void LibmEKFilter2::makeMeasure()
{
    z(1)=x(3)-x(1);
    z(2)=x(4);    
    z(3)=x(5)-x(1);
    z(4)=x(6);
    z(5)=x(9)-x(7);
    z(6)=x(10);
    z(7)=x(11)-x(7);
    z(8)=x(12);
    z(9)=x(13);
    z(10)=x(17)-x(16);
    z(11)=x(18)-x(16);
   
}

void LibmEKFilter2::makeA(){
    
    A(2,1) = Gravity/Height*Period;
    
    A(2,3) = -Gravity/Height*Period*x(13);
    A(2,4) = -Gravity/Height*Period*x(13);
    
    A(2,5) = -Gravity/Height*Period*(1-x(13));
    A(2,6) = -Gravity/Height*Period*(1-x(13));
        
    A(2,13) = Gravity/Height*Period*((x(1)-x(3)-x(4)) - (x(1)-x(5)-x(6)));
    
    A(8,7) = Gravity/Height*Period;
    
    A(8,9) = -Gravity/Height*Period*x(13);
    A(8,10) = -Gravity/Height*Period*x(13);
    
    A(8,11) = -Gravity/Height*Period*(1-x(13));
    A(8,12) = -Gravity/Height*Period*(1-x(13));
        
    A(8,13) = Gravity/Height*Period*((x(7)-x(9)-x(10)) - (x(7)-x(11)-x(12)));    
    
}

void LibmEKFilter2::makeH(){
    // linear
}

void LibmEKFilter2::makeBaseA()
{
    int i,j;
    
    for(i=1;i<=N_X;i++){
        for(j=1;j<=N_X;j++) A(i,j) = 0.0;
        A(i,i) = 1.0;
    }

    A(1,2) = Period;
    A(2,14) = 1/Mass*Period;    

    A(7,8) = Period;
    A(8,15) = 1/Mass*Period;
    
    A(17,17) = 1.0 - 2.0*Period;
    A(18,18) = 1.0 - 2.0*Period;    
    
      
}

void LibmEKFilter2::makeBaseW()
{
    int i,j;
    for(i=1;i<=N_X;i++){
        for(j=1;j<=N_X;j++){
            W(i,j) = 0.0;
        }
        W(i,i) = 1.0;
    }
}

static double q0 = 1e-7;
static double q1 = 10*q0;
static double qf = 1e-10;
static double qc = 10*q0;
static double qw = 100*q0;
static double qe = 1e-0;
static double qz = 1e-9;
         
static double Qv[N_X] = {q0,q1,qf,qc,qf,qc,q0,q1,qf,qc,qf,qc,qw,qe,qe,qz,qf,qf};


void LibmEKFilter2::makeBaseQ()
{
    int i,j;
    for(i=1;i<=N_X;i++){
        for(j=1;j<=N_X;j++) Q(i,j) = 0.0;    
        Q(i,i) = Qv[i-1];
    }
}
 
void LibmEKFilter2::makeQ()
{
	double CUTOFF = 0.5;

    if (x(13)<CUTOFF){  // right foot support
        Q(3,3) = Qv[2] + 1.0e-6*(CUTOFF-x(13))/CUTOFF;
        Q(4,4) = Qv[3] + 1.0e-6*(CUTOFF-x(13))/CUTOFF;
        Q(9,9) = Qv[8] + 1.0e-6*(CUTOFF-x(13))/CUTOFF;
        Q(10,10) = Qv[9] + 1.0e-6*(CUTOFF-x(13))/CUTOFF;
        Q(17,17) = Qv[16] + 1.0e-7*(CUTOFF-x(13))/CUTOFF;
	}else{
        Q(3,3) = Qv[2];
        Q(4,4) = Qv[3];
        Q(9,9) = Qv[8];
        Q(10,10) = Qv[9];
        Q(17,17) = Qv[16];
	}
    
    if (x(13)>=CUTOFF){  // left foot support
        Q(5,5) = Qv[4] + 1.0e-6*(x(13)-CUTOFF)/CUTOFF;
        Q(6,6) = Qv[5] + 1.0e-6*(x(13)-CUTOFF)/CUTOFF;
        Q(11,11) = Qv[10] + 1.0e-6*(x(13)-CUTOFF)/CUTOFF;
        Q(12,12) = Qv[11] + 1.0e-6*(x(13)-CUTOFF)/CUTOFF;
        Q(18,18) = Qv[17] + 1.0e-7*(x(13)-CUTOFF)/CUTOFF;
    }else{
        Q(5,5) = Qv[4] ;
        Q(6,6) = Qv[5] ;
        Q(11,11) = Qv[10] ;
        Q(12,12) = Qv[11] ;
        Q(18,18) = Qv[17] ;
    }
    
}

void LibmEKFilter2::makeBaseH()
{
    int i,j;
                
    for(i=1;i<=N_Y;i++)
        for(j=1;j<=N_X;j++) H(i,j) = 0.0;

    H(1,1) = -1.0;
    H(1,3) =  1.0;
    
    H(2,4) = 1.0;
    
    H(3,1) = -1.0;
    H(3,5) =  1.0;
    
    H(4,6) =  1.0;
    
    H(5,7) = -1.0;
    H(5,9) =  1.0;
    
    H(6,10) = 1.0;
    
    H(7,7) = -1.0;
    H(7,11) = 1.0;
    
    H(8,12) = 1.0;
    
    H(9,13) = 1.0;   
        
    H(10,17) = 1.0;
    H(10,16) = -1.0;
    
    H(11,18) = 1.0;
    H(11,16) = -1.0;    
}


void LibmEKFilter2::makeBaseV()
{
    int i,j;
    for(i=1;i<=N_Y;i++){
        for(j=1;j<=N_Y;j++){
            V(i,j) = 0.0;
        }
        V(i,i) = 1.0;
    }
}


double rf = 1e-4;
double rc = 1e-4;
double rz = 1e-2;

double Rv[N_Y] = {rf,rc,rf,rc,rf,rc,rf,rc,rz,rz,rz}; 

void LibmEKFilter2::makeBaseR()
{
    int i,j;
    for(i=1;i<=N_Y;i++){
        for(j=1;j<=N_Y;j++) R(i,j) = 0.0;    
        R(i,i) = Rv[i-1];
    }   
}

void LibmEKFilter2_init(LibmEKFilter2 *p, double *x0, double *P0){
 	Vector x0v(N_X);
     Matrix P0m(N_X,N_X);
 	int i,j;
        
 	for(i=0;i<N_X;i++){
 		x0v(i+1) = x0[i];
 		for(j=0;j<N_X;j++)	P0m(1+i,1+j) = P0[N_X*i + j];
 	}
 
 	p->init(x0v, P0m);
 }
 
 void LibmEKFilter2_step(LibmEKFilter2 *p, double *input, double *meas, double *est){
 	Vector u(N_U),m(N_Y),xv(N_X);
 	int i,j;
 
 	for(i=0;i<N_U;i++) u(1+i) = input[i];
     for(i=0;i<N_Y;i++) m(1+i) = meas[i];
        
 	p->step(u,m);
 
 	xv = p->getX();
 
 	for(i=0;i<N_X;i++)	est[i] = xv(1+i);
 }