#ifndef MPC_MODEL_H
#define MPC_MODEL_H

#include "config_file.h"
#include <iostream>

#include <Eigen/Core>
using namespace Eigen;

class mpcInput{
public:
	mpcInput(){};
	mpcInput(double _t0, VectorXd &_x0){t0 = _t0; x0.resizeLike(_x0); x0 = _x0;};
	double t0;
	VectorXd x0;
	mpcInput operator= (mpcInput A){  t0 = A.t0;  x0.resizeLike(A.x0);  x0 = A.x0;  return *this; };
};

class mpcOutput{
public:
	mpcOutput(){};
	VectorXd T;
	VectorXd X;
	VectorXd Y;
	VectorXd U;
	double cost;
	mpcOutput operator= (mpcOutput A){
		T.resizeLike(A.T);
		T = A.T;
		X.resizeLike(A.X);
		X = A.X;
		Y.resizeLike(A.Y);
		Y = A.Y;
		U.resizeLike(A.U);
		U = A.U;
		cost = A.cost;
		return *this; 
	};
};

class mpcModel{

public:

	mpcModel(){init();}

	void init();

	void mpc_allocate();

	//======== Virtual Functions
	
	virtual MatrixXd forward_dynamics_x(int i, double t0, VectorXd &x0, double DT)=0;	
	virtual MatrixXd forward_dynamics_u(int i, double t0, VectorXd &x0, double DT)=0;	
	virtual MatrixXd forward_dynamics_ud(int i, double t0, VectorXd &x0, double DT)=0;	
	
	virtual MatrixXd output_dynamics_x(int i, double t0, VectorXd &x0, double DT)=0;	
	virtual MatrixXd output_dynamics_u(int i, double t0, VectorXd &x0, double DT)=0;	

	//virtual void initial_trajectory(double t0, VectorXd &x0)=0;

	// compute the A B matrices
	virtual void precompute_dynamics(double t0, VectorXd &x0)=0;
	void precompute_dynamics();

	// precompute H, f, f0, CI, ci0, CE, ce0
	virtual void precompute_cost(double t0, VectorXd &x0)=0;
	void precompute_cost(void);
	virtual void precompute_constraints(double t0, VectorXd &x0)=0;
	void precompute_constraints(void);

	// update H, f, f0, CI, ci0, CE, ce0
	virtual void update_cost(double t0, VectorXd &x0)=0;
	void update_cost(void);
	virtual void update_constraints(double t0, VectorXd &x0)=0;
	void update_constraints(void);

	virtual VectorXd simulate(double t0, VectorXd &x0, VectorXd &U)=0;
	VectorXd simulate();

	virtual void set_params(ConfigurationData *cd)=0;
	
	//virtual double one_step_cost(double t0, VectorXd &x0, double t, VectorXd &x, VectorXd &u)=0;
	double total_cost(double t0, VectorXd &x0, VectorXd &U);
	double total_cost(void);

	double output_cost(double t0, VectorXd &x0, VectorXd &U);
	double output_cost(void);

	//======== End: Virtual Functions
		
	VectorXd output_trajectory(double t0, VectorXd &x0, VectorXd &U);
	VectorXd output_trajectory(void);

	VectorXd select_action(int i){return U.segment((i)*num_actions_continuous,num_actions_continuous);};
	double select_action(int i, int j){return U.segment((i)*num_actions_continuous,num_actions_continuous)(j);};

	VectorXd select_daction(int i){return U.segment(N_TIMESTEPS*num_actions_continuous+i,1);};

	VectorXd select_state(int i){return X.segment((i)*num_states,num_states);};
	double select_state(int i, int j){return X.segment((i)*num_states,num_states)(j);};

	VectorXd select_output(int i){return Y.segment((i)*num_outputs,num_states);};
	double select_output(int i, int j){return Y.segment((i)*num_outputs,num_states)(j);};

	int num_vars(){return N_TIMESTEPS*num_actions_continuous + num_actions_discrete;};

	void set_mpc_input(mpcInput *in){assert(is_params_set); assert(in->x0.size()==num_states); input = *in; is_input_set=true;};
	void set_mpc_input(double t0, VectorXd &x0){assert(x0.size()==num_states); mpcInput in(t0,x0);  set_mpc_input(&in); };

	mpcOutput get_output();

	double start_time(){ assert(is_input_set); return input.t0; };
	VectorXd start_state(){ assert(is_input_set); return input.x0; };
		
	//===== State info

	int N_TIMESTEPS;

	int num_states;

	int num_states_continuous;
	int num_actions_continuous;

	int num_states_discrete;
	int num_actions_discrete;

	int num_inequality_constraints;
	int num_equality_constraints;

	int num_outputs;
	
	//===== Flags

	bool is_params_set;
	bool precomputed_dynamics;
	bool precomputed_cost;
	bool precomputed_constraints;
	bool is_initialized;
	bool is_nonlinear;
	bool is_constant_hessian;
	bool is_unconstrained;
	

	//===== Trajectory Storage
		
	VectorXd X;	// state trajectory
	VectorXd U;	// action trajectory
	VectorXd Y;	// output trajectory

	MatrixXd A;	// open loop dynamics A = [I;Fx;Fx^2;Fx^3;...];
	MatrixXd B;	// feedback dynamics B = [0,0,0,...;Fu,0,0,...;Fx*Fu,Fu,0,...;Fx*Fx*Fu,Fx*Fu,Fu,...];
	MatrixXd C; // output mapping C = [Hx*I;Hx*Fx;Hx*Fx^2;...];
	MatrixXd D; // output mapping D = [0;Hu*B1;Hu*B2;...];
	VectorXd T; // Timestep trajectory [t0,t1,t2,...,tN]

	// QP matrices
	MatrixXd H;	// quadratic cost
	VectorXd f;	// linear cost

	// precomputed portions of the costs
	MatrixXd PreLuu;
	MatrixXd PreLux;
	MatrixXd PreLxx;
	VectorXd PreLu;
	VectorXd PreLx;	
	double PreL0;

	// online cost matrices
	MatrixXd Luu;
	MatrixXd Lux;
	MatrixXd Lxx;
	VectorXd Lu;
	VectorXd Lx;	
	double L0;
	
	MatrixXd CI;	// linear inequalities
	VectorXd ci0;   // linear inequalities (precomputed)
	VectorXd ci1;	// linear inequalities (updated)

	MatrixXd CE;	// linear equalities
	VectorXd ce0;	// linear equalities

private:
	mpcOutput output;
	mpcInput input;
	bool is_input_set;

};

#endif