#ifdef MATLAB
#include "mex.h"
#endif


#include "lipmstep2d_FEL_model.h"
#include "config_file.h"
#include <iostream>
#include <Eigen/Core>
using namespace Eigen;


lipmStep2dModel_FEL::lipmStep2dModel_FEL(){	
	//ConfigurationData *cd = load_conf_data("lipmstep2d_model.cf");
	//*this = lipmStep2dModel_FEL(cd);
	//free_conf_data(cd);
}

lipmStep2dModel_FEL::lipmStep2dModel_FEL(ConfigurationData *cd){
	set_params(cd);
}

void lipmStep2dModel_FEL::set_params(ConfigurationData *cd){
	N_TIMESTEPS = get_value_with_default(cd,"N_TIMESTEP",15);
	TIMESTEP = get_value_with_default(cd,"TIMESTEP",0.1);	
	HEIGHT = get_value_with_default(cd,"HEIGHT",0.9);	
	N_STEPS = get_value_with_default(cd,"N_STEPS",1);	
	N_SHIFT = get_value_with_default(cd,"N_SHIFT",2);
	N_SWING = get_value_with_default(cd,"N_SWING",4);
	capture_cost = get_value_with_default(cd,"CAP_COST",0);
	global_pos_cost = get_value_with_default(cd,"GPOS_COST",0);
	pos_cost = get_value_with_default(cd,"POS_COST",0.1);
	vel_cost = get_value_with_default(cd,"VEL_COST",0.01);
	zmp_cost = get_value_with_default(cd,"ZMP_COST",0.01);
	zmpd_cost = get_value_with_default(cd,"ZMPD_COST",0.01);
	step_cost = get_value_with_default(cd,"STEP_COST",100);
	STANCE_WIDTH = get_value_with_default(cd,"STANCE_WIDTH",0.2);
	MAX_STANCE_WIDTH = get_value_with_default(cd,"MAX_STANCE_WIDTH",0.4);
	DES_STEP_LENGTH = get_value_with_default(cd,"DES_STEP_LENGTH",0.0);
	final_cost = get_value_with_default(cd,"FINAL_COST",10.0);
	FOOT_SIZE_X = get_value_with_default(cd,"FOOT_SIZE_X",0.1);
	FOOT_SIZE_Y = get_value_with_default(cd,"FOOT_SIZE_Y",0.05);

	FIRST_FOOT = get_value_with_default(cd,"FIRST_FOOT",0);
	USE_DOUBLE_SUPPORT_CONSTRAINTS = get_value_with_default(cd,"USE_DOUBLE_SUPPORT_CONSTRAINTS",0);

	T_SHIFT = N_SHIFT*TIMESTEP;
	T_SWING = N_SWING*TIMESTEP;

	printf("%i %i\n",N_SHIFT,N_SWING);

	if(N_STEPS*(N_SHIFT+N_SWING)>N_TIMESTEPS){
		printf("Warning: horizon too short, increasing N_TIMESTEPS\n");
		N_TIMESTEPS = (int)(N_STEPS*(N_SHIFT+N_SWING));
	}

	USE_FEEDBACK_ERROR_TRAJECTORY = get_value_with_default(cd,"USE_FEEDBACK_ERROR_TRAJECTORY",0);

	num_states = 10;

	if (USE_FEEDBACK_ERROR_TRAJECTORY){
		printf("***USING FEEDBACK ERROR TRAJECTORY***\n");
		num_states += 2*N_TIMESTEPS;
	}

	num_states_continuous = 6;	// position, velocity of COM
	num_actions_continuous = 2;	// zmp

	num_states_discrete = 4;	// positions of feet
	num_actions_discrete = 2*N_STEPS;	// footstep locations

	num_equality_constraints = 0;

	num_inequality_constraints = 2*N_TIMESTEPS*num_actions_continuous + 4*N_STEPS;
	
	if(USE_DOUBLE_SUPPORT_CONSTRAINTS){
		if(N_STEPS>0){
			num_inequality_constraints += 2*(N_SHIFT+1);
		}else{
			num_inequality_constraints += 2*N_TIMESTEPS;
		}
	}

	num_outputs = 12;

	is_constant_hessian = true;
	is_nonlinear = false;

	is_unconstrained = false;

	step_indices.setZero(N_TIMESTEPS+1);
	contact_state.setZero(N_TIMESTEPS+1);
	
	is_params_set = true;

	mpc_allocate();

	//printf("N_STEPS = %i\n",N_STEPS);

	//printf("lipm2dstepModel Created\n");
		
}


MatrixXd lipmStep2dModel_FEL::forward_dynamics_x(int i, double t0, VectorXd &x0, double DT){
	
	MatrixXd Fx(MatrixXd::Identity(num_states,num_states));

	// continuous dynamics
	//comx
	Fx(0,0) += 0.5*DT*DT*9.81/HEIGHT;
	Fx(0,1) = DT;
	Fx(0,2) = -0.5*DT*DT*9.81/HEIGHT;
	//comxd
	Fx(1,0) = DT*9.81/HEIGHT;
	Fx(1,2) = -DT*9.81/HEIGHT;

	//copx
	//Fx(2,2) = 1;
	//leftx
	//rightx

	//comy
	Fx(5,5) += 0.5*DT*DT*9.81/HEIGHT;
	Fx(5,6) = DT;
	Fx(5,7) = -0.5*DT*DT*9.81/HEIGHT;

	//comyd
	Fx(6,5) = DT*9.81/HEIGHT;
	Fx(6,7) = -DT*9.81/HEIGHT;

	//copy
	//Fx(7,7) = 1;
	//lefty
	//righty

	if(step_indices(i)){
		Fx.row(3+((n_steps+FIRST_FOOT+1)%2)) = MatrixXd::Zero(1,num_states);
		Fx.row(8+((n_steps+FIRST_FOOT+1)%2)) = MatrixXd::Zero(1,num_states);
	}

	if (USE_FEEDBACK_ERROR_TRAJECTORY && i>=0 && i<N_TIMESTEPS){
		Fx(0,10+2*i+0) = 0.5*DT*DT;
		Fx(5,10+2*i+1) = 0.5*DT*DT;
		Fx(1,10+2*i+0) = DT;
		Fx(6,10+2*i+1) = DT;
	}

	return Fx;
}

MatrixXd lipmStep2dModel_FEL::forward_dynamics_u(int i, double t0, VectorXd &x0, double DT){
		
	MatrixXd Fu(MatrixXd::Zero(num_states,num_actions_continuous));
	
	Fu(0,0) = -0.5*DT*DT*9.81/HEIGHT;
	Fu(1,0) = -DT*9.81/HEIGHT;
	Fu(2,0) = 1;

	Fu(5,1) = -0.5*DT*DT*9.81/HEIGHT;
	Fu(6,1) = -DT*9.81/HEIGHT;
	Fu(7,1) = 1;

	//if(step_indices(i)){
	//	Fu.row(2+((n_steps+FIRST_FOOT)%2)) = MatrixXd::Zero(1,num_actions_continuous);
	//	Fu.row(6+((n_steps+FIRST_FOOT)%2)) = MatrixXd::Zero(1,num_actions_continuous);
	//}

	return Fu;
}

MatrixXd lipmStep2dModel_FEL::forward_dynamics_ud(int i, double t0, VectorXd &x0, double DT){
		
	MatrixXd Fud(MatrixXd::Zero(num_states,num_actions_discrete));

	if(step_indices(i)){
		Fud(3+((n_steps+FIRST_FOOT+1)%2),2*n_steps+0) = 1;
		Fud(8+((n_steps+FIRST_FOOT+1)%2),2*n_steps+1) = 1;
	}

	return Fud;
}

MatrixXd lipmStep2dModel_FEL::output_dynamics_x(int i, double t0, VectorXd &x0, double DT){
	
	MatrixXd Hx(MatrixXd::Zero(num_outputs,num_states));

	// continuous dynamics
	Hx(0,0) = 1;	// comx
	Hx(1,1) = 1;	// velx
	Hx(2,0) = 9.81/HEIGHT;	// accx
	Hx(2,2) = -9.81/HEIGHT;	// accx
	Hx(3,2) = 1;	// copx

	Hx(4,3) = 1;	// leftx
	Hx(5,4) = 1;	// rightx

	Hx(6,5) = 1;	// comy
	Hx(7,6) = 1;	// vely
	Hx(8,5) = 9.81/HEIGHT;	// accy
	Hx(8,7) = -9.81/HEIGHT;	// accy
	Hx(9,7) = 1;	// copy
	Hx(10,8) = 1;	// lefty
	Hx(11,9) = 1;	// righty

	if(USE_FEEDBACK_ERROR_TRAJECTORY && i>=0 && i<N_TIMESTEPS){
		Hx(2,10+2*i+0) = 1;
		Hx(8,10+2*i+1) = 1;
	}

	return Hx;
}

MatrixXd lipmStep2dModel_FEL::output_dynamics_u(int i, double t0, VectorXd &x0, double DT){

	
	MatrixXd Hu(MatrixXd::Zero(num_outputs,num_actions_continuous));

	//Hu(2,0) = -9.81/HEIGHT;	// accx
	//Hu(3,0) = 1.0;	// copx
	//Hu(8,1) = -9.81/HEIGHT;	// accy
	//Hu(9,1) = 1.0;	// copy

	return Hu;
}


void lipmStep2dModel_FEL::precompute_dynamics(double t0, VectorXd &x0){
	int i;
	double DT;
	double T_PHASE;
	int N_PHASE;
	int n0 = floor(t0/TIMESTEP);

	n_steps = 0;	

	T(0) = n0*TIMESTEP;
	A.block(0,0,num_states,num_states) = MatrixXd::Identity(num_states,num_states);
	
	MatrixXd Hx(output_dynamics_x(0, t0, x0, TIMESTEP));
	MatrixXd Hu(output_dynamics_u(0, t0, x0, TIMESTEP));
	
	C.block(0,0,num_outputs,num_states) = Hx;
	D.block(0,0,num_outputs,num_vars()) = Hu*MatrixXd::Identity(num_actions_continuous,num_vars());

	if(n0<N_SHIFT)	contact_state(0) = 0;
	else			contact_state(0) = 1 + FIRST_FOOT;
	
	for(i=1;i<N_TIMESTEPS+1;i++){
		// build trajectory dynamics matrices A and B

		T(i) = T(i-1)+TIMESTEP;

		DT = TIMESTEP;

		if(n0<N_SHIFT){
			if ((T(i)-T_SHIFT>0) && (T(i-1)-T_SHIFT<0)){
				printf("1 %f %f\n",T(i-1),T(i));
				DT = T(i) - T_SHIFT;
				T(i) = T_SHIFT;
			}
		}else{
			if ((T(i)>(T_SWING+T_SHIFT)) && (T(i-1)<(T_SWING+T_SHIFT))){
				printf("2 %f %f\n",T(i-1),T(i));
				DT = T(i) - (T_SWING+T_SHIFT);
				T(i) = (T_SWING+T_SHIFT);
			}
		}

		T_PHASE = T(i) - n_steps*(T_SWING+T_SHIFT);

		if(T_PHASE>=(T_SWING+T_SHIFT) && n_steps<N_STEPS){
			step_indices(i) = 1;
			T_PHASE -= (T_SWING+T_SHIFT);
		}


		if(T_PHASE<T_SHIFT || n_steps>=N_STEPS)	contact_state(i) = 0;
		else	contact_state(i) = 1+((n_steps+FIRST_FOOT)%2);
		
		MatrixXd Fx(forward_dynamics_x(i, t0, x0, DT));
		MatrixXd Fu(forward_dynamics_u(i, t0, x0, DT));

		A.block(i*num_states,0,num_states,num_states) = Fx*A.block((i-1)*num_states,0,num_states,num_states);
		B.block(i*num_states,0,num_states,num_vars()) = Fx*B.block((i-1)*num_states,0,num_states,num_vars());
		B.block(i*num_states,(i-1)*num_actions_continuous,num_states,num_actions_continuous) = Fu;

		if(num_actions_discrete>0){
			MatrixXd Fud(forward_dynamics_ud(i, t0, x0, DT));
			B.block(i*num_states,N_TIMESTEPS*num_actions_continuous,num_states,num_actions_discrete) += Fud;
		}

		MatrixXd Hx(output_dynamics_x(i, t0, x0, TIMESTEP));
		MatrixXd Hu(output_dynamics_u(i, t0, x0, TIMESTEP));

		C.block(i*num_outputs,0,num_outputs,num_states) = Hx*A.block(i*num_states,0,num_states,num_states);
		D.block(i*num_outputs,0,num_outputs,num_vars()) = Hx*B.block(i*num_states,0,num_states,num_vars());

		if(i<N_TIMESTEPS){
			MatrixXd G(MatrixXd::Zero(num_actions_continuous,num_vars()));
			G.block(0,i*num_actions_continuous,num_actions_continuous,num_actions_continuous) = MatrixXd::Identity(num_actions_continuous,num_actions_continuous);
			D.block(i*num_outputs,0,num_outputs,num_vars()) += Hu*G;
		}

		if (step_indices(i))	n_steps++;

	}

	std::cout << T.transpose() << std::endl;
	std::cout << step_indices.transpose() << std::endl;
	std::cout << contact_state.transpose() << std::endl;

	precomputed_dynamics = true;
}

void lipmStep2dModel_FEL::precompute_cost(double t0, VectorXd &x0){
	
	// requires A and B precomputed
	if(!precomputed_dynamics)	precompute_dynamics(t0,x0);

	int i;
	
	MatrixXd Goal_x(0.5*(A.row(num_states*(N_TIMESTEPS)+3)+A.row(num_states*(N_TIMESTEPS)+4)));
	MatrixXd Goal_ux(0.5*(B.row(num_states*(N_TIMESTEPS)+3)+B.row(num_states*(N_TIMESTEPS)+4)));
	MatrixXd Goal_y(0.5*(A.row(num_states*(N_TIMESTEPS)+8)+A.row(num_states*(N_TIMESTEPS)+9)));
	MatrixXd Goal_uy(0.5*(B.row(num_states*(N_TIMESTEPS)+8)+B.row(num_states*(N_TIMESTEPS)+9)));
	
	PreLuu.setZero(num_vars(),num_vars());
	PreLux.setZero(num_vars(),num_states);
	PreLxx.setZero(num_states,num_states);
	PreLu.setZero(num_vars());
	PreLx.setZero(num_states);
	PreL0 = 0;	
		
	n_steps = 0;
	
	for(i=0;i<N_TIMESTEPS+1;i++){
		MatrixXd GlobalPosErr_x(- A.row(num_states*i+0));
		MatrixXd GlobalPosErr_ux(- B.row(num_states*i+0));
		MatrixXd GlobalPosErr_y(- A.row(num_states*i+5));
		MatrixXd GlobalPosErr_uy(- B.row(num_states*i+5));
		
		MatrixXd PosErr_x(Goal_x - A.row(num_states*i+0));
		MatrixXd PosErr_ux(Goal_ux - B.row(num_states*i+0));
		
		MatrixXd VelErr_x(A.row(num_states*i+1));
		MatrixXd VelErr_ux(B.row(num_states*i+1));
		
		MatrixXd PosErr_y(Goal_y - A.row(num_states*i+5));
		MatrixXd PosErr_uy(Goal_uy - B.row(num_states*i+5));
		
		MatrixXd VelErr_y(A.row(num_states*i+6));
		MatrixXd VelErr_uy(B.row(num_states*i+6));

		MatrixXd CapErr_x(A.row(num_states*i+0)+sqrt(HEIGHT/9.81)*A.row(num_states*i+1));
		MatrixXd CapErr_ux(B.row(num_states*i+0)+sqrt(HEIGHT/9.81)*B.row(num_states*i+1));
		MatrixXd CapErr_y(A.row(num_states*i+5)+sqrt(HEIGHT/9.81)*A.row(num_states*i+6));
		MatrixXd CapErr_uy(B.row(num_states*i+5)+sqrt(HEIGHT/9.81)*B.row(num_states*i+6));


		if(i<N_TIMESTEPS){

			MatrixXd ZmpErr_x(A.row(num_states*i+2));
			MatrixXd ZmpErr_ux(B.row(num_states*i+2));
		
			MatrixXd ZmpErr_y(A.row(num_states*i+7));
			MatrixXd ZmpErr_uy(B.row(num_states*i+7));

			MatrixXd MF_x(MatrixXd::Zero(1,num_states));
			MatrixXd MF_ux(MatrixXd::Zero(1,num_vars()));
			MatrixXd MF_y(MatrixXd::Zero(1,num_states));
			MatrixXd MF_uy(MatrixXd::Zero(1,num_vars()));

			//=========== ZMP error depends on contact state
			if(contact_state(i)==0){	// double

				MF_x = 0.5*(A.row(num_states*i+3) + A.row(num_states*i+4));
				MF_ux = 0.5*(B.row(num_states*i+3) + B.row(num_states*i+4));
			
				MF_y = 0.5*(A.row(num_states*i+8) + A.row(num_states*i+9));
				MF_uy = 0.5*(B.row(num_states*i+8) + B.row(num_states*i+9));
	
			}else if (contact_state(i)==1){	// left support

				MF_x = A.row(num_states*i+3);
				MF_ux = B.row(num_states*i+3);
						
				MF_y = A.row(num_states*i+8);
				MF_uy = B.row(num_states*i+8);

			}else if (contact_state(i)==2){	// right support

				MF_x = A.row(num_states*i+4);
				MF_ux = B.row(num_states*i+4);
			
				MF_y = A.row(num_states*i+9);
				MF_uy = B.row(num_states*i+9);

			}
		
			//error = zmp - mf
			ZmpErr_x = ZmpErr_x - MF_x;
			ZmpErr_ux = ZmpErr_ux - MF_ux;
			ZmpErr_y = ZmpErr_y - MF_y;
			ZmpErr_uy = ZmpErr_uy - MF_uy;

			// capture point error
			CapErr_x = CapErr_x - MF_x;
			CapErr_ux = CapErr_ux - MF_ux;
			CapErr_y = CapErr_y - MF_y;
			CapErr_uy = CapErr_uy - MF_uy;

			PreLuu += pos_cost*PosErr_ux.transpose()*PosErr_ux 
				+ pos_cost*PosErr_uy.transpose()*PosErr_uy 
				+ vel_cost*VelErr_ux.transpose()*VelErr_ux 
				+ vel_cost*VelErr_uy.transpose()*VelErr_uy 
				+ zmp_cost*ZmpErr_ux.transpose()*ZmpErr_ux 
				+ zmp_cost*ZmpErr_uy.transpose()*ZmpErr_uy
				+ capture_cost*CapErr_ux.transpose()*CapErr_ux 
				+ capture_cost*CapErr_uy.transpose()*CapErr_uy
				+ global_pos_cost*GlobalPosErr_ux.transpose()*GlobalPosErr_ux 
				+ global_pos_cost*GlobalPosErr_uy.transpose()*GlobalPosErr_uy;

			if(zmpd_cost>0){
				MatrixXd ZmpDif_ux(MatrixXd::Zero(1,num_vars()));
				ZmpDif_ux(2*i+0) = 1;		
				MatrixXd ZmpDif_uy(MatrixXd::Zero(1,num_vars()));
				ZmpDif_uy(2*i+1) = 1;
				PreLuu += zmpd_cost*(ZmpDif_ux.transpose()*ZmpDif_ux + ZmpDif_uy.transpose()*ZmpDif_uy);
			}

			PreLux += 2*(pos_cost*PosErr_ux.transpose()*PosErr_x 
				+ pos_cost*PosErr_uy.transpose()*PosErr_y 
				+ vel_cost*VelErr_ux.transpose()*VelErr_x 
				+ vel_cost*VelErr_uy.transpose()*VelErr_y 
				+ zmp_cost*ZmpErr_ux.transpose()*ZmpErr_x 
				+ zmp_cost*ZmpErr_uy.transpose()*ZmpErr_y
				+ capture_cost*CapErr_ux.transpose()*CapErr_x 
				+ capture_cost*CapErr_uy.transpose()*CapErr_y
				+ global_pos_cost*GlobalPosErr_ux.transpose()*GlobalPosErr_x 
				+ global_pos_cost*GlobalPosErr_uy.transpose()*GlobalPosErr_y);
			
			PreLxx += pos_cost*PosErr_x.transpose()*PosErr_x 
				+ pos_cost*PosErr_y.transpose()*PosErr_y 
				+ vel_cost*VelErr_x.transpose()*VelErr_x 
				+ vel_cost*VelErr_y.transpose()*VelErr_y 
				+ zmp_cost*ZmpErr_x.transpose()*ZmpErr_x 
				+ zmp_cost*ZmpErr_y.transpose()*ZmpErr_y
				+ capture_cost*CapErr_x.transpose()*CapErr_x 
				+ capture_cost*CapErr_y.transpose()*CapErr_y
				+ global_pos_cost*GlobalPosErr_x.transpose()*GlobalPosErr_x 
				+ global_pos_cost*GlobalPosErr_y.transpose()*GlobalPosErr_y;

			if(step_indices(i)){
				// x
				
				
				MatrixXd FootErr_x(A.row(num_states*i+3 + ((n_steps+FIRST_FOOT+1)%2) ) - A.row(num_states*i+3 + ((n_steps+FIRST_FOOT)%2) ));
				MatrixXd FootErr_ux(B.row(num_states*i+3 + ((n_steps+FIRST_FOOT+1)%2)  ) - B.row(num_states*i+3 + ((n_steps+FIRST_FOOT)%2) ));

				PreLuu += step_cost*FootErr_ux.transpose()*FootErr_ux;
				PreLux += 2*step_cost*FootErr_ux.transpose()*FootErr_x;	
				PreLxx += step_cost*FootErr_x.transpose()*FootErr_x;	
				// moved to update_cost() to allow changing DES_STEP_LENGTH
				//PreLu += -2*step_cost*DES_STEP_LENGTH*FootErr_ux.transpose();	
				//PreLx += -2*step_cost*DES_STEP_LENGTH*FootErr_x.transpose();	
				//PreL0 += DES_STEP_LENGTH*DES_STEP_LENGTH;	

				// y

				MatrixXd FootErr_y(A.row(num_states*i+8) - A.row(num_states*i+9));
				MatrixXd FootErr_uy(B.row(num_states*i+8) - B.row(num_states*i+9));

				PreLuu += step_cost*FootErr_uy.transpose()*FootErr_uy;
				PreLux += 2*step_cost*FootErr_uy.transpose()*FootErr_y;
				PreLxx += step_cost*FootErr_y.transpose()*FootErr_y;
				PreLu += -2*step_cost*STANCE_WIDTH*FootErr_uy.transpose();
				PreLx += -2*step_cost*STANCE_WIDTH*FootErr_y.transpose();
				PreL0 += STANCE_WIDTH*STANCE_WIDTH;

				n_steps++;
			}


		}else{
			// final state cost
			PreLuu += (pos_cost*PosErr_ux.transpose()*PosErr_ux 
				+ pos_cost*PosErr_uy.transpose()*PosErr_uy 
				+ vel_cost*VelErr_ux.transpose()*VelErr_ux 
				+ vel_cost*VelErr_uy.transpose()*VelErr_uy)*final_cost;

			PreLux += 2*(pos_cost*PosErr_ux.transpose()*PosErr_x 
				+ pos_cost*PosErr_uy.transpose()*PosErr_y 
				+ vel_cost*VelErr_ux.transpose()*VelErr_x 
				+ vel_cost*VelErr_uy.transpose()*VelErr_y)*final_cost;

			PreLxx += (pos_cost*PosErr_x.transpose()*PosErr_x 
				+ pos_cost*PosErr_y.transpose()*PosErr_y 
				+ vel_cost*VelErr_x.transpose()*VelErr_x 
				+ vel_cost*VelErr_y.transpose()*VelErr_y)*final_cost;
			
			//PreLu += -2*step_cost*STANCE_WIDTH*FootErr_uy.transpose()*final_cost;
			//PreLx += -2*step_cost*STANCE_WIDTH*FootErr_y.transpose()*final_cost;
			//PreL0 += STANCE_WIDTH*STANCE_WIDTH*final_cost;
		}
	}
	
	precomputed_cost = true;

}

void lipmStep2dModel_FEL::precompute_constraints(double t0, VectorXd &x0){
	
	// requires A and B precomputed
	if(!precomputed_dynamics)	precompute_dynamics(t0,x0);

	precomputed_constraints = true;
}

void lipmStep2dModel_FEL::update_constraints(double t0, VectorXd &x0){
	
	// requires A and B precomputed
	if(!precomputed_dynamics)		precompute_dynamics(t0,x0);
	if(!precomputed_constraints)	precompute_constraints(t0,x0);

	n_steps = 0;

	int i;
	int nc=0;
	for(i=0;i<N_TIMESTEPS;i++){

		if(nc>=num_inequality_constraints){
			printf("Warning - num_inequality_constraints exceded!\n");
			break;
		}			
		
		MatrixXd Z_x(A.row(num_states*i+2));
		MatrixXd Z_ux(B.row(num_states*i+2));
		
		MatrixXd Z_y(A.row(num_states*i+7));
		MatrixXd Z_uy(B.row(num_states*i+7));
		
		MatrixXd LF_x(A.row(num_states*i+3));
		MatrixXd LF_ux(B.row(num_states*i+3));

		MatrixXd RF_x(A.row(num_states*i+4));
		MatrixXd RF_ux(B.row(num_states*i+4));

		MatrixXd LF_y(A.row(num_states*i+8));
		MatrixXd LF_uy(B.row(num_states*i+8));

		MatrixXd RF_y(A.row(num_states*i+9));
		MatrixXd RF_uy(B.row(num_states*i+9));

		//CI^T x + ci0 >= 0

		//=========== ZMP constraints
		if(contact_state(i)==0){	// double
			
			if(n_steps==0 && USE_DOUBLE_SUPPORT_CONSTRAINTS){
				// box constraints around both feet
				//Zx < max(pL,pR) + FOOT_SIZE_X	==>  -Zx + max(pL,pR) + FOOT_SIZE_X >= 0
				//Zx > min(pL,pR) - FOOT_SIZE_X ==>   Zx - min(pL,pR) + FOOT_SIZE_X >= 0
				CI.block(0,nc,num_vars(),1) = -Z_ux.transpose();
				ci0.segment(nc,1) = - Z_x*x0 + (  std::max(x0(3),x0(4)) + FOOT_SIZE_X)*VectorXd::Ones(1);
				nc++;

				CI.block(0,nc,num_vars(),1) = Z_ux.transpose();
				ci0.segment(nc,1) =   Z_x*x0 + (- std::min(x0(3),x0(4)) + FOOT_SIZE_X)*VectorXd::Ones(1);		
				nc++;			

				CI.block(0,nc,num_vars(),1) = -Z_uy.transpose();
				ci0.segment(nc,1) = - Z_y*x0 + (  std::max(x0(8),x0(9)) + FOOT_SIZE_Y)*VectorXd::Ones(1);
				nc++;

				CI.block(0,nc,num_vars(),1) = Z_uy.transpose();
				ci0.segment(nc,1) =   Z_y*x0 + (- std::min(x0(8),x0(9)) + FOOT_SIZE_Y)*VectorXd::Ones(1);		
				nc++;	

				//double support constraints

				double dx;
				double dy;
				double z1x,z2x;
				double z1y,z2y;

				// ys0 = yL(0)
				// yf0 = yR(0)

				dy = x0(8)-x0(9);
				dx = x0(3)-x0(4);
				z1x = x0(4) + FOOT_SIZE_X;
				z2x = x0(4) - FOOT_SIZE_X;
				if (x0(3)>x0(4)){
					z1y = x0(9) - FOOT_SIZE_Y;
					z2y = x0(9) + FOOT_SIZE_Y;
				}else{
					z1y = x0(9) + FOOT_SIZE_Y;
					z2y = x0(9) - FOOT_SIZE_Y;
				}		
			
				CI.block(0,nc,num_vars(),1) =   - dy*Z_ux.transpose() + dx*Z_uy.transpose();
				ci0.segment(nc,1) = - dy*Z_x*x0 + dx*Z_y*x0 + (  dy*z1x - dx*z1y)*VectorXd::Ones(1);
				nc++;

				CI.block(0,nc,num_vars(),1) =     dy*Z_ux.transpose() - dx*Z_uy.transpose();
				ci0.segment(nc,1) =   dy*Z_x*x0 - dx*Z_y*x0 + (- dy*z2x + dx*z2y)*VectorXd::Ones(1);
				nc++;
				
			}else if(n_steps<N_STEPS && n_steps>0){

				if ((n_steps+FIRST_FOOT)%2==0){
					// put the weight on the left foot
			
					CI.block(0,nc,num_vars(),1) = -Z_ux.transpose() + LF_ux.transpose();
					ci0.segment(nc,1) = - Z_x*x0 + LF_x*x0 + FOOT_SIZE_X*VectorXd::Ones(1);

					nc++;

					CI.block(0,nc,num_vars(),1) = Z_ux.transpose() - LF_ux.transpose();
					ci0.segment(nc,1) =   Z_x*x0 - LF_x*x0 + FOOT_SIZE_X*VectorXd::Ones(1);
		
					nc++;
			
					CI.block(0,nc,num_vars(),1) = -Z_uy.transpose() + LF_uy.transpose();
					ci0.segment(nc,1) = - Z_y*x0 + LF_y*x0 + FOOT_SIZE_Y*VectorXd::Ones(1);

					nc++;

					CI.block(0,nc,num_vars(),1) = Z_uy.transpose() - LF_uy.transpose();
					ci0.segment(nc,1) =   Z_y*x0 - LF_y*x0 + FOOT_SIZE_Y*VectorXd::Ones(1);
		
					nc++;
				}else{
					// put the weight on the right foot
			
					CI.block(0,nc,num_vars(),1) = -Z_ux.transpose() + RF_ux.transpose();
					ci0.segment(nc,1) = - Z_x*x0 + RF_x*x0 + FOOT_SIZE_X*VectorXd::Ones(1);

					nc++;

					CI.block(0,nc,num_vars(),1) = Z_ux.transpose() - RF_ux.transpose();
					ci0.segment(nc,1) =   Z_x*x0 - RF_x*x0 + FOOT_SIZE_X*VectorXd::Ones(1);
		
					nc++;
			
					CI.block(0,nc,num_vars(),1) = -Z_uy.transpose() + RF_uy.transpose();
					ci0.segment(nc,1) = - Z_y*x0 + RF_y*x0 + FOOT_SIZE_Y*VectorXd::Ones(1);

					nc++;

					CI.block(0,nc,num_vars(),1) = Z_uy.transpose() - RF_uy.transpose();
					ci0.segment(nc,1) =   Z_y*x0 - RF_y*x0 + FOOT_SIZE_Y*VectorXd::Ones(1);
		
					nc++;
				}
			}else{
				// constraints are relative to center of two feet
				//Zx < 0.5*(pL + pR) + FOOT_SIZE_X	==>  -Zx + 0.5*(pL + pR) + FOOT_SIZE_X >= 0
				//Zx > 0.5*(pL + pR) - FOOT_SIZE_X	==>   Zx - 0.5*(pL + pR) + FOOT_SIZE_X >= 0
				MatrixXd MF_x(0.5*(LF_x + RF_x));
				MatrixXd MF_ux(0.5*(LF_ux + RF_ux));
				MatrixXd MF_y(0.5*(LF_y + RF_y));
				MatrixXd MF_uy(0.5*(LF_uy + RF_uy));

				CI.block(0,nc,num_vars(),1) = -Z_ux.transpose() + MF_ux.transpose();
				ci0.segment(nc,1) = - Z_x*x0 + MF_x*x0 + FOOT_SIZE_X*VectorXd::Ones(1);

				nc++;

				CI.block(0,nc,num_vars(),1) = Z_ux.transpose() - MF_ux.transpose();
				ci0.segment(nc,1) =   Z_x*x0 - MF_x*x0 + FOOT_SIZE_X*VectorXd::Ones(1);
		
				nc++;			

				CI.block(0,nc,num_vars(),1) = -Z_uy.transpose() + MF_uy.transpose();
				ci0.segment(nc,1) = - Z_y*x0 + MF_y*x0 + FOOT_SIZE_Y*VectorXd::Ones(1);

				nc++;

				CI.block(0,nc,num_vars(),1) = Z_uy.transpose() - MF_uy.transpose();
				ci0.segment(nc,1) =   Z_y*x0 - MF_y*x0 + FOOT_SIZE_Y*VectorXd::Ones(1); 
		
				nc++;
			}

		}else if (contact_state(i)==1){	// left support
			
			CI.block(0,nc,num_vars(),1) = -Z_ux.transpose() + LF_ux.transpose();
			ci0.segment(nc,1) = - Z_x*x0 + LF_x*x0 + FOOT_SIZE_X*VectorXd::Ones(1);

			nc++;

			CI.block(0,nc,num_vars(),1) = Z_ux.transpose() - LF_ux.transpose();
			ci0.segment(nc,1) =   Z_x*x0 - LF_x*x0 + FOOT_SIZE_X*VectorXd::Ones(1);
		
			nc++;
			
			CI.block(0,nc,num_vars(),1) = -Z_uy.transpose() + LF_uy.transpose();
			ci0.segment(nc,1) = - Z_y*x0 + LF_y*x0 + FOOT_SIZE_Y*VectorXd::Ones(1);

			nc++;

			CI.block(0,nc,num_vars(),1) = Z_uy.transpose() - LF_uy.transpose();
			ci0.segment(nc,1) =   Z_y*x0 - LF_y*x0 + FOOT_SIZE_Y*VectorXd::Ones(1);
		
			nc++;

		}else if (contact_state(i)==2){	// right support
			
			CI.block(0,nc,num_vars(),1) = -Z_ux.transpose() + RF_ux.transpose();
			ci0.segment(nc,1) = - Z_x*x0 + RF_x*x0 + FOOT_SIZE_X*VectorXd::Ones(1);

			nc++;

			CI.block(0,nc,num_vars(),1) = Z_ux.transpose() - RF_ux.transpose();
			ci0.segment(nc,1) =   Z_x*x0 - RF_x*x0 + FOOT_SIZE_X*VectorXd::Ones(1);
		
			nc++;
			
			CI.block(0,nc,num_vars(),1) = -Z_uy.transpose() + RF_uy.transpose();
			ci0.segment(nc,1) = - Z_y*x0 + RF_y*x0 + FOOT_SIZE_Y*VectorXd::Ones(1);

			nc++;

			CI.block(0,nc,num_vars(),1) = Z_uy.transpose() - RF_uy.transpose();
			ci0.segment(nc,1) =   Z_y*x0 - RF_y*x0 + FOOT_SIZE_Y*VectorXd::Ones(1);
		
			nc++;
		}

		//=========== Foot Step Constraints
		if(step_indices(i)){

			// yL >= yR + FOOT_SIZE_Y + 0.05	==>		+ yL - yR - FOOT_SIZE_Y - 0.05	>= 0
			CI.block(0,nc,num_vars(),1) =   LF_uy.transpose() - RF_uy.transpose();
			ci0.segment(nc,1) = LF_y*x0 - RF_y*x0 - (FOOT_SIZE_Y + 0.05)*VectorXd::Ones(1);
			nc++;

			// yL <= yR + MAX_STANCE_WIDTH		==>		- yL + yR + MAX_STANCE_WIDTH	>= 0
			CI.block(0,nc,num_vars(),1) = - LF_uy.transpose() + RF_uy.transpose();
			ci0.segment(nc,1) = - LF_y*x0 + RF_y*x0 + (MAX_STANCE_WIDTH)*VectorXd::Ones(1);
			nc++;

			// xL <= xR + MAX_STANCE_WIDTH		==>		- xL + xR + MAX_STANCE_WIDTH	>= 0
			CI.block(0,nc,num_vars(),1) = - LF_ux.transpose() + RF_ux.transpose();
			ci0.segment(nc,1) = - LF_x*x0 + RF_x*x0 + (MAX_STANCE_WIDTH)*VectorXd::Ones(1);
			nc++;

			// xL >= yR - MAX_STANCE_WIDTH		==>		+ yL - yR + MAX_STANCE_WIDTH	>= 0
			CI.block(0,nc,num_vars(),1) =   LF_ux.transpose() - RF_ux.transpose();
			ci0.segment(nc,1) =   LF_x*x0 - RF_x*x0 + (MAX_STANCE_WIDTH)*VectorXd::Ones(1);
			nc++;

			// swing velocity
		}

		if (step_indices(i))	n_steps++;

	}

	if(nc<num_inequality_constraints){
		printf("Warning - num_inequality_constraints too high (%i>%i)\n",num_inequality_constraints,nc);
	}

}

void lipmStep2dModel_FEL::update_cost(double t0, VectorXd &x0){
	int i;
	// requires A and B precomputed
	if(!precomputed_dynamics)	precompute_dynamics(t0,x0);
	if(!precomputed_cost)		precompute_cost(t0,x0);
	
	// copy precomputed matrices
	Luu = PreLuu;
	Lux = PreLux;
	Lxx = PreLxx;
	Lu = PreLu;
	Lx = PreLx;	
	L0 = PreL0;

	// do updates
	n_steps = 0;
	for(i=0;i<N_TIMESTEPS+1;i++){
		if(step_indices(i)){

			MatrixXd FootErr_x(A.row(num_states*i+3 + ((n_steps+FIRST_FOOT+1)%2) ) - A.row(num_states*i+3 + ((n_steps+FIRST_FOOT)%2) ));
			MatrixXd FootErr_ux(B.row(num_states*i+3 + ((n_steps+FIRST_FOOT+1)%2)  ) - B.row(num_states*i+3 + ((n_steps+FIRST_FOOT)%2) ));

			//Luu += step_cost*FootErr_ux.transpose()*FootErr_ux;
			//Lux += 2*step_cost*FootErr_ux.transpose()*FootErr_x;	
			//Lxx += step_cost*FootErr_x.transpose()*FootErr_x;	
			Lu += -2*step_cost*DES_STEP_LENGTH*FootErr_ux.transpose();	
			Lx += -2*step_cost*DES_STEP_LENGTH*FootErr_x.transpose();	
			L0 += DES_STEP_LENGTH*DES_STEP_LENGTH;	

			n_steps++;
		}
	}


	H = Luu;
	f = 0.5*(Lu + Lux*x0);

}

VectorXd lipmStep2dModel_FEL::simulate(double t0, VectorXd &x0, VectorXd &U){
	X = A*x0 + B*U;
	Y = C*x0 + D*U;
	return X;
}

//double lipmStep2dModel_FEL::one_step_cost(double t0, VectorXd &x0, double t, VectorXd &x, VectorXd &u){
//	return pos_cost*x(0)*x(0) + vel_cost*x(1)*x(1);
//}

//double lipmStep2dModel_FEL::total_cost(double t0, VectorXd &x0, VectorXd &U){
//	VectorXd cost(U.transpose()*(H*U + f*x0) + x0.transpose()*V0*x0);
//
//	//VectorXd X(simulate(t0,x0,U));
//	//int i;
//	//for(i=0;i<N_TIMESTEPS;i++){
//	//	VectorXd x(X.segment(num_states*i,num_states));
//	//	VectorXd u(U.segment(num_actions_continuous*i,num_actions_continuous));
//	//	cost += one_step_cost(t0,x0,t0,x,u);
//	//}
//
//	return cost(0);
//}