#include "config_file.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include "string_util.h"
#include "search_dirs.h"

char *get_next_dataline( FILE *file ) {
  static char buf[1024],buf2[1024];
  while( fgets( buf, 1024, file ) ) {
    strip_newline( buf );
    if( buf[0] == '#' ) continue;
    if( 0 == sscanf( buf, "%1000s",buf2 ) ) continue;
    if( strlen(buf) == 0 ) continue;
    return buf;
  }
  return NULL;
}

void add_search_paths( ConfigurationData *cd ) {
  if( conf_has_string( cd, "SEARCH_PATHS" ) ) {
    char *search_path_fn = get_conf_string( cd, "SEARCH_PATHS" );
    load_search_dirs_fn( search_path_fn );
    free( search_path_fn );
  }
}

char *get_next_stringline( FILE *file ) {
  static char buf[1024],buf2[1024];
  while( fgets( buf, 1024, file  ) ) {
    if( buf[0] != '!' ) continue;
    if( 0 == sscanf( buf, "%1000s",buf2 ) ) continue;
    if( strlen( buf ) == 0 ) continue;
    return buf;
  }
  return NULL;
}  

ConfigurationData *load_conf_data( const char *filename ) {
  int i, buflen=1024;
  char buf[1024];
  FILE *cf = NULL;
  char *found_file = search_file( filename );
  int lc,dc;
  char *line;
  char name[1024];
  int max_dc = 0, pos, cc;
  int sc = 0;
  ConfigurationData *cd;
  int ln = 0;

  if( filename == NULL ) {
    printf( "load_conf_data: NULL filename\n" );
    return NULL;
  }
  if( !found_file ) {
    printf( "load_conf_data: %s could not be found\n", filename );
    return NULL;
  }

  cf = fopen( found_file, "r" );

  if( cf == NULL ) {
    printf( "load_conf_data: %s: %s", filename, strerror(errno) );
    free( found_file );
    return NULL;
  }

  lc=0;
  
  while( NULL != (line = get_next_dataline(cf) ) ) {
	int pos=0,cc;
    if( strlen(line) == 0 ) continue;

    lc++;
    
    dc = 0;
    while( 0 < sscanf( line+pos, "%s%n", name,&cc ) ) {
      pos+=cc;
      dc++;
    }
    
    dc--; //Row name is not a data element

    if( dc > max_dc ) max_dc = dc;
  }

  fseek( cf, 0,SEEK_SET );
  
  while( NULL != get_next_stringline(cf) ) {
    sc++;
  }

  cd = (ConfigurationData *)malloc( sizeof( ConfigurationData ) );

  //  printf( "Loading %s: %d by %d\n", filename, lc, dc );

  cd->names = (char **)malloc( sizeof(char*) * lc );
  cd->row_els = (int *)malloc( sizeof(char*) * lc );
  cd->data = (float **)malloc( sizeof(float*) * lc );
  cd->filename = strdup( found_file );
  free( found_file );
  if( sc > 0 ) {
    cd->string_names = (char **)malloc( sizeof(char*) * sc );
    cd->strings = (char **)malloc( sizeof(char*) * sc );
  }
  else {
    cd->string_names = NULL;
    cd->strings = NULL;
  }
  
  fseek( cf, 0, SEEK_SET );

  for( i=0; i<lc;i++ ) {
    if( max_dc > 0 ) {
      cd->data[i] = (float *)malloc( sizeof(float) * max_dc );
    }
    else {
      cd->data[i] = NULL;
    }
  }

  cd->rows = lc;
  cd->elements = max_dc;
  cd->num_strings = sc;

  fseek( cf, 0, SEEK_SET );

  
  while( NULL != (line = get_next_dataline(cf) ) ) {
    pos=0;
    sscanf( line, "%1023s%n", buf,&cc );
    cd->names[ln] = strdup( buf );
    pos += cc;
    for( i=0; i<max_dc; i++ ) {
      if( !(0 < sscanf( line+pos, "%1023s%n", buf, &cc )) ) {
	break;
      }
      cd->data[ln][i] = atof( buf );
      pos += cc;
    }
    cd->row_els[ln] = i;
    //    printf( "Row %d has len %d\n", ln, i );
    ln++;
  }
  fseek( cf, 0, SEEK_SET );

  ln = 0;
  while( NULL != (line = get_next_stringline(cf) ) ) {
    sscanf( line, "%1023s%n", buf, &cc );
    cd->string_names[ln] = strdup( buf+1 );
    sscanf( line+cc, "%1023s", buf );
    cd->strings[ln] = strdup( buf );
    ln++;
  }

  fclose( cf );

  add_search_paths( cd );
  
  return cd;
}
  
int conf_has_field( ConfigurationData *cd, const char *name ) {
  int i;
  if( cd == NULL ) {
    printf( "conf_has_field: null config data" );
    return 0;
  }
  for( i=0; i < cd->rows; i++ ) {
    if( 0==strcmp( cd->names[i], name ) ) {
      return 1;
    }
  }
  return 0;
}

float *get_conf_row( ConfigurationData *cd, const char *name ) {
  int i;
  if( cd == NULL ) {
    printf( "get_conf_row: null config data" );
    return NULL;
  }
  for( i=0; i < cd->rows; i++ ) {
    if( 0==strcmp( cd->names[i], name ) ) {
      return cd->data[i];
    }
  }

  printf( "get_conf_row: %s was not found in %s\n", name, cd->filename );

  return NULL;
}

int conf_has_string( const ConfigurationData *cd, const char *name ) {
  int i;
  if( cd == NULL ) {
    printf( "conf_has_field: null config data" );
    return 0;
  }
  for( i=0; i < cd->num_strings; i++ ) {
    if( 0==strcmp( cd->string_names[i], name ) ) {
      return 1;
    }
  }
  return 0;
}

char *get_conf_string( const ConfigurationData *cd, const char *name ) {
  int i;
  if( cd == NULL ) {
    printf( "get_conf_string: null config data" );
    return NULL;
  }
  for( i=0; i < cd->num_strings; i++ ) {
    if( 0==strcmp( cd->string_names[i], name ) ) {
      return strdup( cd->strings[i] );
    }
  }

  printf( "get_conf_string: %s was not found in %s", name, cd->filename );

  return NULL;
}

int config_write( ConfigurationData *data ) {
  int i,j;
  FILE *cf;

  if( data == NULL ) {
    printf( "config_write: null config data" );
    return -1;
  }

  if( data->filename == NULL ) {
    perror( "config_write: no filename" );
    return -1;
  }

  cf = fopen( data->filename, "w" );
  if( cf == NULL ) {
    perror( "config_write: fopen" );
    return -1;
  }

  for( i=0; i<data->rows;i++ ) {
    fprintf( cf, "%s", data->names[i] );
    for( j=0; j<data->row_els[i];j++ ) {
      fprintf( cf, "\t%f", data->data[i][j] );
    }
    fprintf( cf, "\n" );
  }  

  for( i=0; i<data->num_strings;i++ ) {
    fprintf( cf, "!%s %s\n", data->string_names[i], data->strings[i] );
  }

  fclose( cf );
  return 0;
}

void free_conf_data( ConfigurationData *data ) {
  int i;
  if( data == NULL ) {
    printf( "free_conf_data: cannot free NULL data" );
    return;
  }

  if( data->filename != NULL ) {
    free( data->filename );
  }
  
  for( i=0; i<data->rows; i++ ) {
    if( data->names[i] != NULL ) {
      free( data->names[i] );
    }
    if( data->data[i] != NULL ) {
      free( data->data[i] );
    }
  }

  for( i=0;i<data->num_strings;i++ ) {
    if( data->string_names[i] != NULL ) {
      free( data->string_names[i] );
    }
    if( data->strings[i] != NULL ) {
      free( data->strings[i] );
    }
  }

  if( data->rows > 0 ) {
    free( data->names );
    free( data->data );
    free( data->row_els );
  }

  if( data->num_strings > 0 ) {
    free( data->string_names );
    free( data->strings );
  }
}

ConfigurationData *new_conf_data( int elements, int rows ) {
  ConfigurationData *cd;
  int i;
  
  cd = (ConfigurationData *)malloc( sizeof( ConfigurationData ) );
  cd->elements = elements;
  cd->rows = rows;

  cd->num_strings = 0;
  cd->strings = NULL;
  cd->string_names = NULL;

  cd->names = (char**)malloc( sizeof( char * ) * rows );
  cd->data = (float**)malloc( sizeof( float * ) * rows );
  cd->row_els = (int *)malloc( sizeof( int ) * rows );

  for( i = 0; i<rows; i++ ) {
    cd->data[i] = (float*)malloc( sizeof( float ) * elements );
    cd->names[i] = NULL;
    cd->row_els[i] = elements;
  }
  
  cd->filename = NULL;

  return cd;
}

int get_conf_row_len( const ConfigurationData *data, const char *buf ) {
  int i;
  if( data == NULL ) {
    printf( "get_conf_row_len: null config data" );
    return 0;
  }
  for( i=0; i < data->rows; i++ ) {
    if( 0==strcmp( data->names[i], buf ) ) {
      return data->row_els[i];
    }
  }

  return 0;
}

void save_vector( ConfigurationData *conf, char *name, char *filenum ) {
  char buf[1024];
  char *filename = get_conf_string( conf, name );
  sprintf( buf, "cp %s %s.%s", filename, filename, filenum );
  system( buf );
}

void load_vector( ConfigurationData *conf, char *name, float **vec, int *v_len ) {
  char *filename;
  FILE *file;
  int i;
  float tmp;

  *vec = NULL;
  *v_len = 0;

  printf(  "Loading Vector: %s\n", name );

  filename = get_conf_string( conf, name );
  if( filename == NULL ) {
    return;
  }
  else {
    printf( "\tvector stored in: %s\n", filename );
    file = fopen( filename, "rt" );    
    if( file == NULL ) { return; }
    while ( !feof( file ) ) {
      int err = fscanf( file, "%f", &tmp ); ;
      if (err == 1 ) {
	(*v_len)+=1;
      }
    }

    printf( "\tvector length: %d\n", *v_len );

    fseek( file, 0, SEEK_SET );    
    *vec = (float *)malloc( sizeof( float ) * (*v_len) );
    for( i=0; i<*v_len; i++ ) {
      fscanf( file, "%f", (*vec)+i );
    }
    fclose( file );

    printf( "\tsuccess\n" );
  }
}

void set_value( ConfigurationData *cd, const char *name, float value ){
  int i;
  if( conf_has_field( cd, name ) ) {
	  for( i=0; i < cd->rows; i++ ) {
		if( 0==strcmp( cd->names[i], name ) ) {
			cd->data[i][0] = value;
			return;
		}
	}
  }
  else {
    printf( "field \"%s\" was not found, using default value (%f)\n", name, value );
  }
}

float get_value_with_default( ConfigurationData *conf, const char *name, float default_value ) {
  if( conf_has_field( conf, name ) ) {
    return *get_conf_row( conf, name );
  }
  else {
    printf( "field \"%s\" was not found, using default value (%f)\n", name, default_value );
    return default_value;
  }
}

char *get_conf_string_with_default( const ConfigurationData *data, const char *string_name, const char *default_string ) {
  if( conf_has_string( data, string_name ) ){
    return get_conf_string( data, string_name );
  }
  printf( "%s not found, using default: %s\n", string_name, default_string );
  return strdup( default_string );
}

int conf_num_rows( ConfigurationData *cd ) {
  return cd->rows;
}
char *conf_row_name( ConfigurationData *cd, int idx ) {
  if( idx < 0 || idx >= cd->rows ) {
    printf( "requested bad row number: %d in %s (has %d)", idx, cd->filename, cd->rows );
    return NULL;
  }

  return cd->names[idx];
}
