function [err] = asymp_exp(d,N,r,k1,k2,m,varargin),
%% simulation routine for empirically evaluating the rate of
%% convergence of the compressive subspace learning method. 
%% Generates a dxN matrix (either low_rank or aligned) of
%% "approximate" rank r and runs the compressive algorithm,
%% plotting and saving the error and rescaled error as a function
%% of N.

%% Parameters:
%% d,N -- problem dimension
%% r -- rank (or approximate rank)
%% k1 -- range parameter on the singular values in the directions
%% of interest.
%% k2 -- gap parameter between upper and lower ranks
%% m -- number of measurements to make per column
%% varargin -- 4 optional arguments
%%   decor -- do we uncorrelate the two measurements we make
%%   compress -- compressive or subsampled
%%   save -- save the data
%%   plot -- plot the error rates. 
  p = inputParser;
  p.addParamValue('decor', 1, @isscalar);
  p.addParamValue('compress', 1, @isscalar);
  p.addParamValue('save', 0, @isscalar);
  p.addParamValue('plot', 1, @isscalar);

  p.parse(varargin{:});
  Prms = p.Results;

  if k2 == 0,
      M = low_rank_matrix(d,N,r,k1);
  else,
% $$$       M = sep_matrix_additive(d,N,r,k1,k2);
      M = aligned_matrix(d,N,r,k2);
  end;

  [u,s,v] = eigs(M*M',r);
  err = [];
  Chat = zeros(d,d);
  for t=1:N,
      if Prms.compress,
          if Prms.decor,
              At1 = normrnd(0,1,d,int64(m/2));
              Pt1 = At1*(At1'*At1)^(-1)*At1';
              At2 = normrnd(0,1,d,int64(m/2));
              Pt2 = At2*(At2'*At2)^(-1)*At2';
              Ct = 0.5* (Pt1*M(:,t)*(Pt2*M(:,t))' + Pt2*M(:,t)*(Pt1*M(:,t))');
          else,
              At = normrnd(0,1,d,m);
              Pt = At*(At'*At)^(-1)*At';
              Ct = Pt*M(:,t)*(Pt*M(:,t))';
          end;
      else,
          if Prms.decor,
              u_obs = unique(randi(d,[int32(m/2),1]));
              v_obs = unique(randi(d,[int32(m/2),1]));
              uvec = zeros(d,1);
              vvec = zeros(d,1);
              
              uvec(u_obs,1) = M(u_obs,t);
              vvec(v_obs,1) = M(v_obs,t);
              Ct = 0.5*(uvec*vvec' + vvec*uvec');
          else,
              u_obs = unique(randi(d,[m,1]));
              uvec = zeros(d,1);
              uvec(u_obs,1) = M(u_obs,t);
              Ct = uvec*uvec';
          end;
      end;
      Chat = Chat + Ct;
      if mod(t, 20) == 0,
          [up,sp,vp] = eigs(Chat,r);
          err(t/20) = norm(up*up'-u*u',2);
          fprintf('t = %d, err = %0.3f\n', t, err(t/20));
      end;
  end;

  if Prms.plot == 0,
      return;
  end;

  f = figure();
  plot(err);
  h = xlabel('# of Samples (n)');
  set(h, 'FontSize', 20);
  h = ylabel('Error');
  set(h, 'FontSize', 20);
  set(gca, 'Fontsize', 20);
  if Prms.save == 1  && k2 == 0,
      saveas(f, sprintf('./figs/conv_d=%d_r=%d_m=%d.eps', d, r, m),'eps');
  elseif Prms.save == 1,
      saveas(f, sprintf('./figs/conv_approx_d=%d_r=%d_m=%d.eps', d, r, m), 'eps');
  end;

  f = figure();
  plot(sqrt(20:20:N).*err);
  h = xlabel('# of Samples (n)');
  set(h, 'FontSize', 20);
  h = ylabel('sqrt(n) * Error');
  set(h, 'FontSize', 20);
  set(gca, 'Fontsize', 20);
  if Prms.save == 1 && k2 == 0,
      saveas(f, sprintf('./figs/conv_rescaled_d=%d_r=%d_m=%d.eps', ...
                         d, r, m), 'eps');
  elseif Prms.save == 1,
      saveas(f, sprintf('./figs/conv_approx_rescaled_d=%d_r=%d_m=%d.eps', ...
                         d, r, m), 'eps');
  end;