// Loop editor for the Pausch Bridge lighting system written using p5js
// (https://p5js.org/).  This implements a 'piano roll' drawing canvas in which
// each row can be sequenced out to the 57 lighting groups of the bridge.  The
// preview playback continuously loops through the image as it is drawn.
// 
// Original version by Nick Diamant <ndiamant@andrew.cmu.edu>,
// downloaded from https://editor.p5js.org/dinkolas/sketches/Ai9tqWsYU

// Changes:

// 2021-03-26:  garthz@cmu.edu
//		added comments to the code
//		added the keyPressed implementation to save the image

// -------------------------------------------------------------------------------------
let N = 300;  // Number of video frames in the loop.
let w = 57;   // Width of the preview display, e.g. number of lighting groups on bridge.
let row = 0;  // Playhead index, current frame of loop.
let h = 0;    // Current hue index for drawing brush.

function setup() {
    createCanvas(windowWidth, windowHeight);
    frameRate(30);
    background(0);
}

function windowResized() {
    resizeCanvas(windowWidth, windowHeight);
    background(0);
}

function keyPressed() {
    // fetch the 'piano roll' portion of the canvas
    pixels = get(width * 0.1, height * 0.2, width * 0.8, height * 0.7)
    pixels.save('Pausch01', 'png');
}

function draw() {

    // Draw a new line segment whenever the mouse button is pressed.  The color
    // continuously cycles through hues.  Four overlapping segments are drawn with
    // varying width to create a soft edge effect.
    if (mouseIsPressed) {
	h = (h + 1) % 100;
	colorMode(HSB, 100);
	blendMode(LIGHTEST);
	for (let i = 0; i < 4; i++) {
	    strokeWeight(map(i, 0, 3, 40, 5));
	    stroke(h, 100, map(i, 0, 3, 10, 100));
	    line(mouseX, mouseY, pmouseX, pmouseY);
	}
    }

    // Clear the margins to black all around to leave space for the playbar and preview pane.
    blendMode(BLEND);
    noStroke();
    fill(0);
    rect(0, 0, width, height * 0.2);
    rect(0, 0, width * 0.1, height);
    rect(width * 0.9, 0, width * 0.1, height);
    rect(0, height * 0.9, width, height * 0.1);

    // Draw the playhead dot moving vertically in the playbar.   
    let rowYCoord = map(row, 0, N - 1, height * 0.2, height * 0.9);
    colorMode(RGB, 100);
    noStroke();
    fill(100);
    ellipse(width * 0.05, rowYCoord, 10, 10);

    // Read back the canvas image pixels and extract the currently playing row into an array.
    let colors = [];
    loadPixels();
    let d = pixelDensity();
    let y = floor(rowYCoord);
    for (let i = 0; i < w; i++) {
	let x = floor(map(i, 0, w - 1, width * 0.1, width * 0.9));
	//4 * ((y * d + j) * width * d + (x * d + i));

	let p = 4 * ((y * d) * width * d + (x * d));
	let r = pixels[p];
	let g = pixels[p + 1];
	let b = pixels[p + 2];
	colors.push({
	    r: r,
	    g: g,
	    b: b
	});
    }

    // Redraw the preview pane at the top with rectangles representing the
    // bridge lighting colors using the color data extracted from the currently playing row.
    for (let i = 0; i < w; i++) {
	let x = map(i, 0, w - 1, width * 0.1, width * 0.9);
	let c = colors[i];
	noStroke();
	fill(c.r, c.g, c.b);
	rect(x, height * 0.05, width * 0.8 / w, height * 0.1);
    }

    // Advance the playhead.
    row = (row + 1) % N;
}

