structure Typing :> TYPING =
struct

  open MinML

  exception TypeError

  fun greatest (FAIL, t) = t
    | greatest (t, FAIL) = t
    | greatest (t1, t2) = t1

  (* Make sure two types are the same. *)

  fun equal (FAIL, _) = ()
    | equal (_, FAIL) = ()
    | equal (t,t') = if t = t' then () else raise TypeError

  (* The empty environment. *)

  val empty = fn _ => raise TypeError

  (* g++(x,e) adds a variable x with type e to environment g. *)

  infix 1 ++
  fun (g ++ (x,e)) = fn x' => if x = x' then e else g(x')

  (* match : env*pat*typ -> env* define the patternmatching judgment *)

  fun match (g,WildPat,t) = g
    | match (g,UnitPat,t) = (equal(t,UNIT); g)
    | match (g,VarPat(v,vt),t) = (equal(t,vt); g++(v,t))
    | match (g,PairPat(p1,p2),t) = 
      (case t 
	 of TIMES(t1,t2) =>
	   let val g' = match(g,p1,t1)
	     val g'' = match(g',p2,t2)
	   in g'' end
	  | _ => raise TypeError)

  (* typing(g,e) returns a type of e in context g, or raises
     TypeError if e has no type in context g. *)

  fun typing (_,_,Int _) = INT
    | typing (_,_,Bool _) = BOOL
    | typing (_,_,UnitE) = UNIT

    | typing (l,g,If(e,e1,e2)) =
      let val (t,t1,t2) = (typing(l,g,e), typing(l,g,e1), typing(l,g,e2))
       in equal(t,BOOL); equal(t1,t2); greatest(t1,t2) end

    | typing (l,g,Primop(p,es)) =
      let val ts = map (fn e => typing(l,g,e)) es
          val (args,res) = typeOfPrimop p
       in List.app equal (ListPair.zip(ts,args)); res end

    | typing (l,g,Fun(f,x,t,t',e)) =
      let val t'' = typing(l,g++(f,ARROW(t,t'))++(x,t),e)
       in equal(t',t''); ARROW(t,t') end

    | typing (l,g,Apply(e,e')) =
      (case (typing(l,g,e),typing(l,g,e'))
	 of (ARROW(t,t'),t'') => (equal(t,t''); t')
	  | _ => raise TypeError)

    | typing (l,g,Pair(e1,e2)) =
      let val (t1,t2) = (typing(l,g,e1),typing(l,g,e2))
       in TIMES(t1,t2) end

    | typing (l,g,Bind(p,e1,e2)) =
      let val t1 = typing(l,g,e1)
	  val g' = match(g,p,t1)
	  val t = typing(l,g',e2)
       in t end

    | typing (l,g,Inleft(t1,t2,e)) =
      let val t = typing(l,g,e)
       in equal(t,t1); PLUS(t1,t2) end

    | typing (l,g,Inright(t1,t2,e)) =
      let val t = typing(l,g,e)
       in equal(t,t2); PLUS(t1,t2) end

    | typing (l,g,Case(e,v1,t1,e1,v2,t2,e2)) =
      let val (t, t1', t2') = 
       (typing(l,g,e),typing(l,g++(v1,t1),e1),typing(l,g++(v2,t2),e2))
        in equal(t,PLUS(t1,t2)); equal(t1',t2'); greatest (t1',t2') end

    | typing (l,g,Loc x) = ...

    | typing (l,g,Ref e) = ...

    | typing (l,g,Deref e) = 
	  ...

    | typing (l,g,Assign(e1,e2)) =
	  ...

    | typing (l,g,Fail) = ...

    | typing (l,g,Try(e1,e2)) =
	 ...

    | typing (l,g,Var(x)) = g(x)

  fun typeOf (e,l) = typing(l,empty,e)
  fun typeOpt (e,l) = SOME (typeOf (e,l)) handle TypeError => NONE

end
