structure Print :> PRINT =
struct

  open MinML UnParse

  (* Expression constructors for type operators. *)

  val up_INT = atom("int")
  val up_BOOL = atom("bool")
  val up_ARROW = infixx(RightAssoc,0,"->")
  val up_TIMES = infixx(RightAssoc,2,"*")
  val up_UNIT = atom("unit")
  val up_PLUS = infixx(RightAssoc,1,"+")
  val up_REF = postfix(3,"ref")
  val up_FAIL = atom("failure")

  (* Convert a type to an unparsing expression. *)

  fun typ INT = up_INT
    | typ BOOL = up_BOOL
    | typ (ARROW(t,t')) = up_ARROW(typ(t),typ(t'))
    | typ (TIMES(t,t')) = up_TIMES(typ(t),typ(t'))
    | typ UNIT = up_UNIT
    | typ (PLUS(t,t')) = up_PLUS(typ(t),typ(t'))
    | typ (REF t) = up_REF(typ(t))
    | typ FAIL = up_FAIL

  fun typToString t = parens(done(typ(t)))

  (* Convert a pat to an unparsing expression. *)

  fun pat WildPat = atom("_")
    | pat (VarPat(s, t)) = atom(s ^ " : " ^ typToString t)
	| pat (PairPat(p1, p2)) = atom("<" ^ patToString p1 ^ "," ^ 
				       patToString p2 ^">")
	| pat UnitPat = atom("<>")

  and patToString p = parens(done(pat(p)))

  (* Expression constructors for primops. *)

  val negate = prefix(3,"~")
  val plus = infixx(LeftAssoc,1,"+")
  val minus = infixx(LeftAssoc,1,"-")
  val times = infixx(LeftAssoc,2,"*")
  val equal = infixx(NonAssoc,0,"=")

  (* Convert an exp to an unparsing expression. *)

  fun exp (Int(i)) = atom(Int.toString(i))

    | exp (Bool(b)) = atom(Bool.toString(b))
    | exp (If(e,e1,e2)) =
          atom("if " ^ expToString e ^ " then " ^ expToString e1 ^
               " else " ^ expToString e2 ^ " fi")

    | exp (Primop(Negate,[e])) = negate(exp(e))
    | exp (Primop(Plus,[e,e'])) = plus(exp(e),exp(e'))
    | exp (Primop(Minus,[e,e'])) = minus(exp(e),exp(e'))
    | exp (Primop(Times,[e,e'])) = times(exp(e),exp(e'))
    | exp (Primop(Equal,[e,e'])) = equal(exp(e),exp(e'))
    | exp (Primop _) = atom("<bad primop>")

    | exp (Fun(f,x,t,t',e)) =
          atom("fun " ^ f ^ " (" ^ x ^ " : " ^ typToString t ^ ") : " ^
               typToString t' ^ " is\n    " ^ expToString e ^ "\nend\n")
    | exp (Apply(e,e')) = adj(exp(e),exp(e'))

    | exp (Var(x)) = atom(x)

    | exp (Pair(e,e')) = atom("<" ^ expToString e ^ "," ^ expToString e' ^ ">")

    | exp (Bind(p,e,e')) = atom("bind " ^ patToString p ^ " to " ^ 
				expToString e ^ " in " ^ expToString e')

    | exp UnitE = atom("<>")
	  
    | exp (Inleft(t,t',e)) = atom("inl (" ^ typToString t ^ "," ^ 
				  typToString t'^") " ^ expToString e) 

    | exp (Inright(t,t',e)) = atom("inr (" ^ typToString t ^ "," ^ 
				   typToString t'^") " ^ expToString e) 

    | exp (Case(e,v1,t1,e1,v2,t2,e2)) =
	  atom("case " ^ expToString e ^ " of " ^
	       "inl" ^ v1 ^":" ^ typToString t1 ^"=>"^ expToString e1 ^ 
		 " | " ^ "inr" ^ v2 ^":" ^ typToString t2 ^"=>" 
		   ^ expToString e2)
    | exp (Ref(e)) = atom("ref " ^ expToString e)
    | exp (Deref(e)) = atom("! " ^ expToString e)
    | exp (Assign(e1,e2)) = atom(expToString e1 ^ " := " ^
				 expToString e2)
    | exp (Loc(l)) = atom("MLOC" ^ l)
    | exp (Fail) = atom("fail")
    | exp (Try(e1,e2)) = atom("try (" ^ expToString e1 ^ ") ow (" ^
	  expToString e2 ^ ")")

  and expToString e = parens(done(exp(e)))

end
