structure UnParse :> UNPARSE =
struct

  datatype Associativity = LeftAssoc | RightAssoc | NonAssoc

  datatype 'a Item =
      Unit of 'a
    | Paren of 'a Seq
  withtype 'a Seq = 'a Item list

  datatype PartType =
      Atm
    | Adj
    | Prefix of int
    | Infix of Associativity * int
    | Postfix of int

  (* A subexpression has a PartType and can prepend itself to an 'a Seq. *)

  type 'a Part = PartType * ('a Seq -> 'a Seq)

  (* one(i) is a function which prepends i to a sequence. *)

  fun one i = (fn s => i::s)

  fun prec (Prefix(p)) = p
    | prec (Infix(_,p)) = p
    | prec (Postfix(p)) = p

  fun assoc (Prefix(_)) = NonAssoc
    | assoc (Infix(a,_)) = a
    | assoc (Postfix(_)) = NonAssoc

  (* right(t,t',f') tests whether an operator of type t can accept
     a subexpression of type t' without parentheses.  If not, it is
     parenthesized. *)

  fun right (t,t',f') =
      let fun accept (_,Atm) = true
            | accept (Adj,_) = false
            | accept (_,Adj) = true
            | accept (_,Prefix _) = true
            | accept (Prefix(p),t) = prec(t) > p
            | accept (Infix(a,p),t) =
                  if prec(t) > p then true
                  else if p > prec(t) then false
                  else a = RightAssoc andalso assoc(t) = RightAssoc

       in if accept(t,t') then f' else one(Paren(f'[])) end

  (* ... on the left ... *)

  fun left (t,t',f') =
      let fun accept (_,Atm) = true
            | accept (_,Adj) = true
            | accept (Adj,_) = false
            | accept (_,Postfix _) = true
            | accept (Postfix(p),t) = prec(t) > p
            | accept (Infix(a,p),t) =
                  if prec(t) > p then true
                  else if p > prec(t) then false
                  else a = LeftAssoc andalso assoc(t) = LeftAssoc

       in if accept(t,t') then f' else one(Paren(f'[])) end

  (* Constructors for various kinds of expressions. *)

  fun atom c = (Atm, one(Unit(c)))

  fun adj ((tl,fl),(tr,fr)) = (Adj, left(Adj,tl,fl) o right(Adj,tr,fr))

  fun prefix (p,c) (tr,fr) =
          (Prefix(p), one(Unit(c)) o right(Prefix(p),tr,fr))

  fun infixx (a,p,c) ((tl,fl),(tr,fr)) =
      let val t = Infix(a,p)
       in (t, left(t,tl,fl) o one(Unit(c)) o right(t,tr,fr)) end

  fun postfix (p,c) (tl,fl) =
          (Postfix(p), left(Postfix(p),tl,fl) o one(Unit(c)))

  (* Convert a subexpression to a Seq. *)

  fun done (_,f) = f []

  fun compose cseq cparen s =
      let fun citem (Unit a) = a
            | citem (Paren s) = cparen(cseq(map citem s))
       in cseq(map citem s) end

  (* Concatenate a list of strings with spaces in between. *)

  fun spaces [] = ""
    | spaces (h::t) = h ^ foldr op^ "" (map (fn s => " " ^ s) t)

  val parens = compose spaces (fn s => "(" ^ s ^ ")")

end