structure CMachine :> CMACHINE =
struct

  val Debug = ref false

  infix 1 ++
  infixr |>
  open MinML

  datatype frame =
      PrimopFrame of primop * exp list * int
    | IfFrame of exp * exp
    | Apply1Frame of exp
    | Apply2Frame of exp
    | Pair1Frame of exp
    | Pair2Frame of exp
    | BindFrame of pat * exp
    | InleftFrame of typ * typ
    | InrightFrame of typ * typ
    | CaseFrame of string * typ * exp * string * typ * exp
    | RefFrame
    | DerefFrame
    | Assign1Frame of exp
    | Assign2Frame of exp
    | TryFrame

  datatype 'a stack =
      EmptyStack
    | |> of 'a * 'a stack

  type memory = subst * (string -> MinML.typ)

  type handlers = (frame stack * exp) stack

  type state = memory * handlers * (frame stack) * exp

  fun frameToString (PrimopFrame (po, el, n)) = "PrimopFrame<" ^
    Print.expToString (MinML.Primop (po, Var "???" :: el)) ^ ">"
    | frameToString (IfFrame (e1, e2)) = "IfFrame<"^
    Print.expToString (e1) ^ ", " ^ Print.expToString (e2) ^ ">"
    | frameToString (Apply1Frame (e)) = "Apply1Frame<"^
    Print.expToString (e) ^ ">"
    | frameToString (Apply2Frame (e)) = "Apply2Frame<"^
    Print.expToString (e) ^ ">"
    | frameToString (Pair1Frame (e)) = "Pair1Frame<"^
    Print.expToString (e) ^ ">"
    | frameToString (Pair2Frame (e)) = "Pair2Frame<"^
    Print.expToString (e) ^ ">"
    | frameToString (BindFrame (p, e)) = "BindFrame<"^
    Print.patToString (p) ^ ", " ^ Print.expToString (e) ^ ">"
    | frameToString (InleftFrame (t1, t2)) = "InleftFrame<"^
    Print.typToString (t1) ^ ", " ^ Print.typToString (t2) ^ ">"
    | frameToString (InrightFrame (t1, t2)) = "InrightFrame<"^
    Print.typToString (t1) ^ ", " ^ Print.typToString (t2) ^ ">"
    | frameToString (CaseFrame (s1, t1, e1, s2, t2, e2)) = "CaseFrame<" ^
    s1 ^ ", " ^ Print.typToString (t1) ^ ", " ^ Print.expToString (e1) ^
    s2 ^ ", " ^ Print.typToString (t2) ^ ", " ^ Print.expToString (e2) ^ ">"
    | frameToString (RefFrame) = "RefFrame"
    | frameToString (DerefFrame) = "DerefFrame"
    | frameToString (Assign1Frame (e)) = "Assign1Frame<"^
    Print.expToString (e) ^ ">"
    | frameToString (Assign2Frame (e)) = "Assign2Frame<"^
    Print.expToString (e) ^ ">"
    | frameToString (TryFrame) = "TryFrame"

  fun handlerToString (EmptyStack) = "()"
    | handlerToString ((EmptyStack, e)|>EmptyStack) = "(<(), " ^
    Print.expToString e ^ ">)"
    | handlerToString ((EmptyStack, e)|>_) = "(<(), " ^
    Print.expToString e ^ ">, ...)"
    | handlerToString ((k|>EmptyStack, e)|>EmptyStack) = "(<(" ^
    frameToString (k) ^ "), " ^
    Print.expToString e ^ ">)"
    | handlerToString ((k|>EmptyStack, e)|>_) = "(<(" ^
    frameToString (k) ^ "), " ^
    Print.expToString e ^ ">, ...)"
    | handlerToString ((k|>_, e)|>EmptyStack) = "(<(" ^
    frameToString (k) ^ ", ...), " ^
    Print.expToString e ^ ">)"
    | handlerToString ((k|>_, e)|>_) = "(<(" ^
    frameToString (k) ^ ", ...), " ^
    Print.expToString e ^ ">, ...)"

  fun stateToString ((m, h, k, e) : state) = 
    case k of 
      EmptyStack => "<<" ^ handlerToString h ^ "(),\n  " ^ 
	Print.expToString e ^ ">>"
    | (hk|>EmptyStack) => "<<" ^ handlerToString h ^ "(" ^ 
	frameToString hk ^ "),\n  " ^ Print.expToString e ^ ">>"
    | (hk|>_) => "<<" ^ handlerToString h ^ "(" ^ 
	frameToString hk ^ ", ...),\n  " ^ Print.expToString e ^ ">>"

  fun value (Int _ | Bool _ | Fun _ | UnitE | Loc _ | Fail) = true
    | value (Pair(v1,v2)) = value v1 andalso value v2
    | value (Inleft(t1,t2,v)) = value v
    | value (Inright(t1,t2,v)) = value v
    | value _ = false

  exception Stuck
  
  local
    val counter = ref 0
  in
    fun newloc () = (counter := (!counter) + 1;
		       Int.toString (!counter))
  end

  val empty = fn _ => raise Stuck

  infix 1 ++
  fun (g ++ (x,e)) = fn x' => if x = x' then e else g(x')

  fun nextinlist ([],v,0) = ([v],NONE)
    | nextinlist (e::es,v,0) = (v::es, SOME e)
    | nextinlist (e::es,v,i) =
      let val (es',enext) = nextinlist(es,v,i-1)
      in (e::es',enext)
      end
    | nextinlist _ = raise Stuck

  fun Cstep(m,h,k,e) = if value e then
    (case (m,h,k,e) of
      (m, h, k, Fail) => 
	...
    | (m, h, PrimopFrame(p,es,i) |> k, v) =>
	  let val (es',enext) = nextinlist(es,v,i)
	  in case enext of
	       SOME enext => (m, h, PrimopFrame(p,es',i+1) |> k, enext)
	     | NONE => (m, h, k, valOf(evalPrimop(p,es'))
			handle Option => raise Stuck)
	  end
    | (m, h, IfFrame(e1,e2) |> k, Bool true) =>
	  ...
    | (m, h, IfFrame(e1,e2) |> k, Bool false) =>
	  ...
    | (m, h, Apply1Frame(e2) |> k, v1) => 
	  ...
    | (m, h, Apply2Frame(v1 as Fun(f,x,t1,t2,e)) |> k, v2) => 
	  ...
    | (m, h, Pair1Frame(e2) |> k, v1) =>
	  ...
    | (m, h, Pair2Frame(v1) |> k, v2) => 
	  ...
    | (m, h, BindFrame(WildPat,e) |> k, v) => 
	  ...
    | (m, h, BindFrame(VarPat(x,t),e) |> k, v) =>
	  ...
    | (m, h, BindFrame(PairPat(p1,p2),e) |> k, Pair(v1,v2)) => 
	  ...
    | (m, h, InleftFrame(t1,t2) |> k, v) =>
	  ...
    | (m, h, InrightFrame(t1,t2) |> k, v) => 
	  ...
    | (m, h, CaseFrame(x1,t1,e1,x2,t2,e2) |> k, Inleft(_,_,v)) => 
	  ...
    | (m, h, CaseFrame(x1,t1,e1,x2,t2,e2) |> k, Inright(_,_,v)) => 
	  ...
    | (m, h, RefFrame |> k, v) =>
	  ...
    | (m, h, DerefFrame |> k, v) =>
	  ...
    | (m, h, Assign1Frame(e2) |> k, v1) =>
	  ...
    | (m, h, Assign2Frame(v1) |> k, v2) =>
	  ...
    | (m, _ |> h, TryFrame |> k, v) =>
	  ...
    | _ => raise Stuck)
  else
    (case e of
      (Primop(p,e::es)) =>
	  (m, h, PrimopFrame(p,es,0) |> k, e)
    | (If(e,e1,e2)) =>
	  ...
    | (Apply(e1,e2)) => 
	  ...
    | (Pair(e1,e2)) => 
	  ...
    | (Bind(p,e1,e2)) => 
	  ...
    | (Inleft(t1,t2,e)) => 
	  ...
    | (Inright(t1,t2,e)) => 
	  ...
    | (Case(e,x1,t1,e1,x2,t2,e2)) =>
	  ...
    | (Ref (e)) =>
	  ...
    | (Deref (e)) =>
	  ...
    | (Assign (e1, e2)) => 
	  ...
    | (Try (e1, e2)) =>
	  ...
    | _ => raise Stuck)
	 

  fun CstepOpt state = SOME (Cstep state) handle Stuck => NONE

  fun CmultiStep (state : state) = (if (!Debug) then
			    print (stateToString state ^ "\n")
			  else ();
			    CmultiStep (Cstep state) handle Stuck => state)

  fun CstepStream e =
      let fun steps (m,h,k,e) =
              (case CstepOpt (m,h,k,e)
                 of NONE => Stream.empty
                  | SOME (m',h',k',e') => Stream.lcons((m',h',k',e'), fn () => 
						 steps (m',h',k',e')))
       in Stream.lcons(((empty,empty),EmptyStack,EmptyStack,e), fn () => 
		       steps ((empty,empty),EmptyStack,EmptyStack,e)) end

  fun eval e =
    let val (m,h,k,e') = CmultiStep((empty,empty),EmptyStack,EmptyStack,e)
    in if k = EmptyStack then (#2 m, e') else raise Stuck
    end

end
