(* Chris Okasaki
   School of Computer Science
   Carnegie Mellon University
   Pittsburgh, PA 15213
   cokasaki@cs.cmu.edu *)

functor Error (val defaultMaxErrors : int
               val defaultWarningThreshold : int) :> ERROR =
  (* error/warning messages suitable for a compiler *)
struct

  open TextIO

  exception TooManyErrors
  exception FatalError

(***** INTERNAL STATE *****)

  val errcount = ref 0
  val warncount = ref 0  (* reported warnings *)
  val warncount' = ref 0 (* suppressed warnings *)

  fun numErrors () = !errcount
  fun numWarnings () = (!warncount,!warncount')

  fun reset () = (errcount := 0;
		  warncount := 0;
		  warncount' := 0)

  fun inc r = (r := !r + 1)

(***** USER SETTINGS *****)

  val maxErrors = ref defaultMaxErrors
  val warningThreshold = ref defaultWarningThreshold

(***** REPORT AN ERROR/WARNING *****)
  fun display errtype unitname msg pos =
        (output (stdErr,"[" ^ Pos.toString pos ^ "] "
                             ^ unitname ^ " "
                             ^ errtype ^ ": "
                             ^ msg ^ "\n");
	 flushOut stdErr)

  fun warning severity unitname msg pos =
        if severity >= !warningThreshold
	  then (inc warncount;
		display ("Warning(" ^ Int.toString severity ^")")
                        unitname msg pos)
	  else inc warncount'

  fun error unitname msg pos =
       (display "Error" unitname msg pos;
        inc errcount;
        if !errcount >= !maxErrors then raise TooManyErrors else ())

  fun fatalerror unitname msg pos =
       (display "Fatal Error" unitname msg pos;
        raise FatalError)

  fun okay unitname msg =
      if !errcount > 0
      then (output (stdErr,"Errors found in "
                            ^ unitname ^ ": "
		            ^ msg ^ "\n");
	    flushOut stdErr;
	    raise TooManyErrors)
      else ()
end
