structure Parse :> PARSE =
struct

  open Parsing MinML Tokens

  infixr 4 << >>
  infixr 3 &&
  infix  2 -- ##
  infixr 2 wth suchthat return guard
  infixr 1 ||



  (**** Types ****)

  val intT = litWord "int" return INT
  val boolT = litWord "bool" return BOOL
  val arrowT = litWord "->" return Infix(RightAssoc,0,ARROW)
  val timesT = litWord "*" return Infix(RightAssoc,2,TIMES)
  val unitT = litWord "unit" return UNIT
  val plusT = litWord "+" return Infix(RightAssoc,1,PLUS)
  val refT = litWord "ref" return Postfix(3,REF)

  fun parentheticalT () = litWord "(" >> $typ << litWord ")"

  and typItem () = alt [intT, boolT, unitT, $parentheticalT] wth Atm
                || alt [arrowT, timesT, plusT, refT] wth Opr

  and typ () = parsefixity ($typItem)

  val typ = $typ

  (**** Identifier ****)

  fun reserved s = List.exists (fn x => x = s)
         ["int","bool","->","true","false","=","+","-","*","~","if",
          "then","else","fi","fun","end","(",")",":",";",",","<",">", "=>",
	  "unit", "_", "inl", "inr", "case", "of", "=>", "|", "bind","to",
	  "in" , "ref", "!", ":=", "try", "ow", "is", "fail", "<>"]
  val identifier = anyWord suchthat (not o reserved)

  (**** Patterns ****)

  val wildPatP = litWord "_" return WildPat
  val varPatP = identifier << litWord ":" && typ 
                wth VarPat
  val unitPatP = litWord "<>" return UnitPat
  fun pairPatP () = litWord "<" >> $pat << litWord "," && $pat << litWord ">"
                wth PairPat
  and pat () = alt [wildPatP, varPatP, unitPatP, $pairPatP] 

  val pat = $pat

  (**** Expressions ****)

  val variable = identifier wth Var

  val int_const = anyNumber wth Int

  val tru = litWord "true" return Bool true
  val fals = litWord "false" return Bool false
  val fail = litWord "fail" return Fail

  val unitE = litWord "<>" return UnitE

  (* prefix/infix_primop construct operators for the fixity parser
     which apply Primops *)

  fun prefix_primop (prec,primop) =
          Prefix(prec, fn a => Primop(primop, [a]))

  fun infix_primop (assoc,prec,primop) =
          Infix(assoc,prec, fn (a,b) => Primop(primop, [a,b]))

  fun flat7 (a,(b,(c,(d,(e,(f, g)))))) = (a,b,c,d,e,f,g)

  val equal = litWord "=" return infix_primop(NonAssoc,0,Equal)
  val plus = litWord "+" return infix_primop(LeftAssoc,1,Plus)
  val minus = litWord "-" return infix_primop(LeftAssoc,1,Minus)
  val times = litWord "*" return infix_primop(LeftAssoc,2,Times)
  val negate = litWord "~" return prefix_primop(3,Negate)
  val assign = litWord ":=" return Infix(NonAssoc,0,Assign)

  fun conditional () = litWord "if"   >> $exp
                    && litWord "then" >> $exp
                    && litWord "else" >> $exp << litWord "fi"
                   wth (If o flat3)

  and function () = litWord "fun" >> identifier
                 && litWord "("   >> identifier
                 && litWord ":"   >> typ << litWord ")"
                 && litWord ":"   >> typ
                 && litWord "is"  >> $exp << litWord "end"
                wth (Fun o flat5)

  and pair () = litWord "<" >> $exp
             && litWord "," >> $exp << litWord ">"
			 wth Pair

  and binding () = litWord "bind" >> pat
             && litWord "to" >> $exp
			 && litWord "in" >> $exp
			 wth (Bind o flat3)

  and inleft () = litWord "inl" >> litWord "(" >> typ 
             && litWord "," >> typ << litWord ")" && $exp 
			 wth (Inleft o flat3)

  and inright () = litWord "inr" >> litWord "(" >> typ 
             && litWord "," >> typ << litWord ")" && $exp 
			 wth (Inright o flat3)

  and caseE () = litWord "case" >> $exp << litWord "of"
             && litWord "inl" >> identifier << litWord ":" 
			 && typ << litWord "=>"
			 && $exp << litWord "|"
			 && litWord "inr" >> identifier << litWord ":"
			 && typ << litWord "=>"
			 && $exp 
			 wth (Case o flat7)

  and refE () = litWord "ref" >> $exp
                wth Ref

  and derefE () = litWord "!" >> $exp
                wth Deref
		
  and tryE () = litWord "try" >> $exp
             && litWord "ow" >> $exp
                        wth Try

  and parenthetical () = litWord "(" >> $exp << litWord ")"

  (* NOTE: There is no concrete syntax for "apply".  It is written
     by just writing the function and its argument next to each
     other, as in SML.  The fixity parser takes care of this. *)

  and expItem () = alt [variable, int_const, tru, fals, fail, unitE,
                        $conditional, $function, $parenthetical,
			$pair, $binding, $inleft, $inright, $caseE,
			$refE, $derefE, $tryE] wth Atm
                || alt [equal, plus, minus, times, negate, assign] wth Opr

  and exp () = parsefixityadj ($expItem) Apply

  val exp = $exp

end
