structure MinML :> MINML =
struct

  datatype typ =
      INT
    | BOOL
    | ARROW of typ * typ
    | TIMES of typ * typ
    | UNIT
    | PLUS of typ * typ
    | REF of typ
    | FAIL

  datatype pat =
      WildPat
    | VarPat of string * typ
    | PairPat of pat * pat
    | UnitPat

  datatype primop = Equal | Plus | Minus | Times | Negate

  datatype exp =
      Int of int

    | Bool of bool
    | If of exp * exp * exp

    | Primop of primop * exp list

    | Fun of string * string * typ * typ * exp
    | Apply of exp * exp

    | Var of string

    | Pair of exp * exp
    | Bind of pat * exp * exp
    | UnitE 

    | Inleft of typ * typ * exp
    | Inright of typ * typ * exp
    | Case of exp * string * typ * exp * string * typ * exp

    | Loc of string
    | Ref of exp
    | Deref of exp
    | Assign of exp * exp
      
    | Fail
    | Try of exp * exp

  (* Argument and result types of all the primops. *)

  fun typeOfPrimop Equal  = ([INT, INT], BOOL)
    | typeOfPrimop Plus   = ([INT, INT], INT)
    | typeOfPrimop Minus  = ([INT, INT], INT)
    | typeOfPrimop Times  = ([INT, INT], INT)
    | typeOfPrimop Negate = ([INT], INT)

  (* Evaluation for all the primops. *)

  fun evalPrimop (Equal, [Int i, Int i']) = SOME (Bool (i = i'))
    | evalPrimop (Plus, [Int i, Int i']) = SOME (Int (i + i'))
    | evalPrimop (Minus, [Int i, Int i']) = SOME (Int (i - i'))
    | evalPrimop (Times, [Int i, Int i']) = SOME (Int (i * i'))
    | evalPrimop (Negate, [Int i]) = SOME (Int (~i))
    | evalPrimop _ = NONE

  (**** Substitution ****)

  type subst = string -> exp

  (* id is the identity substitution (does nothing). *)

  val id = fn x => Var x

  (* r++(x,e) is the substitution taking x to e and applying r
       to all other variables. *)

  infix 1 ++
  fun (r ++ (x,e)) = fn y => if x=y then e else r(y)

  (* csubst(r,e) applies the CLOSED substitution r to the
       expression e. *)

  fun pat2subst (r, WildPat) = r
    | pat2subst (r, VarPat(x,t)) = r ++ (x, Var x)
    | pat2subst (r, PairPat(p1,p2)) =
      pat2subst(pat2subst(r,p1),p2)

  fun csubst (r,e) =
    let fun rsub e = csubst(r,e)
    in
      case e of
        Var x => r x
      | Primop(p,es) => Primop(p, map rsub es)
      | If(e,e1,e2) => If(rsub e, rsub e1, rsub e2)
      | Fun(f,x,t1,t2,e) =>
          Fun(f,x,t1,t2,csubst(r ++ (f, Var f) ++ (x, Var x), e))
      | Apply(e1,e2) => Apply(rsub e1, rsub e2)
      | Pair(e1,e2) => Pair(rsub e1, rsub e2)
      | Bind(p,e1,e2) =>
          Bind(p, rsub e1, csubst(pat2subst(r,p),e2))
      | Inleft(t1,t2,e) => Inleft(t1, t2, rsub e)
      | Inright(t1,t2,e) => Inright(t1, t2, rsub e)
      | Case(e,x1,t1,e1,x2,t2,e2) =>
          Case(rsub e, x1, t1, csubst(r ++ (x1, Var x1), e1),
               x2, t2, csubst(r ++ (x2, Var x2), e2))
      | Ref(e) => Ref(rsub e)
      | Deref(e) => Deref(rsub e)
      | Assign(e1,e2) => Assign(rsub e1, rsub e2)
      | Try (e1, e2) => Try(rsub e1, rsub e2)
      | _ => e
    end

end
